<?php
/**
 * Multi-Tenant Isolation Test Suite
 * Tests that schools cannot access each other's data
 */

require_once __DIR__ . '/../config.php';

echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Multi-Tenant Isolation Test Suite                        ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n\n";

$pdo = getDB();
$passed = 0;
$failed = 0;

// Test 1: Check if academy_reference column exists
echo "Test 1: Checking database structure...\n";
try {
    $result = $pdo->query("SHOW COLUMNS FROM students LIKE 'academy_reference'");
    if ($result->fetch()) {
        echo "  ✓ students.academy_reference exists\n";
        $passed++;
    } else {
        echo "  ✗ students.academy_reference MISSING\n";
        $failed++;
    }
    
    $result = $pdo->query("SHOW COLUMNS FROM payments LIKE 'academy_reference'");
    if ($result->fetch()) {
        echo "  ✓ payments.academy_reference exists\n";
        $passed++;
    } else {
        echo "  ✗ payments.academy_reference MISSING\n";
        $failed++;
    }
    
    $result = $pdo->query("SHOW TABLES LIKE 'academy_references'");
    if ($result->fetch()) {
        echo "  ✓ academy_references table exists\n";
        $passed++;
    } else {
        echo "  ✗ academy_references table MISSING\n";
        $failed++;
    }
    
    $result = $pdo->query("SHOW TABLES LIKE 'tenant_settings'");
    if ($result->fetch()) {
        echo "  ✓ tenant_settings table exists\n";
        $passed++;
    } else {
        echo "  ✗ tenant_settings table MISSING\n";
        $failed++;
    }
    
} catch (PDOException $e) {
    echo "  ✗ Database structure test failed: " . $e->getMessage() . "\n";
    $failed++;
}

echo "\n";

// Test 2: Check all students have academy_reference
echo "Test 2: Checking data integrity...\n";
try {
    $stmt = $pdo->query("SELECT COUNT(*) FROM students WHERE academy_reference IS NULL");
    $nullCount = $stmt->fetchColumn();
    
    if ($nullCount == 0) {
        echo "  ✓ All students have academy_reference\n";
        $passed++;
    } else {
        echo "  ✗ {$nullCount} students missing academy_reference\n";
        $failed++;
    }
    
    // Check payments
    $stmt = $pdo->query("SELECT COUNT(*) FROM payments WHERE academy_reference IS NULL AND status = 'SUCCESSFUL'");
    $nullPayments = $stmt->fetchColumn();
    
    if ($nullPayments == 0) {
        echo "  ✓ All successful payments have academy_reference\n";
        $passed++;
    } else {
        echo "  ⚠ {$nullPayments} payments missing academy_reference\n";
    }
    
} catch (PDOException $e) {
    echo "  ✗ Data integrity test failed: " . $e->getMessage() . "\n";
    $failed++;
}

echo "\n";

// Test 3: Test cross-school data isolation
echo "Test 3: Testing cross-school isolation...\n";
try {
    // Get all schools
    $schools = $pdo->query("SELECT reference_code FROM academy_references WHERE is_active = 1")->fetchAll();
    
    if (count($schools) < 2) {
        echo "  ⚠ Only 1 school found - creating test school for isolation testing...\n";
        
        // Create a test school
        $pdo->exec("
            INSERT IGNORE INTO academy_references (admin_id, academy_name, reference_code, description, is_active)
            VALUES (1, 'Test School', 'TEST', 'For isolation testing', 1)
        ");
        
        $schools = $pdo->query("SELECT reference_code FROM academy_references WHERE is_active = 1")->fetchAll();
    }
    
    if (count($schools) >= 2) {
        $school1 = $schools[0]['reference_code'];
        $school2 = $schools[1]['reference_code'];
        
        // Get student counts per school
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM students WHERE academy_reference = ?");
        $stmt->execute([$school1]);
        $count1 = $stmt->fetchColumn();
        
        $stmt->execute([$school2]);
        $count2 = $stmt->fetchColumn();
        
        echo "  ✓ School '{$school1}' has {$count1} students\n";
        echo "  ✓ School '{$school2}' has {$count2} students\n";
        
        // Verify query filtering works
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM students 
            WHERE academy_reference = ? AND deleted_at IS NULL
        ");
        $stmt->execute([$school1]);
        $filtered = $stmt->fetchColumn();
        
        if ($filtered <= $count1) {
            echo "  ✓ Query filtering by academy_reference works\n";
            $passed++;
        } else {
            echo "  ✗ Query filtering returned incorrect results\n";
            $failed++;
        }
        
    } else {
        echo "  ⚠ Cannot test cross-school isolation with only 1 school\n";
    }
    
} catch (PDOException $e) {
    echo "  ✗ Isolation test failed: " . $e->getMessage() . "\n";
    $failed++;
}

echo "\n";

// Test 4: Check indexes exist
echo "Test 4: Checking performance indexes...\n";
try {
    $indexes = [
        'students' => 'idx_students_academy_ref',
        'payments' => 'idx_payments_academy_ref'
    ];
    
    foreach ($indexes as $table => $index) {
        $stmt = $pdo->query("SHOW INDEX FROM $table WHERE Key_name = '$index'");
        if ($stmt->fetch()) {
            echo "  ✓ Index {$table}.{$index} exists\n";
            $passed++;
        } else {
            echo "  ⚠ Index {$table}.{$index} missing (performance warning)\n";
        }
    }
    
} catch (PDOException $e) {
    echo "  ✗ Index check failed: " . $e->getMessage() . "\n";
}

echo "\n";

// Summary
echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Test Summary                                              ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n";
echo "  ✓ Passed: $passed\n";
echo "  ✗ Failed: $failed\n";
echo "\n";

if ($failed === 0) {
    echo "✅ All tests passed! Multi-tenant isolation is working correctly.\n\n";
    exit(0);
} else {
    echo "⚠️  Some tests failed. Review the output above.\n\n";
    exit(1);
}

