<?php
/**
 * Comprehensive Clean URLs Testing Suite
 * 
 * This script tests all URL rewriting rules and security configurations.
 * Run this from command line: php tools/test_clean_urls_comprehensive.php
 * 
 * Or access via browser: https://your-domain.com/Multi-Tanent/tools/test_clean_urls_comprehensive.php
 */

// Prevent direct web access in production
if (php_sapi_name() !== 'cli' && $_SERVER['HTTP_HOST'] !== 'localhost' && strpos($_SERVER['HTTP_HOST'], '127.0.0.1') === false) {
    // Uncomment in production:
    // die('Access denied. Run from command line only.');
}

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Clean URLs Testing Suite</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            padding: 30px;
        }
        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 2em;
        }
        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 1.1em;
        }
        .test-section {
            margin-bottom: 30px;
            padding: 20px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            background: #f9f9f9;
        }
        .test-section h2 {
            color: #667eea;
            margin-bottom: 15px;
            font-size: 1.5em;
            border-bottom: 2px solid #667eea;
            padding-bottom: 10px;
        }
        .test-item {
            margin: 10px 0;
            padding: 15px;
            background: white;
            border-radius: 5px;
            border-left: 4px solid #667eea;
        }
        .test-item.pass {
            border-left-color: #28a745;
            background: #d4edda;
        }
        .test-item.fail {
            border-left-color: #dc3545;
            background: #f8d7da;
        }
        .test-item.warning {
            border-left-color: #ffc107;
            background: #fff3cd;
        }
        .test-label {
            font-weight: bold;
            color: #333;
            margin-bottom: 5px;
        }
        .test-url {
            color: #667eea;
            font-family: monospace;
            background: #f0f0f0;
            padding: 5px 10px;
            border-radius: 3px;
            display: inline-block;
            margin: 5px 0;
            word-break: break-all;
        }
        .test-result {
            margin-top: 10px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 3px;
            font-family: monospace;
            font-size: 0.9em;
        }
        .summary {
            margin-top: 30px;
            padding: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 8px;
            text-align: center;
        }
        .summary h2 {
            margin-bottom: 15px;
            font-size: 2em;
        }
        .summary-stats {
            display: flex;
            justify-content: space-around;
            margin-top: 20px;
        }
        .stat {
            text-align: center;
        }
        .stat-value {
            font-size: 2.5em;
            font-weight: bold;
            margin-bottom: 5px;
        }
        .stat-label {
            font-size: 1.1em;
            opacity: 0.9;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin: 5px;
            transition: background 0.3s;
        }
        .btn:hover {
            background: #5568d3;
        }
        .info-box {
            background: #d1ecf1;
            border-left: 4px solid #0c5460;
            padding: 15px;
            margin: 15px 0;
            border-radius: 5px;
        }
        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🧪 Clean URLs Testing Suite</h1>
        <p class="subtitle">Comprehensive test for URL rewriting, security, and tenant isolation</p>
        
        <?php
        $tests = [];
        $baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . 
                   '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost') . 
                   '/Multi-Tanent';
        
        // Test 1: Check if mod_rewrite is enabled
        $modRewriteEnabled = function_exists('apache_get_modules') ? in_array('mod_rewrite', apache_get_modules()) : null;
        $tests[] = [
            'section' => 'Server Configuration',
            'label' => 'mod_rewrite Module',
            'status' => $modRewriteEnabled === true ? 'pass' : ($modRewriteEnabled === false ? 'fail' : 'warning'),
            'message' => $modRewriteEnabled === true ? 'mod_rewrite is enabled' : 
                        ($modRewriteEnabled === false ? 'mod_rewrite is NOT enabled - URL rewriting will not work!' : 
                        'Unable to detect mod_rewrite status (may work if running via Apache)')
        ];
        
        // Test 2: Check .htaccess file exists
        $htaccessExists = file_exists(__DIR__ . '/../.htaccess');
        $tests[] = [
            'section' => 'Server Configuration',
            'label' => '.htaccess File',
            'status' => $htaccessExists ? 'pass' : 'fail',
            'message' => $htaccessExists ? '.htaccess file found' : '.htaccess file NOT found in root directory!'
        ];
        
        // Test 3: Test school-specific clean URLs
        $schoolTests = [
            [
                'label' => 'Student Registration',
                'clean_url' => '/kine/register',
                'expected' => 'register_student.php?school=kine'
            ],
            [
                'label' => 'Parent Registration',
                'clean_url' => '/kine/parent/register',
                'expected' => 'parent_register.php?school=kine'
            ],
            [
                'label' => 'Parent Login',
                'clean_url' => '/kine/parent/login',
                'expected' => 'parent_login.php?school=kine'
            ],
            [
                'label' => 'Student Login',
                'clean_url' => '/kine/login',
                'expected' => 'student_login.php?school=kine'
            ]
        ];
        
        foreach ($schoolTests as $test) {
            $tests[] = [
                'section' => 'Tenant Isolation URLs',
                'label' => $test['label'],
                'status' => 'warning',
                'url' => $baseUrl . $test['clean_url'],
                'expected' => $test['expected'],
                'message' => 'Click URL to test manually. Should rewrite to: ' . $test['expected']
            ];
        }
        
        // Test 4: Test admin clean URLs
        $adminTests = [
            [
                'label' => 'Admin Dashboard',
                'clean_url' => '/admin/dashboard',
                'expected' => 'admin/dashboard.php'
            ],
            [
                'label' => 'Admin Students List',
                'clean_url' => '/admin/students/list',
                'expected' => 'admin/students/list.php'
            ]
        ];
        
        foreach ($adminTests as $test) {
            $tests[] = [
                'section' => 'Clean URLs (.php removal)',
                'label' => $test['label'],
                'status' => 'warning',
                'url' => $baseUrl . $test['clean_url'],
                'expected' => $test['expected'],
                'message' => 'Click URL to test manually. Should rewrite to: ' . $test['expected']
            ];
        }
        
        // Test 5: Test super admin clean URLs
        $superAdminTests = [
            [
                'label' => 'Super Admin Dashboard',
                'clean_url' => '/super_admin/dashboard',
                'expected' => 'super_admin/dashboard.php'
            ],
            [
                'label' => 'Super Admin Schools List',
                'clean_url' => '/super_admin/schools/list',
                'expected' => 'super_admin/schools/list.php'
            ]
        ];
        
        foreach ($superAdminTests as $test) {
            $tests[] = [
                'section' => 'Clean URLs (.php removal)',
                'label' => $test['label'],
                'status' => 'warning',
                'url' => $baseUrl . $test['clean_url'],
                'expected' => $test['expected'],
                'message' => 'Click URL to test manually. Should rewrite to: ' . $test['expected']
            ];
        }
        
        // Test 6: Test protected directories
        $protectedDirs = [
            'includes' => 'includes/.htaccess',
            'vendor' => 'vendor/.htaccess',
            'uploads' => 'uploads/.htaccess'
        ];
        
        foreach ($protectedDirs as $dir => $htaccess) {
            $exists = file_exists(__DIR__ . '/../' . $htaccess);
            $tests[] = [
                'section' => 'Directory Protection',
                'label' => ucfirst($dir) . ' Directory',
                'status' => $exists ? 'pass' : 'fail',
                'message' => $exists ? "{$htaccess} found - directory is protected" : 
                            "{$htaccess} NOT found - directory may be accessible!"
            ];
        }
        
        // Test 7: Check if sensitive files are accessible (should be blocked)
        $sensitiveFiles = [
            'config.php',
            'includes/functions.php',
            'vendor/autoload.php'
        ];
        
        foreach ($sensitiveFiles as $file) {
            $exists = file_exists(__DIR__ . '/../' . $file);
            $tests[] = [
                'section' => 'File Protection',
                'label' => basename($file),
                'status' => $exists ? 'warning' : 'warning',
                'url' => $baseUrl . '/' . $file,
                'message' => $exists ? "File exists. Click to test if access is blocked (should return 403)" : 
                            "File not found - may be expected"
            ];
        }
        
        // Group tests by section
        $sections = [];
        foreach ($tests as $test) {
            $section = $test['section'];
            if (!isset($sections[$section])) {
                $sections[$section] = [];
            }
            $sections[$section][] = $test;
        }
        
        // Display tests by section
        foreach ($sections as $sectionName => $sectionTests) {
            echo "<div class='test-section'>";
            echo "<h2>{$sectionName}</h2>";
            
            foreach ($sectionTests as $test) {
                echo "<div class='test-item {$test['status']}'>";
                echo "<div class='test-label'>{$test['label']}</div>";
                
                if (isset($test['url'])) {
                    echo "<div><a href='{$test['url']}' target='_blank' class='test-url'>{$test['url']}</a></div>";
                }
                
                echo "<div class='test-result'><strong>Status:</strong> {$test['message']}</div>";
                
                if (isset($test['expected'])) {
                    echo "<div class='test-result'><strong>Expected:</strong> <code>{$test['expected']}</code></div>";
                }
                
                echo "</div>";
            }
            
            echo "</div>";
        }
        
        // Calculate summary
        $passed = count(array_filter($tests, fn($t) => $t['status'] === 'pass'));
        $failed = count(array_filter($tests, fn($t) => $t['status'] === 'fail'));
        $warnings = count(array_filter($tests, fn($t) => $t['status'] === 'warning'));
        $total = count($tests);
        
        echo "<div class='summary'>";
        echo "<h2>Test Summary</h2>";
        echo "<div class='summary-stats'>";
        echo "<div class='stat'><div class='stat-value'>{$total}</div><div class='stat-label'>Total Tests</div></div>";
        echo "<div class='stat'><div class='stat-value' style='color:#28a745'>{$passed}</div><div class='stat-label'>Passed</div></div>";
        echo "<div class='stat'><div class='stat-value' style='color:#dc3545'>{$failed}</div><div class='stat-label'>Failed</div></div>";
        echo "<div class='stat'><div class='stat-value' style='color:#ffc107'>{$warnings}</div><div class='stat-label'>Warnings</div></div>";
        echo "</div>";
        echo "</div>";
        
        echo "<div class='info-box'>";
        echo "<h3>📋 Manual Testing Instructions</h3>";
        echo "<ol>";
        echo "<li>Click on each URL above to verify it works correctly</li>";
        echo "<li>Check browser developer tools (Network tab) to see actual requests</li>";
        echo "<li>Verify that protected files return 403 Forbidden</li>";
        echo "<li>Test clean URLs work without .php extension</li>";
        echo "<li>Verify tenant isolation URLs work correctly</li>";
        echo "</ol>";
        echo "</div>";
        
        echo "<div class='info-box'>";
        echo "<h3>🔧 Debugging Tips</h3>";
        echo "<ul>";
        echo "<li>Check Apache error logs: <code>/var/log/apache2/error.log</code> or <code>C:\\xampp\\apache\\logs\\error.log</code></li>";
        echo "<li>Enable mod_rewrite logging in Apache config if issues persist</li>";
        echo "<li>Verify <code>AllowOverride All</code> is set in Apache virtual host config</li>";
        echo "<li>Test with <code>curl -I</code> to see HTTP headers</li>";
        echo "<li>Use browser developer tools to check redirects</li>";
        echo "</ul>";
        echo "</div>";
        ?>
        
        <div style="text-align: center; margin-top: 30px;">
            <a href="<?= $baseUrl ?>" class="btn">← Back to Home</a>
            <a href="javascript:location.reload()" class="btn">🔄 Refresh Tests</a>
        </div>
    </div>
</body>
</html>

