<?php
/**
 * Migrate Existing Files to Tenant Folder Structure
 * 
 * Migrates files from old structure to new tenant-based structure
 * 
 * USAGE:
 *   php tools/migrate_to_tenant_folders.php --dry-run
 *   php tools/migrate_to_tenant_folders.php --execute --tenant-id=default
 * 
 * OPTIONS:
 *   --dry-run        Show what would be migrated without making changes
 *   --execute        Actually perform the migration
 *   --tenant-id      Target tenant ID (default: 'default')
 *   --backup         Create backup before migration (default: true)
 *   --no-backup      Skip backup creation
 *   --verbose        Verbose output
 * 
 * EXAMPLE:
 *   php tools/migrate_to_tenant_folders.php --dry-run --verbose
 *   php tools/migrate_to_tenant_folders.php --execute --tenant-id=legacy_data
 */

// Only run from CLI
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from command line.\n");
}

// Include dependencies
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/tenant_directory_manager.php';

// Colors for CLI
class Colors {
    const GREEN = "\033[0;32m";
    const RED = "\033[0;31m";
    const YELLOW = "\033[1;33m";
    const BLUE = "\033[0;34m";
    const RESET = "\033[0m";
}

// Parse arguments
$options = getopt('', [
    'dry-run',
    'execute',
    'tenant-id::',
    'backup',
    'no-backup',
    'verbose',
    'help'
]);

// Show help
if (isset($options['help']) || (empty($options))) {
    showHelp();
    exit(0);
}

// Configuration
$dryRun = isset($options['dry-run']);
$execute = isset($options['execute']);
$tenantId = $options['tenant-id'] ?? 'default';
$createBackup = !isset($options['no-backup']);
$verbose = isset($options['verbose']);

// Validate mode
if (!$dryRun && !$execute) {
    echo Colors::RED . "Error: Must specify either --dry-run or --execute\n" . Colors::RESET;
    exit(1);
}

// Start migration
echo "\n";
echo str_repeat("=", 70) . "\n";
echo Colors::BLUE . "FILE MIGRATION TO TENANT STRUCTURE\n" . Colors::RESET;
echo str_repeat("=", 70) . "\n";
echo "Mode:       " . ($dryRun ? Colors::YELLOW . "DRY RUN (no changes)" : Colors::GREEN . "EXECUTE (real migration)") . Colors::RESET . "\n";
echo "Tenant ID:  " . Colors::BLUE . $tenantId . Colors::RESET . "\n";
echo "Backup:     " . ($createBackup ? "Yes" : "No") . "\n";
echo str_repeat("=", 70) . "\n\n";

// Database connection
try {
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    if ($verbose) echo Colors::GREEN . "✓" . Colors::RESET . " Database connected\n\n";
    
} catch (PDOException $e) {
    echo Colors::RED . "✗ Database connection failed: " . $e->getMessage() . "\n" . Colors::RESET;
    exit(1);
}

// Initialize directory manager
$dirManager = new TenantDirectoryManager($conn);

// =====================================================
// STEP 1: CREATE BACKUP
// =====================================================

if ($createBackup && $execute) {
    echo Colors::BLUE . "Step 1: Creating backup...\n" . Colors::RESET;
    
    $backupDir = __DIR__ . '/../backups/migration_' . date('Ymd_His');
    if (!file_exists($backupDir)) {
        mkdir($backupDir, 0755, true);
    }
    
    // Backup database
    $dbBackupFile = $backupDir . '/database_before_migration.sql';
    exec("mysqldump -h " . DB_HOST . " -u " . DB_USER . " -p" . DB_PASS . " " . DB_NAME . " > " . escapeshellarg($dbBackupFile));
    
    // Backup uploads directory
    $uploadsBackup = $backupDir . '/uploads_backup.tar.gz';
    if (file_exists(__DIR__ . '/../uploads')) {
        exec("cd " . escapeshellarg(__DIR__ . '/../uploads') . " && tar -czf " . escapeshellarg($uploadsBackup) . " . 2>&1");
    }
    
    echo Colors::GREEN . "  ✓ Backup created: {$backupDir}\n" . Colors::RESET;
    echo "\n";
}

// =====================================================
// STEP 2: INITIALIZE TARGET TENANT
// =====================================================

echo Colors::BLUE . "Step 2: Initializing target tenant...\n" . Colors::RESET;

if ($execute) {
    $initResult = $dirManager->initializeTenant($tenantId, 'Migrated Legacy Data');
    
    if ($initResult['success']) {
        echo Colors::GREEN . "  ✓ Tenant initialized\n" . Colors::RESET;
    } else {
        echo Colors::RED . "  ✗ Failed: {$initResult['error']}\n" . Colors::RESET;
        exit(1);
    }
} else {
    echo Colors::YELLOW . "  → Would initialize tenant: {$tenantId}\n" . Colors::RESET;
}

echo "\n";

// =====================================================
// STEP 3: SCAN EXISTING FILES
// =====================================================

echo Colors::BLUE . "Step 3: Scanning existing files...\n" . Colors::RESET;

$oldUploadsDir = __DIR__ . '/../uploads';
$fileCategories = [
    'documents' => 'documents',
    'profile_photos' => 'profile_photos',
    'payment_receipts' => 'payment_receipts',
    'reports' => 'reports'
];

$filesFound = [];
$totalSize = 0;

foreach ($fileCategories as $oldDir => $category) {
    $scanPath = $oldUploadsDir . '/' . $oldDir;
    
    if (!file_exists($scanPath)) {
        if ($verbose) echo "  → Directory not found: {$oldDir}\n";
        continue;
    }
    
    $files = scandir($scanPath);
    $count = 0;
    
    foreach ($files as $file) {
        if ($file === '.' || $file === '..' || $file === '.htaccess' || $file === 'index.php') {
            continue;
        }
        
        $fullPath = $scanPath . '/' . $file;
        if (is_file($fullPath)) {
            $filesize = filesize($fullPath);
            $filesFound[] = [
                'old_path' => $fullPath,
                'filename' => $file,
                'category' => $category,
                'size' => $filesize
            ];
            $totalSize += $filesize;
            $count++;
        }
    }
    
    echo "  → Found {$count} files in {$oldDir}\n";
}

echo Colors::GREEN . "  ✓ Total files found: " . count($filesFound) . " (" . formatBytes($totalSize) . ")\n" . Colors::RESET;
echo "\n";

// =====================================================
// STEP 4: MIGRATE FILES
// =====================================================

echo Colors::BLUE . "Step 4: Migrating files to tenant structure...\n" . Colors::RESET;

$migrated = 0;
$failed = 0;
$skipped = 0;

foreach ($filesFound as $fileInfo) {
    $filename = $fileInfo['filename'];
    $category = $fileInfo['category'];
    $oldPath = $fileInfo['old_path'];
    
    // Get new path
    $newDir = $dirManager->getTenantUploadPath($tenantId, $category);
    $newPath = $newDir . '/' . $filename;
    
    if ($dryRun) {
        echo Colors::YELLOW . "  → Would move: {$filename} to {$category}/\n" . Colors::RESET;
        $migrated++;
        continue;
    }
    
    // Execute migration
    if (!file_exists($oldPath)) {
        if ($verbose) echo Colors::YELLOW . "  ⊘ Skipped (not found): {$filename}\n" . Colors::RESET;
        $skipped++;
        continue;
    }
    
    // Create directory if needed
    if (!file_exists($newDir)) {
        mkdir($newDir, 0755, true);
    }
    
    // Copy file (safer than move - can rollback)
    if (copy($oldPath, $newPath)) {
        chmod($newPath, 0644);
        
        // Insert into tenant_files table
        try {
            $stmt = $conn->prepare("
                INSERT INTO tenant_files (
                    tenant_id, file_category, original_filename, stored_filename,
                    file_path, file_size, mime_type, upload_date
                ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $relativePath = 'school_' . $tenantId . '/uploads/' . $category . '/' . $filename;
            $mimeType = mime_content_type($newPath);
            
            $stmt->execute([
                $tenantId,
                $category,
                $filename,
                $filename,
                $relativePath,
                $fileInfo['size'],
                $mimeType
            ]);
        } catch (PDOException $e) {
            // Table might not exist yet, skip DB insert
            if ($verbose) echo Colors::YELLOW . "  ⚠ DB insert skipped: {$e->getMessage()}\n" . Colors::RESET;
        }
        
        if ($verbose) echo Colors::GREEN . "  ✓ Migrated: {$filename}\n" . Colors::RESET;
        $migrated++;
    } else {
        echo Colors::RED . "  ✗ Failed: {$filename}\n" . Colors::RESET;
        $failed++;
    }
}

echo "\n";
echo Colors::GREEN . "  ✓ Migrated: {$migrated} files\n" . Colors::RESET;
if ($failed > 0) {
    echo Colors::RED . "  ✗ Failed: {$failed} files\n" . Colors::RESET;
}
if ($skipped > 0) {
    echo Colors::YELLOW . "  ⊘ Skipped: {$skipped} files\n" . Colors::RESET;
}

echo "\n";

// =====================================================
// STEP 5: UPDATE DATABASE PATHS
// =====================================================

if (!$dryRun) {
    echo Colors::BLUE . "Step 5: Updating database file paths...\n" . Colors::RESET;
    
    $tablesUpdated = 0;
    
    // Update students.profile_photo paths
    try {
        $stmt = $conn->prepare("
            UPDATE students 
            SET profile_photo = CONCAT('school_', ?, '/uploads/profile_photos/', SUBSTRING_INDEX(profile_photo, '/', -1))
            WHERE profile_photo IS NOT NULL 
            AND profile_photo != ''
            AND profile_photo NOT LIKE 'school_%'
        ");
        $stmt->execute([$tenantId]);
        $updated = $stmt->rowCount();
        
        if ($updated > 0) {
            echo Colors::GREEN . "  ✓ Updated {$updated} student photo paths\n" . Colors::RESET;
            $tablesUpdated++;
        }
    } catch (PDOException $e) {
        if ($verbose) echo Colors::YELLOW . "  ⚠ Students table: " . $e->getMessage() . "\n" . Colors::RESET;
    }
    
    // Update payments.receipt_path
    try {
        $stmt = $conn->prepare("
            UPDATE payments 
            SET receipt_path = CONCAT('school_', ?, '/uploads/payment_receipts/', SUBSTRING_INDEX(receipt_path, '/', -1))
            WHERE receipt_path IS NOT NULL 
            AND receipt_path != ''
            AND receipt_path NOT LIKE 'school_%'
        ");
        $stmt->execute([$tenantId]);
        $updated = $stmt->rowCount();
        
        if ($updated > 0) {
            echo Colors::GREEN . "  ✓ Updated {$updated} payment receipt paths\n" . Colors::RESET;
            $tablesUpdated++;
        }
    } catch (PDOException $e) {
        if ($verbose) echo Colors::YELLOW . "  ⚠ Payments table: " . $e->getMessage() . "\n" . Colors::RESET;
    }
    
    if ($tablesUpdated === 0) {
        echo Colors::YELLOW . "  → No database paths to update\n" . Colors::RESET;
    }
    
    echo "\n";
}

// =====================================================
// STEP 6: VERIFY MIGRATION
// =====================================================

echo Colors::BLUE . "Step 6: Verifying migration...\n" . Colors::RESET;

if (!$dryRun) {
    $verification = $dirManager->verifyTenantDirectories($tenantId);
    
    if ($verification['success']) {
        echo Colors::GREEN . "  ✓ All directories verified\n" . Colors::RESET;
        echo "    Directories OK: {$verification['directories_ok']}\n";
        echo "    Writable: {$verification['directories_writable']}\n";
    } else {
        echo Colors::RED . "  ✗ Verification failed\n" . Colors::RESET;
    }
    
    // Count migrated files
    $tenantRoot = $dirManager->getTenantRootPath($tenantId);
    $migratedCount = 0;
    
    foreach ($fileCategories as $category) {
        $path = $tenantRoot . '/uploads/' . $category;
        if (file_exists($path)) {
            $files = array_diff(scandir($path), ['.', '..', '.htaccess', 'index.php']);
            $migratedCount += count($files);
        }
    }
    
    echo "    Files in tenant structure: {$migratedCount}\n";
} else {
    echo Colors::YELLOW . "  → Dry run mode, skipping verification\n" . Colors::RESET;
}

echo "\n";

// =====================================================
// STEP 7: CLEANUP OLD FILES (OPTIONAL)
// =====================================================

if ($execute && $migrated > 0) {
    echo Colors::BLUE . "Step 7: Cleanup old files?\n" . Colors::RESET;
    echo Colors::YELLOW . "  ⚠ Old files are still in uploads/ directory\n" . Colors::RESET;
    echo "  → To remove old files after verifying migration:\n";
    echo "    rm -rf uploads/documents/*\n";
    echo "    rm -rf uploads/profile_photos/*\n";
    echo "    rm -rf uploads/payment_receipts/*\n";
    echo "\n";
}

// =====================================================
// SUMMARY
// =====================================================

echo str_repeat("=", 70) . "\n";
echo Colors::GREEN . "MIGRATION COMPLETE\n" . Colors::RESET;
echo str_repeat("=", 70) . "\n";

if ($dryRun) {
    echo "Mode:            DRY RUN (no changes made)\n";
    echo "Files to migrate: {$migrated}\n";
    echo "\n";
    echo "To execute migration:\n";
    echo "  php tools/migrate_to_tenant_folders.php --execute --tenant-id={$tenantId}\n";
} else {
    echo "Mode:            EXECUTED\n";
    echo "Files migrated:  {$migrated}\n";
    echo "Failed:          {$failed}\n";
    echo "Skipped:         {$skipped}\n";
    echo "Target tenant:   {$tenantId}\n";
    
    if ($createBackup && isset($backupDir)) {
        echo "Backup location: {$backupDir}\n";
    }
    
    echo "\n";
    echo Colors::GREEN . "✓ Migration successful!\n" . Colors::RESET;
    
    if ($failed > 0) {
        echo Colors::YELLOW . "⚠ Some files failed to migrate. Check logs.\n" . Colors::RESET;
    }
}

echo str_repeat("=", 70) . "\n\n";

// =====================================================
// FUNCTIONS
// =====================================================

function showHelp() {
    echo "\n";
    echo str_repeat("=", 70) . "\n";
    echo "FILE MIGRATION TO TENANT STRUCTURE\n";
    echo str_repeat("=", 70) . "\n\n";
    echo "DESCRIPTION:\n";
    echo "  Migrates existing uploaded files to new tenant-based folder structure.\n";
    echo "  Creates backup before migration for safety.\n\n";
    echo "USAGE:\n";
    echo "  php tools/migrate_to_tenant_folders.php [OPTIONS]\n\n";
    echo "MODES:\n";
    echo "  --dry-run        Preview migration without making changes\n";
    echo "  --execute        Perform actual migration\n\n";
    echo "OPTIONS:\n";
    echo "  --tenant-id=ID   Target tenant ID (default: 'default')\n";
    echo "  --no-backup      Skip backup creation (not recommended)\n";
    echo "  --verbose        Show detailed output\n";
    echo "  --help           Show this help message\n\n";
    echo "EXAMPLES:\n";
    echo "  # Preview migration\n";
    echo "  php tools/migrate_to_tenant_folders.php --dry-run --verbose\n\n";
    echo "  # Execute migration to 'default' tenant\n";
    echo "  php tools/migrate_to_tenant_folders.php --execute\n\n";
    echo "  # Execute migration to specific tenant\n";
    echo "  php tools/migrate_to_tenant_folders.php --execute --tenant-id=legacy_school\n\n";
    echo "SAFETY:\n";
    echo "  - Always run --dry-run first to preview\n";
    echo "  - Backup is created automatically (unless --no-backup)\n";
    echo "  - Files are copied (not moved) for safety\n";
    echo "  - Database paths are updated automatically\n\n";
    echo str_repeat("=", 70) . "\n\n";
}

function formatBytes($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}

