<?php
/**
 * Tenant Initialization Script
 * 
 * Automatically creates directory structure and configuration for new tenants
 * 
 * USAGE:
 *   php tools/initialize_tenant.php --tenant-id=abc123 --tenant-name="School Name"
 * 
 * OPTIONS:
 *   --tenant-id      Unique tenant identifier (required)
 *   --tenant-name    Display name for tenant (required)
 *   --email          Contact email (optional)
 *   --phone          Contact phone (optional)
 *   --quota          Storage quota in GB (default: 1)
 *   --skip-email     Skip confirmation email
 *   --force          Force recreate if exists
 *   --verbose        Verbose output
 * 
 * EXAMPLE:
 *   php tools/initialize_tenant.php --tenant-id=soshigh_demo --tenant-name="SOS SOSHIGH" --email=admin@soshigh.edu --quota=5
 */

// Only run from command line
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from command line.\n");
}

// Include dependencies
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/tenant_directory_manager.php';

// Colors for CLI output
class Colors {
    const GREEN = "\033[0;32m";
    const RED = "\033[0;31m";
    const YELLOW = "\033[1;33m";
    const BLUE = "\033[0;34m";
    const RESET = "\033[0m";
    
    public static function success($text) {
        return self::GREEN . "✓ " . $text . self::RESET;
    }
    
    public static function error($text) {
        return self::RED . "✗ " . $text . self::RESET;
    }
    
    public static function warning($text) {
        return self::YELLOW . "⚠ " . $text . self::RESET;
    }
    
    public static function info($text) {
        return self::BLUE . "ℹ " . $text . self::RESET;
    }
}

// Parse command line arguments
$options = getopt('', [
    'tenant-id:',
    'tenant-name:',
    'email::',
    'phone::',
    'quota::',
    'skip-email',
    'force',
    'verbose',
    'help'
]);

// Show help
if (isset($options['help']) || empty($options)) {
    echo "\n";
    echo Colors::info("Tenant Initialization Script") . "\n";
    echo str_repeat("=", 60) . "\n\n";
    echo "USAGE:\n";
    echo "  php tools/initialize_tenant.php --tenant-id=ID --tenant-name=NAME\n\n";
    echo "REQUIRED OPTIONS:\n";
    echo "  --tenant-id      Unique tenant identifier\n";
    echo "  --tenant-name    Display name for tenant\n\n";
    echo "OPTIONAL OPTIONS:\n";
    echo "  --email          Contact email address\n";
    echo "  --phone          Contact phone number\n";
    echo "  --quota          Storage quota in GB (default: 1)\n";
    echo "  --skip-email     Skip sending confirmation email\n";
    echo "  --force          Force recreate if tenant exists\n";
    echo "  --verbose        Show detailed output\n";
    echo "  --help           Show this help message\n\n";
    echo "EXAMPLE:\n";
    echo "  php tools/initialize_tenant.php --tenant-id=soshigh --tenant-name=\"SOS SOSHIGH\" --email=admin@school.edu --quota=5\n\n";
    exit(0);
}

// Verbose mode
$verbose = isset($options['verbose']);

// Validate required arguments
if (!isset($options['tenant-id']) || !isset($options['tenant-name'])) {
    echo Colors::error("Missing required arguments: --tenant-id and --tenant-name") . "\n";
    echo Colors::info("Run with --help for usage information") . "\n";
    exit(1);
}

$tenant_id = $options['tenant-id'];
$tenant_name = $options['tenant-name'];
$email = $options['email'] ?? null;
$phone = $options['phone'] ?? null;
$quota_gb = isset($options['quota']) ? (int)$options['quota'] : 1;
$quota_bytes = $quota_gb * 1024 * 1024 * 1024;
$skip_email = isset($options['skip-email']);
$force = isset($options['force']);

// Start initialization
echo "\n";
echo str_repeat("=", 60) . "\n";
echo Colors::info("TENANT INITIALIZATION SCRIPT") . "\n";
echo str_repeat("=", 60) . "\n";
echo "Tenant ID:   " . Colors::BLUE . $tenant_id . Colors::RESET . "\n";
echo "Tenant Name: " . Colors::BLUE . $tenant_name . Colors::RESET . "\n";
if ($email) echo "Email:       " . $email . "\n";
if ($phone) echo "Phone:       " . $phone . "\n";
echo "Quota:       " . $quota_gb . " GB\n";
echo str_repeat("=", 60) . "\n\n";

// Create database connection
try {
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]
    );
    
    if ($verbose) echo Colors::success("Database connection established") . "\n";
    
} catch (PDOException $e) {
    echo Colors::error("Database connection failed: " . $e->getMessage()) . "\n";
    exit(1);
}

// Initialize directory manager
$dirManager = new TenantDirectoryManager($conn);

// =====================================================
// STEP 1: VALIDATE TENANT ID
// =====================================================

echo "\n" . Colors::info("Step 1: Validating tenant ID...") . "\n";

if (!validateTenantId($tenant_id)) {
    echo Colors::error("Invalid tenant ID format") . "\n";
    echo Colors::info("Tenant ID must be 3-50 characters, alphanumeric with underscore/hyphen only") . "\n";
    exit(1);
}

echo Colors::success("Tenant ID is valid") . "\n";

// =====================================================
// STEP 2: CHECK IF TENANT EXISTS
// =====================================================

echo "\n" . Colors::info("Step 2: Checking if tenant exists...") . "\n";

$tenantRoot = $dirManager->getTenantRootPath($tenant_id);
$tenantExists = file_exists($tenantRoot);

if ($tenantExists && !$force) {
    echo Colors::warning("Tenant directory already exists: $tenantRoot") . "\n";
    echo Colors::info("Use --force to recreate") . "\n";
    exit(1);
}

if ($tenantExists && $force) {
    echo Colors::warning("Tenant exists but --force specified, will recreate") . "\n";
}

// =====================================================
// STEP 3: CREATE DIRECTORY STRUCTURE
// =====================================================

echo "\n" . Colors::info("Step 3: Creating directory structure...") . "\n";

try {
    $result = createDirectoryStructure($tenant_id, $dirManager, $verbose);
    
    if ($result['success']) {
        echo Colors::success("Created {$result['count']} directories") . "\n";
    } else {
        throw new Exception($result['error']);
    }
    
} catch (Exception $e) {
    echo Colors::error("Failed to create directories: " . $e->getMessage()) . "\n";
    cleanupOnFailure($tenant_id, $dirManager);
    exit(1);
}

// =====================================================
// STEP 4: SET DIRECTORY PERMISSIONS
// =====================================================

echo "\n" . Colors::info("Step 4: Setting directory permissions...") . "\n";

try {
    $result = setDirectoryPermissions($tenantRoot, $verbose);
    
    if ($result['success']) {
        echo Colors::success("Permissions set on {$result['count']} items") . "\n";
    } else {
        throw new Exception($result['error']);
    }
    
} catch (Exception $e) {
    echo Colors::error("Failed to set permissions: " . $e->getMessage()) . "\n";
    cleanupOnFailure($tenant_id, $dirManager);
    exit(1);
}

// =====================================================
// STEP 5: CREATE .HTACCESS FILES
// =====================================================

echo "\n" . Colors::info("Step 5: Creating .htaccess security files...") . "\n";

try {
    $result = createHtaccessFiles($tenant_id, $tenantRoot, $verbose);
    
    if ($result['success']) {
        echo Colors::success("Created {$result['count']} .htaccess files") . "\n";
    } else {
        throw new Exception($result['error']);
    }
    
} catch (Exception $e) {
    echo Colors::error("Failed to create .htaccess files: " . $e->getMessage()) . "\n";
    cleanupOnFailure($tenant_id, $dirManager);
    exit(1);
}

// =====================================================
// STEP 6: CREATE INDEX.PHP PROTECTION FILES
// =====================================================

echo "\n" . Colors::info("Step 6: Creating index.php protection files...") . "\n";

try {
    $result = createIndexFiles($tenant_id, $tenantRoot, $verbose);
    
    if ($result['success']) {
        echo Colors::success("Created {$result['count']} index.php files") . "\n";
    } else {
        throw new Exception($result['error']);
    }
    
} catch (Exception $e) {
    echo Colors::error("Failed to create index.php files: " . $e->getMessage()) . "\n";
    cleanupOnFailure($tenant_id, $dirManager);
    exit(1);
}

// =====================================================
// STEP 7: GENERATE CONFIG.JSON
// =====================================================

echo "\n" . Colors::info("Step 7: Generating tenant configuration...") . "\n";

try {
    $result = createTenantConfig($tenant_id, $tenant_name, $email, $phone, $quota_bytes, $tenantRoot, $verbose);
    
    if ($result['success']) {
        echo Colors::success("Configuration file created") . "\n";
    } else {
        throw new Exception($result['error']);
    }
    
} catch (Exception $e) {
    echo Colors::error("Failed to create config: " . $e->getMessage()) . "\n";
    cleanupOnFailure($tenant_id, $dirManager);
    exit(1);
}

// =====================================================
// STEP 8: CREATE README FILE
// =====================================================

echo "\n" . Colors::info("Step 8: Creating README file...") . "\n";

try {
    $readmeContent = "TENANT DIRECTORY\n";
    $readmeContent .= str_repeat("=", 60) . "\n\n";
    $readmeContent .= "Tenant ID:   {$tenant_id}\n";
    $readmeContent .= "Tenant Name: {$tenant_name}\n";
    $readmeContent .= "Created:     " . date('Y-m-d H:i:s') . "\n\n";
    $readmeContent .= "This directory contains all files for {$tenant_name}.\n";
    $readmeContent .= "DO NOT manually modify files in this directory.\n";
    $readmeContent .= "All file operations should go through the application.\n\n";
    $readmeContent .= "Storage Quota: {$quota_gb} GB\n";
    
    file_put_contents($tenantRoot . '/README.txt', $readmeContent);
    chmod($tenantRoot . '/README.txt', 0644);
    
    echo Colors::success("README file created") . "\n";
    
} catch (Exception $e) {
    echo Colors::warning("README creation failed (non-critical): " . $e->getMessage()) . "\n";
}

// =====================================================
// STEP 9: CREATE DATABASE ENTRY (OPTIONAL)
// =====================================================

echo "\n" . Colors::info("Step 9: Creating database entry...") . "\n";

try {
    $result = createDatabaseEntry($conn, $tenant_id, $tenant_name, $email, $phone, $quota_bytes, $verbose);
    
    if ($result['success']) {
        echo Colors::success("Database entry created") . "\n";
    } else {
        // Database entry is optional - warn but continue
        echo Colors::warning("Database entry skipped: " . $result['error']) . "\n";
        echo Colors::info("Tenant directories created successfully, database entry can be added later") . "\n";
    }
    
} catch (Exception $e) {
    // Database entry is optional - warn but continue
    echo Colors::warning("Database entry skipped: " . $e->getMessage()) . "\n";
    echo Colors::info("Tenant directories created successfully, database entry can be added later") . "\n";
}

// =====================================================
// STEP 10: VERIFY INITIALIZATION
// =====================================================

echo "\n" . Colors::info("Step 10: Verifying initialization...") . "\n";

try {
    $verification = verifyInitialization($tenant_id, $dirManager, $verbose);
    
    if ($verification['success']) {
        echo Colors::success("All directories verified") . "\n";
        echo Colors::info("  Directories: {$verification['directories_ok']}") . "\n";
        echo Colors::info("  Writable: {$verification['directories_writable']}") . "\n";
        
        if ($verification['directories_missing'] > 0) {
            echo Colors::warning("  Missing: {$verification['directories_missing']}") . "\n";
        }
    } else {
        throw new Exception("Verification failed");
    }
    
} catch (Exception $e) {
    echo Colors::error("Verification failed: " . $e->getMessage()) . "\n";
    cleanupOnFailure($tenant_id, $dirManager);
    exit(1);
}

// =====================================================
// STEP 11: LOG INITIALIZATION (OPTIONAL)
// =====================================================

echo "\n" . Colors::info("Step 11: Logging initialization...") . "\n";

try {
    logInitialization($conn, $tenant_id, $tenant_name, $verbose);
    echo Colors::success("Initialization logged") . "\n";
    
} catch (Exception $e) {
    echo Colors::warning("Logging skipped (tables may not exist): " . $e->getMessage()) . "\n";
}

// =====================================================
// STEP 12: SEND CONFIRMATION EMAIL
// =====================================================

if (!$skip_email && $email) {
    echo "\n" . Colors::info("Step 12: Sending confirmation email...") . "\n";
    
    try {
        $result = sendConfirmationEmail($email, $tenant_name, $tenant_id, $quota_gb);
        
        if ($result['success']) {
            echo Colors::success("Confirmation email sent to {$email}") . "\n";
        } else {
            echo Colors::warning("Email sending failed (non-critical): {$result['error']}") . "\n";
        }
        
    } catch (Exception $e) {
        echo Colors::warning("Email error (non-critical): " . $e->getMessage()) . "\n";
    }
} else {
    echo "\n" . Colors::info("Step 12: Email skipped") . "\n";
}

// =====================================================
// SUCCESS!
// =====================================================

echo "\n";
echo str_repeat("=", 60) . "\n";
echo Colors::success("TENANT INITIALIZATION COMPLETE!") . "\n";
echo str_repeat("=", 60) . "\n";
echo "Tenant ID:   " . $tenant_id . "\n";
echo "Path:        " . $tenantRoot . "\n";
echo "Quota:       " . $quota_gb . " GB\n";
echo str_repeat("=", 60) . "\n\n";

exit(0);

// =====================================================
// FUNCTIONS
// =====================================================

/**
 * Validate tenant ID format
 */
function validateTenantId($id) {
    // Must be 3-50 characters, alphanumeric with underscore/hyphen only
    return preg_match('/^[a-zA-Z0-9_-]{3,50}$/', $id);
}

/**
 * Create directory structure
 */
function createDirectoryStructure($tenant_id, $dirManager, $verbose = false) {
    global $tenantRoot;
    
    $directories = [
        'uploads/documents',
        'uploads/profile_photos',
        'uploads/payment_receipts',
        'uploads/reports',
        'backups/database',
        'backups/files',
        'logs/access',
        'logs/errors',
        'logs/payments',
        'temp'
    ];
    
    $created = 0;
    
    foreach ($directories as $dir) {
        $fullPath = $tenantRoot . '/' . $dir;
        
        if (!file_exists($fullPath)) {
            if (mkdir($fullPath, 0755, true)) {
                $created++;
                if ($verbose) {
                    echo "  Created: {$dir}\n";
                }
            } else {
                return [
                    'success' => false,
                    'error' => "Failed to create directory: {$dir}"
                ];
            }
        } else {
            if ($verbose) {
                echo "  Exists: {$dir}\n";
            }
        }
    }
    
    return [
        'success' => true,
        'count' => $created
    ];
}

/**
 * Set proper permissions on directories
 */
function setDirectoryPermissions($tenantRoot, $verbose = false) {
    $count = 0;
    
    // Set permissions recursively
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($tenantRoot, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    
    foreach ($iterator as $item) {
        if ($item->isDir()) {
            // Upload directories get 0777, others get 0755
            $isUpload = strpos($item->getPathname(), '/uploads/') !== false;
            $perms = $isUpload ? 0777 : 0755;
            
            if (chmod($item->getPathname(), $perms)) {
                $count++;
                if ($verbose) {
                    $permStr = $isUpload ? '0777' : '0755';
                    echo "  Set {$permStr}: " . str_replace($tenantRoot, '', $item->getPathname()) . "\n";
                }
            }
        }
    }
    
    return [
        'success' => true,
        'count' => $count
    ];
}

/**
 * Create .htaccess security files
 */
function createHtaccessFiles($tenant_id, $tenantRoot, $verbose = false) {
    $count = 0;
    
    // Root .htaccess
    $rootHtaccess = "# Tenant Directory Protection\n";
    $rootHtaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n\n";
    $rootHtaccess .= "Order Deny,Allow\n";
    $rootHtaccess .= "Deny from all\n\n";
    $rootHtaccess .= "<Files ~ \"\\.(jpg|jpeg|png|gif|webp|pdf)$\">\n";
    $rootHtaccess .= "    Allow from all\n";
    $rootHtaccess .= "</Files>\n";
    
    file_put_contents($tenantRoot . '/.htaccess', $rootHtaccess);
    chmod($tenantRoot . '/.htaccess', 0644);
    $count++;
    
    if ($verbose) echo "  Created: /.htaccess\n";
    
    // Upload directory .htaccess
    $uploadHtaccess = "# Upload Directory Protection\n";
    $uploadHtaccess .= "Options -Indexes\n";
    $uploadHtaccess .= "DirectoryIndex disabled\n\n";
    $uploadHtaccess .= "# Prevent PHP execution\n";
    $uploadHtaccess .= "php_flag engine off\n\n";
    $uploadHtaccess .= "<FilesMatch \"\\.(jpg|jpeg|png|gif|webp|pdf|doc|docx|xls|xlsx)$\">\n";
    $uploadHtaccess .= "    Order Allow,Deny\n";
    $uploadHtaccess .= "    Allow from all\n";
    $uploadHtaccess .= "</FilesMatch>\n";
    
    $uploadDirs = ['documents', 'profile_photos', 'payment_receipts', 'reports'];
    
    foreach ($uploadDirs as $dir) {
        $path = $tenantRoot . '/uploads/' . $dir . '/.htaccess';
        file_put_contents($path, $uploadHtaccess);
        chmod($path, 0644);
        $count++;
        
        if ($verbose) echo "  Created: /uploads/{$dir}/.htaccess\n";
    }
    
    return [
        'success' => true,
        'count' => $count
    ];
}

/**
 * Create index.php protection files
 */
function createIndexFiles($tenant_id, $tenantRoot, $verbose = false) {
    $count = 0;
    
    $indexContent = "<?php\n";
    $indexContent .= "// Directory access forbidden\n";
    $indexContent .= "http_response_code(403);\n";
    $indexContent .= "die('Access denied');\n";
    
    $directories = [
        '',  // Root
        'uploads/documents',
        'uploads/profile_photos',
        'uploads/payment_receipts',
        'uploads/reports',
        'backups/database',
        'backups/files',
        'logs/access',
        'logs/errors',
        'logs/payments',
        'temp'
    ];
    
    foreach ($directories as $dir) {
        $path = $tenantRoot . ($dir ? '/' . $dir : '') . '/index.php';
        file_put_contents($path, $indexContent);
        chmod($path, 0644);
        $count++;
        
        if ($verbose) {
            $displayPath = $dir ? "/{$dir}/index.php" : "/index.php";
            echo "  Created: {$displayPath}\n";
        }
    }
    
    return [
        'success' => true,
        'count' => $count
    ];
}

/**
 * Create tenant configuration file
 */
function createTenantConfig($tenant_id, $tenant_name, $email, $phone, $quota_bytes, $tenantRoot, $verbose = false) {
    $config = [
        'tenant_id' => $tenant_id,
        'tenant_name' => $tenant_name,
        'contact_email' => $email,
        'contact_phone' => $phone,
        'created_at' => date('Y-m-d H:i:s'),
        'initialized_by' => 'cli_script',
        'storage_quota' => $quota_bytes,
        'storage_used' => 0,
        'file_retention_days' => 365,
        'auto_cleanup_temp' => true,
        'temp_cleanup_days' => 7,
        'backup_retention_days' => 30,
        'log_retention_days' => 90,
        'status' => 'active'
    ];
    
    $result = file_put_contents(
        $tenantRoot . '/config.json',
        json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
    );
    
    if ($result === false) {
        return [
            'success' => false,
            'error' => 'Failed to write config file'
        ];
    }
    
    chmod($tenantRoot . '/config.json', 0644);
    
    if ($verbose) {
        echo "  Config:\n";
        echo "    Storage quota: " . round($quota_bytes / 1024 / 1024 / 1024, 2) . " GB\n";
        echo "    Temp cleanup: {$config['temp_cleanup_days']} days\n";
        echo "    Backup retention: {$config['backup_retention_days']} days\n";
    }
    
    return [
        'success' => true,
        'config' => $config
    ];
}

/**
 * Verify initialization
 */
function verifyInitialization($tenant_id, $dirManager, $verbose = false) {
    $verification = $dirManager->verifyTenantDirectories($tenant_id);
    
    if ($verbose && isset($verification['details'])) {
        if (!empty($verification['details']['missing'])) {
            echo Colors::warning("  Missing directories:") . "\n";
            foreach ($verification['details']['missing'] as $dir) {
                echo "    - {$dir}\n";
            }
        }
        
        if (!empty($verification['details']['not_writable'])) {
            echo Colors::warning("  Not writable directories:") . "\n";
            foreach ($verification['details']['not_writable'] as $dir) {
                echo "    - {$dir}\n";
            }
        }
    }
    
    return $verification;
}

/**
 * Create database entry for tenant
 */
function createDatabaseEntry($conn, $tenant_id, $tenant_name, $email, $phone, $quota_bytes, $verbose = false) {
    try {
        // Check if tenant already exists in database
        $stmt = $conn->prepare("SELECT id FROM tenants WHERE academy_reference = ?");
        $stmt->execute([$tenant_id]);
        
        if ($stmt->fetch()) {
            if ($verbose) {
                echo "  Tenant already exists in database, skipping insert\n";
            }
            return ['success' => true, 'already_exists' => true];
        }
        
        // Insert tenant
        $stmt = $conn->prepare("
            INSERT INTO tenants (academy_reference, name, contact_email, contact_phone, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([$tenant_id, $tenant_name, $email, $phone]);
        
        // Create storage quota entry
        $stmt = $conn->prepare("
            INSERT INTO tenant_storage_quotas (tenant_id, storage_quota, created_at)
            VALUES (?, ?, NOW())
            ON DUPLICATE KEY UPDATE storage_quota = VALUES(storage_quota)
        ");
        
        $stmt->execute([$tenant_id, $quota_bytes]);
        
        if ($verbose) {
            echo "  Tenant record created in database\n";
            echo "  Storage quota set to " . round($quota_bytes / 1024 / 1024 / 1024, 2) . " GB\n";
        }
        
        return ['success' => true];
        
    } catch (PDOException $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Log initialization to database
 */
function logInitialization($conn, $tenant_id, $tenant_name, $verbose = false) {
    try {
        $stmt = $conn->prepare("
            INSERT INTO file_operations_log (tenant_id, operation, message, metadata, created_at)
            VALUES (?, 'tenant_initialized', ?, ?, NOW())
        ");
        
        $metadata = json_encode([
            'tenant_name' => $tenant_name,
            'initialized_via' => 'cli_script',
            'script_user' => get_current_user(),
            'server' => gethostname()
        ]);
        
        $stmt->execute([
            $tenant_id,
            "Tenant '{$tenant_name}' initialized via CLI script",
            $metadata
        ]);
        
        if ($verbose) {
            echo "  Log entry created in database\n";
        }
        
    } catch (PDOException $e) {
        if ($verbose) {
            echo "  Warning: Database logging table may not exist\n";
        }
    }
}

/**
 * Send confirmation email
 */
function sendConfirmationEmail($email, $tenant_name, $tenant_id, $quota_gb) {
    $subject = "Tenant Initialized - {$tenant_name}";
    
    $message = "Hello,\n\n";
    $message .= "Your tenant directory has been successfully initialized.\n\n";
    $message .= "Tenant Details:\n";
    $message .= "- Tenant ID: {$tenant_id}\n";
    $message .= "- Tenant Name: {$tenant_name}\n";
    $message .= "- Storage Quota: {$quota_gb} GB\n";
    $message .= "- Initialized: " . date('Y-m-d H:i:s') . "\n\n";
    $message .= "Your system is now ready to use.\n\n";
    $message .= "Best regards,\n";
    $message .= "System Administrator\n";
    
    $headers = "From: " . SMTP_FROM_EMAIL . "\r\n";
    $headers .= "Reply-To: " . SMTP_FROM_EMAIL . "\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    if (mail($email, $subject, $message, $headers)) {
        return ['success' => true];
    } else {
        return [
            'success' => false,
            'error' => 'mail() function failed'
        ];
    }
}

/**
 * Cleanup on failure (rollback)
 */
function cleanupOnFailure($tenant_id, $dirManager) {
    echo "\n" . Colors::warning("Initiating rollback...") . "\n";
    
    $tenantRoot = $dirManager->getTenantRootPath($tenant_id);
    
    if (file_exists($tenantRoot)) {
        try {
            deleteDirectoryRecursive($tenantRoot);
            echo Colors::success("Cleaned up partial initialization") . "\n";
        } catch (Exception $e) {
            echo Colors::error("Cleanup failed: " . $e->getMessage()) . "\n";
            echo Colors::warning("Manual cleanup required: {$tenantRoot}") . "\n";
        }
    }
}

/**
 * Delete directory recursively
 */
function deleteDirectoryRecursive($dir) {
    if (!file_exists($dir)) {
        return true;
    }
    
    if (!is_dir($dir)) {
        return unlink($dir);
    }
    
    foreach (scandir($dir) as $item) {
        if ($item == '.' || $item == '..') {
            continue;
        }
        
        if (!deleteDirectoryRecursive($dir . DIRECTORY_SEPARATOR . $item)) {
            return false;
        }
    }
    
    return rmdir($dir);
}

