<?php
/**
 * .htaccess Error Diagnostic Tool
 * 
 * This script helps diagnose 500 Internal Server Errors caused by .htaccess
 * Access via: https://your-domain.com/Multi-Tanent/tools/diagnose_htaccess_error.php
 */

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>.htaccess Error Diagnostic Tool</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            max-width: 900px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #dc3545;
            border-bottom: 3px solid #dc3545;
            padding-bottom: 10px;
        }
        .test-item {
            margin: 15px 0;
            padding: 15px;
            border-left: 4px solid #ddd;
            background: #f9f9f9;
        }
        .test-item.pass {
            border-left-color: #28a745;
            background: #d4edda;
        }
        .test-item.fail {
            border-left-color: #dc3545;
            background: #f8d7da;
        }
        .test-item.warning {
            border-left-color: #ffc107;
            background: #fff3cd;
        }
        .test-label {
            font-weight: bold;
            margin-bottom: 5px;
        }
        .test-result {
            margin-top: 10px;
            padding: 10px;
            background: white;
            border-radius: 4px;
            font-family: monospace;
            font-size: 0.9em;
        }
        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            display: inline-block;
            margin: 2px 0;
        }
        .fix-box {
            background: #d1ecf1;
            border-left: 4px solid #0c5460;
            padding: 15px;
            margin: 20px 0;
            border-radius: 4px;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            margin: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 .htaccess Error Diagnostic Tool</h1>
        <p>This tool helps identify why your .htaccess file is causing 500 errors.</p>
        
        <?php
        $issues = [];
        $warnings = [];
        
        // Test 1: Check if .htaccess exists
        $htaccessExists = file_exists(__DIR__ . '/../.htaccess');
        $status1 = $htaccessExists ? 'pass' : 'fail';
        if (!$htaccessExists) {
            $issues[] = '.htaccess file not found in root directory';
        }
        
        echo "<div class='test-item {$status1}'>";
        echo "<div class='test-label'>1. .htaccess File Exists</div>";
        echo "<div class='test-result'>" . ($htaccessExists ? "✅ File found at: " . realpath(__DIR__ . '/../.htaccess') : "❌ File not found") . "</div>";
        echo "</div>";
        
        // Test 2: Check .htaccess readability
        if ($htaccessExists) {
            $readable = is_readable(__DIR__ . '/../.htaccess');
            $status2 = $readable ? 'pass' : 'fail';
            if (!$readable) {
                $issues[] = '.htaccess file is not readable';
            }
            
            echo "<div class='test-item {$status2}'>";
            echo "<div class='test-label'>2. .htaccess File Readable</div>";
            echo "<div class='test-result'>" . ($readable ? "✅ File is readable" : "❌ File is not readable (check permissions)") . "</div>";
            echo "</div>";
            
            // Test 3: Read .htaccess content
            $htaccessContent = file_get_contents(__DIR__ . '/../.htaccess');
            $contentLength = strlen($htaccessContent);
            
            echo "<div class='test-item " . ($contentLength > 0 ? 'pass' : 'fail') . "'>";
            echo "<div class='test-label'>3. .htaccess Content</div>";
            echo "<div class='test-result'>File size: {$contentLength} bytes</div>";
            echo "</div>";
            
            // Test 4: Check for common problematic directives
            $problematicPatterns = [
                '/Order\s+allow,deny/' => 'Apache 2.4+ uses "Require" instead of "Order allow,deny"',
                '/Deny\s+from\s+all/' => 'Apache 2.4+ uses "Require all denied" instead of "Deny from all"',
                '/Allow\s+from\s+all/' => 'Apache 2.4+ uses "Require all granted" instead of "Allow from all"',
            ];
            
            $foundIssues = [];
            foreach ($problematicPatterns as $pattern => $message) {
                if (preg_match($pattern, $htaccessContent)) {
                    $foundIssues[] = $message;
                }
            }
            
            if (!empty($foundIssues)) {
                echo "<div class='test-item warning'>";
                echo "<div class='test-label'>4. Apache 2.4+ Compatibility Check</div>";
                echo "<div class='test-result'>";
                echo "⚠️ Found potentially incompatible directives:<br>";
                foreach ($foundIssues as $issue) {
                    echo "• {$issue}<br>";
                }
                echo "</div>";
                echo "</div>";
                $warnings = array_merge($warnings, $foundIssues);
            } else {
                echo "<div class='test-item pass'>";
                echo "<div class='test-label'>4. Apache 2.4+ Compatibility</div>";
                echo "<div class='test-result'>✅ No legacy directives found</div>";
                echo "</div>";
            }
            
            // Test 5: Check for mod_rewrite usage
            $usesRewrite = strpos($htaccessContent, 'RewriteEngine') !== false || 
                          strpos($htaccessContent, 'RewriteRule') !== false;
            
            if ($usesRewrite) {
                echo "<div class='test-item warning'>";
                echo "<div class='test-label'>5. mod_rewrite Usage Detected</div>";
                echo "<div class='test-result'>⚠️ .htaccess uses mod_rewrite. Ensure mod_rewrite is enabled.</div>";
                echo "</div>";
                $warnings[] = 'mod_rewrite may not be enabled';
            }
            
            // Test 6: Check for mod_headers usage
            $usesHeaders = strpos($htaccessContent, 'Header always set') !== false;
            if ($usesHeaders) {
                echo "<div class='test-item warning'>";
                echo "<div class='test-label'>6. mod_headers Usage Detected</div>";
                echo "<div class='test-result'>⚠️ .htaccess uses mod_headers. If errors occur, mod_headers may not be enabled.</div>";
                echo "</div>";
                $warnings[] = 'mod_headers may not be enabled';
            }
            
            // Test 7: Check for mod_expires usage
            $usesExpires = strpos($htaccessContent, 'ExpiresActive') !== false;
            if ($usesExpires) {
                echo "<div class='test-item warning'>";
                echo "<div class='test-label'>7. mod_expires Usage Detected</div>";
                echo "<div class='test-result'>⚠️ .htaccess uses mod_expires. If errors occur, mod_expires may not be enabled.</div>";
                echo "</div>";
                $warnings[] = 'mod_expires may not be enabled';
            }
            
            // Test 8: Check RewriteBase
            if (preg_match('/RewriteBase\s+(.+)/', $htaccessContent, $matches)) {
                $rewriteBase = trim($matches[1]);
                echo "<div class='test-item pass'>";
                echo "<div class='test-label'>8. RewriteBase Setting</div>";
                echo "<div class='test-result'>Found: <code>{$rewriteBase}</code><br>";
                echo "Current path: <code>" . $_SERVER['SCRIPT_NAME'] . "</code><br>";
                echo "Verify this matches your installation path.</div>";
                echo "</div>";
            }
        }
        
        // Test 9: Check PHP version
        $phpVersion = phpversion();
        echo "<div class='test-item pass'>";
        echo "<div class='test-label'>9. PHP Version</div>";
        echo "<div class='test-result'>PHP {$phpVersion}</div>";
        echo "</div>";
        
        // Test 10: Check server software
        $serverSoftware = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
        echo "<div class='test-item pass'>";
        echo "<div class='test-label'>10. Server Software</div>";
        echo "<div class='test-result'>{$serverSoftware}</div>";
        echo "</div>";
        
        // Summary
        echo "<div style='margin-top: 30px; padding: 20px; background: #f8f9fa; border-radius: 4px;'>";
        echo "<h2>📋 Summary</h2>";
        
        if (empty($issues) && empty($warnings)) {
            echo "<p style='color: #28a745; font-weight: bold;'>✅ No obvious issues found. Check Apache error logs for specific errors.</p>";
        } else {
            if (!empty($issues)) {
                echo "<h3 style='color: #dc3545;'>❌ Critical Issues:</h3><ul>";
                foreach ($issues as $issue) {
                    echo "<li>{$issue}</li>";
                }
                echo "</ul>";
            }
            
            if (!empty($warnings)) {
                echo "<h3 style='color: #ffc107;'>⚠️ Warnings:</h3><ul>";
                foreach ($warnings as $warning) {
                    echo "<li>{$warning}</li>";
                }
                echo "</ul>";
            }
        }
        echo "</div>";
        
        // Fix suggestions
        if (!empty($issues) || !empty($warnings)) {
            echo "<div class='fix-box'>";
            echo "<h3>🔧 Recommended Fixes:</h3>";
            echo "<ol>";
            echo "<li><strong>Backup current .htaccess:</strong> Rename to .htaccess.backup</li>";
            echo "<li><strong>Try simplified version:</strong> Use the compatible .htaccess version (see below)</li>";
            echo "<li><strong>Check Apache error logs:</strong> Look for specific error messages</li>";
            echo "<li><strong>Test in stages:</strong> Add rules incrementally to find the problematic directive</li>";
            echo "<li><strong>Contact hosting:</strong> Verify mod_rewrite, mod_headers, and AllowOverride settings</li>";
            echo "</ol>";
            echo "</div>";
        }
        ?>
        
        <div style="margin-top: 30px; text-align: center;">
            <a href="create_safe_htaccess.php" class="btn">🔧 Create Safe .htaccess Version</a>
            <a href="../" class="btn">← Back to Home</a>
        </div>
    </div>
</body>
</html>

