<?php
/**
 * Test Script for Tenant File System
 * 
 * Run this to verify the file system is working correctly
 * Access: http://your-domain/test_tenant_filesystem.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';
require_once 'includes/tenant_filesystem.php';

// Create database connection
try {
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]
    );
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Initialize
$fs = new TenantFileSystem($conn);

// Test tenant ID
$test_tenant_id = 'test_' . date('Ymd_His');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Tenant File System Test</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 1200px;
            margin: 20px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .test-section {
            background: white;
            padding: 20px;
            margin: 15px 0;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .success {
            color: #4caf50;
            font-weight: bold;
        }
        .error {
            color: #f44336;
            font-weight: bold;
        }
        .info {
            color: #2196f3;
        }
        pre {
            background: #f9f9f9;
            padding: 15px;
            border-left: 4px solid #2196f3;
            overflow-x: auto;
        }
        h1 {
            color: #333;
            border-bottom: 3px solid #2196f3;
            padding-bottom: 10px;
        }
        h2 {
            color: #555;
            margin-top: 0;
        }
        .result {
            padding: 10px;
            background: #f0f0f0;
            border-radius: 4px;
            margin: 10px 0;
        }
        .test-status {
            display: inline-block;
            padding: 5px 10px;
            border-radius: 4px;
            font-weight: bold;
        }
        .pass { background: #4caf50; color: white; }
        .fail { background: #f44336; color: white; }
        .warning { background: #ff9800; color: white; }
    </style>
</head>
<body>
    <h1>🧪 Tenant File System Test Suite</h1>
    <p>Testing tenant file system functionality...</p>
    <p><strong>Test Tenant ID:</strong> <code><?= htmlspecialchars($test_tenant_id) ?></code></p>

    <!-- TEST 1: Create Directory -->
    <div class="test-section">
        <h2>Test 1: Create Tenant Directory Structure</h2>
        <?php
        $result = $fs->createTenantDirectory($test_tenant_id);
        if ($result['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ ' . htmlspecialchars($result['message']) . '</p>';
            echo '<p><strong>Path:</strong> <code>' . htmlspecialchars($result['path']) . '</code></p>';
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Error: ' . htmlspecialchars($result['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 2: Get Tenant Paths -->
    <div class="test-section">
        <h2>Test 2: Get Tenant Paths</h2>
        <?php
        $basePath = $fs->getTenantPath($test_tenant_id);
        $docsPath = $fs->getTenantPath($test_tenant_id, 'documents');
        $photosPath = $fs->getTenantPath($test_tenant_id, 'profile_photos');
        
        if ($basePath && $docsPath && $photosPath) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p><strong>Base Path:</strong> <code>' . htmlspecialchars($basePath) . '</code></p>';
            echo '<p><strong>Documents Path:</strong> <code>' . htmlspecialchars($docsPath) . '</code></p>';
            echo '<p><strong>Photos Path:</strong> <code>' . htmlspecialchars($photosPath) . '</code></p>';
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Failed to get tenant paths</p>';
        }
        ?>
    </div>

    <!-- TEST 3: Path Validation -->
    <div class="test-section">
        <h2>Test 3: Path Traversal Prevention</h2>
        <?php
        // Valid path
        $validPath = 'uploads/documents/test.pdf';
        $validResult = $fs->validateTenantAccess($test_tenant_id, $validPath);
        
        // Invalid paths (should be blocked)
        $maliciousPaths = [
            '../../../etc/passwd',
            'uploads/../../other_tenant/file.pdf',
            '../config.php',
            'uploads/../../../database/users.sql'
        ];
        
        $allBlocked = true;
        foreach ($maliciousPaths as $path) {
            if ($fs->validateTenantAccess($test_tenant_id, $path)) {
                $allBlocked = false;
                break;
            }
        }
        
        if ($validResult && $allBlocked) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Valid paths allowed</p>';
            echo '<p class="success">✅ Path traversal attacks blocked (' . count($maliciousPaths) . ' attempts)</p>';
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            if (!$validResult) {
                echo '<p class="error">❌ Valid path incorrectly blocked</p>';
            }
            if (!$allBlocked) {
                echo '<p class="error">❌ Security Issue: Some malicious paths were not blocked!</p>';
            }
        }
        ?>
    </div>

    <!-- TEST 4: Create Test File for Upload -->
    <div class="test-section">
        <h2>Test 4: Simulate File Upload</h2>
        <?php
        // Create a temporary test file
        $tempDir = sys_get_temp_dir();
        $testFile = $tempDir . '/test_upload.txt';
        $testContent = "Test file created at " . date('Y-m-d H:i:s') . "\n";
        $testContent .= "This is a test file for the tenant file system.\n";
        file_put_contents($testFile, $testContent);
        
        // Simulate $_FILES array
        $simulatedFile = [
            'name' => 'test_document.txt',
            'type' => 'text/plain',
            'tmp_name' => $testFile,
            'error' => UPLOAD_ERR_OK,
            'size' => filesize($testFile)
        ];
        
        $uploadResult = $fs->uploadFile($test_tenant_id, $simulatedFile, 'documents', 'test_file');
        
        if ($uploadResult['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ File uploaded successfully</p>';
            echo '<p><strong>Filename:</strong> <code>' . htmlspecialchars($uploadResult['filename']) . '</code></p>';
            echo '<p><strong>Size:</strong> ' . htmlspecialchars($uploadResult['size']) . ' bytes</p>';
            echo '<p><strong>URL:</strong> <code>' . htmlspecialchars($uploadResult['url']) . '</code></p>';
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Upload failed: ' . htmlspecialchars($uploadResult['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 5: List Files -->
    <div class="test-section">
        <h2>Test 5: List Tenant Files</h2>
        <?php
        $filesResult = $fs->getTenantFiles($test_tenant_id, 'documents');
        
        if ($filesResult['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Successfully retrieved file list</p>';
            echo '<p><strong>Total Files:</strong> ' . $filesResult['count'] . '</p>';
            
            if ($filesResult['count'] > 0) {
                echo '<h3>Files:</h3>';
                foreach ($filesResult['files'] as $file) {
                    echo '<div style="padding: 10px; background: #fafafa; margin: 5px 0; border-radius: 4px;">';
                    echo '<strong>' . htmlspecialchars($file['name']) . '</strong><br>';
                    echo 'Size: ' . htmlspecialchars($file['size_formatted']) . ' | ';
                    echo 'Modified: ' . htmlspecialchars($file['modified']) . '<br>';
                    echo 'Type: ' . htmlspecialchars($file['type']) . ' | ';
                    echo 'Extension: ' . htmlspecialchars($file['extension']);
                    echo '</div>';
                }
            }
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Failed to list files: ' . htmlspecialchars($filesResult['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 6: Storage Usage -->
    <div class="test-section">
        <h2>Test 6: Storage Usage Calculation</h2>
        <?php
        $storageInfo = $fs->getStorageUsage($test_tenant_id);
        
        if ($storageInfo['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Storage usage calculated successfully</p>';
            echo '<p><strong>Used:</strong> ' . htmlspecialchars($storageInfo['used_formatted']) . '</p>';
            echo '<p><strong>Quota:</strong> ' . htmlspecialchars($storageInfo['quota_formatted']) . '</p>';
            echo '<p><strong>Available:</strong> ' . htmlspecialchars($storageInfo['available_formatted']) . '</p>';
            echo '<p><strong>Usage:</strong> ' . htmlspecialchars($storageInfo['percentage']) . '%</p>';
            
            // Visual bar
            echo '<div style="width: 100%; height: 30px; background: #e0e0e0; border-radius: 5px; overflow: hidden; margin-top: 10px;">';
            $color = $storageInfo['percentage'] > 80 ? '#f44336' : '#4caf50';
            echo '<div style="width: ' . $storageInfo['percentage'] . '%; height: 100%; background: ' . $color . ';"></div>';
            echo '</div>';
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Failed to get storage usage: ' . htmlspecialchars($storageInfo['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 7: File Type Validation -->
    <div class="test-section">
        <h2>Test 7: File Type Validation</h2>
        <?php
        // Try to upload an invalid file type
        $invalidFile = $tempDir . '/test_invalid.exe';
        file_put_contents($invalidFile, 'fake executable');
        
        $invalidUpload = [
            'name' => 'malicious.exe',
            'type' => 'application/x-msdownload',
            'tmp_name' => $invalidFile,
            'error' => UPLOAD_ERR_OK,
            'size' => filesize($invalidFile)
        ];
        
        $invalidResult = $fs->uploadFile($test_tenant_id, $invalidUpload, 'documents');
        
        if (!$invalidResult['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Invalid file type correctly rejected</p>';
            echo '<p class="info">Error message: ' . htmlspecialchars($invalidResult['error']) . '</p>';
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Security Issue: Invalid file type was accepted!</p>';
        }
        
        // Cleanup
        @unlink($invalidFile);
        ?>
    </div>

    <!-- TEST 8: Delete File -->
    <div class="test-section">
        <h2>Test 8: Delete File</h2>
        <?php
        // Get the uploaded file
        $filesResult = $fs->getTenantFiles($test_tenant_id, 'documents');
        
        if ($filesResult['success'] && $filesResult['count'] > 0) {
            $fileToDelete = $filesResult['files'][0]['name'];
            $deleteResult = $fs->deleteFile($test_tenant_id, $fileToDelete, 'documents');
            
            if ($deleteResult['success']) {
                echo '<span class="test-status pass">✅ PASS</span>';
                echo '<div class="result">';
                echo '<p class="success">✅ File deleted successfully</p>';
                echo '<p><strong>Deleted:</strong> <code>' . htmlspecialchars($fileToDelete) . '</code></p>';
                echo '</div>';
            } else {
                echo '<span class="test-status fail">❌ FAIL</span>';
                echo '<p class="error">❌ Failed to delete file: ' . htmlspecialchars($deleteResult['error']) . '</p>';
            }
        } else {
            echo '<span class="test-status warning">⚠️ SKIP</span>';
            echo '<p>No files available to test deletion</p>';
        }
        ?>
    </div>

    <!-- TEST 9: Database Logging -->
    <div class="test-section">
        <h2>Test 9: Database Logging</h2>
        <?php
        try {
            $stmt = $conn->query("SELECT COUNT(*) FROM file_operations_log WHERE tenant_id = '{$test_tenant_id}'");
            $logCount = $stmt->fetchColumn();
            
            if ($logCount > 0) {
                echo '<span class="test-status pass">✅ PASS</span>';
                echo '<div class="result">';
                echo '<p class="success">✅ Operations are being logged</p>';
                echo '<p><strong>Log Entries:</strong> ' . $logCount . '</p>';
                
                // Show recent logs
                $stmt = $conn->prepare("SELECT operation, message, created_at FROM file_operations_log WHERE tenant_id = ? ORDER BY created_at DESC LIMIT 5");
                $stmt->execute([$test_tenant_id]);
                $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if ($logs) {
                    echo '<h3>Recent Operations:</h3>';
                    foreach ($logs as $log) {
                        echo '<div style="padding: 8px; background: #fafafa; margin: 3px 0; font-size: 0.9em;">';
                        echo '<strong>' . htmlspecialchars($log['operation']) . '</strong>: ';
                        echo htmlspecialchars($log['message']) . ' ';
                        echo '<em>(' . htmlspecialchars($log['created_at']) . ')</em>';
                        echo '</div>';
                    }
                }
                echo '</div>';
            } else {
                echo '<span class="test-status warning">⚠️ WARNING</span>';
                echo '<p>Logging may not be working (0 log entries found)</p>';
            }
        } catch (PDOException $e) {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Database logging error: ' . htmlspecialchars($e->getMessage()) . '</p>';
            echo '<p class="info">Make sure you imported database/TENANT_FILESYSTEM_TABLES.sql</p>';
        }
        ?>
    </div>

    <!-- TEST 10: Cleanup -->
    <div class="test-section">
        <h2>Test 10: Cleanup Test Files</h2>
        <?php
        // Remove test tenant directory
        function deleteDirectory($dir) {
            if (!file_exists($dir)) return true;
            if (!is_dir($dir)) return unlink($dir);
            
            foreach (scandir($dir) as $item) {
                if ($item == '.' || $item == '..') continue;
                if (!deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) return false;
            }
            return rmdir($dir);
        }
        
        $testPath = $fs->getTenantPath($test_tenant_id);
        if ($testPath && file_exists($testPath)) {
            if (deleteDirectory($testPath)) {
                echo '<span class="test-status pass">✅ PASS</span>';
                echo '<p class="success">✅ Test files cleaned up successfully</p>';
            } else {
                echo '<span class="test-status warning">⚠️ WARNING</span>';
                echo '<p>Failed to cleanup test directory. You may need to manually delete:</p>';
                echo '<p><code>' . htmlspecialchars($testPath) . '</code></p>';
            }
        }
        
        // Cleanup test logs
        try {
            $stmt = $conn->prepare("DELETE FROM file_operations_log WHERE tenant_id = ?");
            $stmt->execute([$test_tenant_id]);
            echo '<p class="info">Test log entries cleaned from database</p>';
        } catch (PDOException $e) {
            // Silent fail
        }
        ?>
    </div>

    <!-- Summary -->
    <div class="test-section" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
        <h2 style="color: white;">🎉 Test Suite Complete!</h2>
        <p>If all tests passed, your tenant file system is ready for production!</p>
        <h3 style="color: white;">Next Steps:</h3>
        <ul>
            <li>Review the <code>TENANT_FILESYSTEM_GUIDE.md</code> for full documentation</li>
            <li>Check <code>TENANT_FILESYSTEM_USAGE_EXAMPLES.php</code> for code examples</li>
            <li>Configure file type restrictions and size limits if needed</li>
            <li>Set up proper file permissions on production server</li>
            <li>Implement file upload forms in your application</li>
        </ul>
    </div>

    <div style="text-align: center; padding: 20px; color: #888;">
        <p>Tenant File System v1.0</p>
        <p><em>Last run: <?= date('Y-m-d H:i:s') ?></em></p>
    </div>
</body>
</html>

