<?php
/**
 * Test Script for Tenant Directory Manager
 * 
 * Verifies the TenantDirectoryManager class functionality
 * Access: http://your-domain/test_tenant_directory_manager.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';
require_once 'includes/tenant_directory_manager.php';

// Create database connection
try {
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]
    );
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Initialize manager
$manager = new TenantDirectoryManager($conn);

// Test tenant ID
$test_tenant_id = 'test_' . date('Ymd_His');
$test_tenant_name = 'Test School ' . date('Y-m-d H:i:s');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Tenant Directory Manager Test</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            max-width: 1200px;
            margin: 20px auto;
            padding: 20px;
            background: #f5f7fa;
        }
        .test-section {
            background: white;
            padding: 25px;
            margin: 20px 0;
            border-radius: 10px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .success { color: #10b981; font-weight: 600; }
        .error { color: #ef4444; font-weight: 600; }
        .warning { color: #f59e0b; font-weight: 600; }
        .info { color: #3b82f6; }
        pre {
            background: #f9fafb;
            padding: 15px;
            border-left: 4px solid #3b82f6;
            overflow-x: auto;
            border-radius: 6px;
        }
        h1 {
            color: #1f2937;
            border-bottom: 3px solid #3b82f6;
            padding-bottom: 15px;
            margin-bottom: 30px;
        }
        h2 {
            color: #374151;
            margin-top: 0;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .result {
            padding: 12px;
            background: #f9fafb;
            border-radius: 6px;
            margin: 12px 0;
            border-left: 4px solid #e5e7eb;
        }
        .test-status {
            display: inline-block;
            padding: 6px 14px;
            border-radius: 6px;
            font-weight: 600;
            font-size: 14px;
        }
        .pass { background: #10b981; color: white; }
        .fail { background: #ef4444; color: white; }
        .skip { background: #f59e0b; color: white; }
        .metric {
            display: inline-block;
            background: #3b82f6;
            color: white;
            padding: 4px 10px;
            border-radius: 4px;
            font-size: 13px;
            margin: 4px;
        }
        .file-list {
            background: #f9fafb;
            padding: 10px;
            border-radius: 6px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
        }
        .file-list div {
            padding: 4px 0;
            border-bottom: 1px solid #e5e7eb;
        }
        .file-list div:last-child {
            border-bottom: none;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e5e7eb;
        }
        th {
            background: #f9fafb;
            font-weight: 600;
            color: #374151;
        }
        .progress-bar {
            width: 100%;
            height: 30px;
            background: #e5e7eb;
            border-radius: 6px;
            overflow: hidden;
            margin: 10px 0;
        }
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #3b82f6, #10b981);
            transition: width 0.3s ease;
        }
    </style>
</head>
<body>
    <h1>🧪 Tenant Directory Manager Test Suite</h1>
    <p><strong>Test Tenant ID:</strong> <code><?= htmlspecialchars($test_tenant_id) ?></code></p>
    <p><strong>Test Tenant Name:</strong> <?= htmlspecialchars($test_tenant_name) ?></p>

    <!-- TEST 1: Initialize Tenant -->
    <div class="test-section">
        <h2>🔧 Test 1: Initialize Tenant Directory Structure</h2>
        <?php
        $result = $manager->initializeTenant($test_tenant_id, $test_tenant_name);
        
        if ($result['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ ' . htmlspecialchars($result['message']) . '</p>';
            echo '<p><strong>Path:</strong> <code>' . htmlspecialchars($result['path']) . '</code></p>';
            echo '<span class="metric">Directories: ' . $result['directories_created'] . '</span>';
            
            if (isset($result['already_exists']) && $result['already_exists']) {
                echo '<p class="warning">⚠️ Note: Tenant was already initialized</p>';
            }
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Error: ' . htmlspecialchars($result['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 2: Get Tenant Paths -->
    <div class="test-section">
        <h2>📂 Test 2: Get Tenant Paths</h2>
        <?php
        $rootPath = $manager->getTenantRootPath($test_tenant_id);
        $paths_ok = $rootPath !== false;
        
        $documentsPath = $manager->getTenantUploadPath($test_tenant_id, 'documents');
        $paths_ok = $paths_ok && $documentsPath !== false;
        
        $photosPath = $manager->getTenantUploadPath($test_tenant_id, 'profile_photos');
        $paths_ok = $paths_ok && $photosPath !== false;
        
        $backupPath = $manager->getTenantBackupPath($test_tenant_id, 'database');
        $paths_ok = $paths_ok && $backupPath !== false;
        
        $logPath = $manager->getTenantLogPath($test_tenant_id, 'errors');
        $paths_ok = $paths_ok && $logPath !== false;
        
        if ($paths_ok) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ All paths retrieved successfully</p>';
            echo '<div class="file-list">';
            echo '<div><strong>Root:</strong> ' . htmlspecialchars($rootPath) . '</div>';
            echo '<div><strong>Documents:</strong> ' . htmlspecialchars($documentsPath) . '</div>';
            echo '<div><strong>Photos:</strong> ' . htmlspecialchars($photosPath) . '</div>';
            echo '<div><strong>Backups (DB):</strong> ' . htmlspecialchars($backupPath) . '</div>';
            echo '<div><strong>Logs (Errors):</strong> ' . htmlspecialchars($logPath) . '</div>';
            echo '</div></div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Failed to retrieve some paths</p>';
        }
        ?>
    </div>

    <!-- TEST 3: Verify Directories -->
    <div class="test-section">
        <h2>✓ Test 3: Verify Directory Structure</h2>
        <?php
        $verification = $manager->verifyTenantDirectories($test_tenant_id);
        
        if ($verification['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ All directories verified successfully</p>';
            echo '<table>';
            echo '<tr><th>Metric</th><th>Value</th></tr>';
            echo '<tr><td>Directories Exist</td><td>' . $verification['directories_ok'] . '</td></tr>';
            echo '<tr><td>Directories Missing</td><td>' . $verification['directories_missing'] . '</td></tr>';
            echo '<tr><td>Directories Writable</td><td>' . $verification['directories_writable'] . '</td></tr>';
            echo '<tr><td>Directories Not Writable</td><td>' . $verification['directories_not_writable'] . '</td></tr>';
            echo '</table>';
            
            if ($verification['directories_missing'] > 0) {
                echo '<p class="warning">⚠️ Missing directories:</p>';
                echo '<pre>' . implode("\n", $verification['details']['missing']) . '</pre>';
            }
            
            if ($verification['directories_not_writable'] > 0) {
                echo '<p class="error">❌ Not writable directories:</p>';
                echo '<pre>' . implode("\n", $verification['details']['not_writable']) . '</pre>';
            }
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Verification failed: ' . htmlspecialchars($verification['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 4: Configuration Management -->
    <div class="test-section">
        <h2>⚙️ Test 4: Configuration Management</h2>
        <?php
        $config = $manager->getTenantConfig($test_tenant_id);
        
        if ($config !== false) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Configuration file exists and is readable</p>';
            echo '<p><strong>Configuration:</strong></p>';
            echo '<pre>' . json_encode($config, JSON_PRETTY_PRINT) . '</pre>';
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Configuration file not found</p>';
        }
        ?>
    </div>

    <!-- TEST 5: Update Configuration -->
    <div class="test-section">
        <h2>📝 Test 5: Update Configuration</h2>
        <?php
        $updateResult = $manager->createTenantConfig($test_tenant_id, [
            'test_setting' => 'test_value',
            'enable_feature_x' => true,
            'max_users' => 1000
        ]);
        
        if ($updateResult['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Configuration updated successfully</p>';
            echo '<p><strong>Updated config includes:</strong></p>';
            echo '<div class="file-list">';
            echo '<div>test_setting: test_value</div>';
            echo '<div>enable_feature_x: true</div>';
            echo '<div>max_users: 1000</div>';
            echo '</div></div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Update failed: ' . htmlspecialchars($updateResult['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 6: Calculate Storage Usage -->
    <div class="test-section">
        <h2>💾 Test 6: Calculate Storage Usage</h2>
        <?php
        // Helper function to format bytes
        function formatBytes($bytes, $precision = 2) {
            $units = ['B', 'KB', 'MB', 'GB', 'TB'];
            for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
                $bytes /= 1024;
            }
            return round($bytes, $precision) . ' ' . $units[$i];
        }
        
        $storage = $manager->calculateTenantStorageUsage($test_tenant_id);
        
        if ($storage['success']) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Storage calculated successfully</p>';
            
            echo '<table>';
            echo '<tr><th>Category</th><th>Size</th></tr>';
            echo '<tr><td>Uploads</td><td>' . formatBytes($storage['usage']['uploads']) . '</td></tr>';
            echo '<tr><td>Backups</td><td>' . formatBytes($storage['usage']['backups']) . '</td></tr>';
            echo '<tr><td>Logs</td><td>' . formatBytes($storage['usage']['logs']) . '</td></tr>';
            echo '<tr><td>Temp</td><td>' . formatBytes($storage['usage']['temp']) . '</td></tr>';
            echo '<tr><td>Other</td><td>' . formatBytes($storage['usage']['other']) . '</td></tr>';
            echo '<tr style="font-weight:600; background:#f9fafb;"><td>TOTAL</td><td>' . $storage['total_formatted'] . '</td></tr>';
            echo '</table>';
            
            echo '<p><strong>Quota Information:</strong></p>';
            echo '<div class="progress-bar">';
            echo '<div class="progress-fill" style="width: ' . $storage['percentage'] . '%;"></div>';
            echo '</div>';
            echo '<p>Quota: ' . $storage['quota_formatted'] . ' | Used: ' . $storage['total_formatted'] . ' (' . $storage['percentage'] . '%) | Available: ' . $storage['available_formatted'] . '</p>';
            
            if ($storage['percentage'] > 80) {
                echo '<p class="warning">⚠️ WARNING: Storage usage over 80%!</p>';
            }
            echo '</div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Calculation failed: ' . htmlspecialchars($storage['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 7: Update Quota -->
    <div class="test-section">
        <h2>📊 Test 7: Update Storage Quota</h2>
        <?php
        $newQuota = 2147483648; // 2GB
        $quotaResult = $manager->updateTenantQuota($test_tenant_id, $newQuota);
        
        if ($quotaResult['success']) {
            $updatedQuota = $manager->getTenantQuota($test_tenant_id);
            
            if ($updatedQuota === $newQuota) {
                echo '<span class="test-status pass">✅ PASS</span>';
                echo '<div class="result">';
                echo '<p class="success">✅ Quota updated successfully</p>';
                echo '<p>New quota: ' . formatBytes($newQuota) . '</p>';
                echo '</div>';
            } else {
                echo '<span class="test-status fail">❌ FAIL</span>';
                echo '<p class="error">❌ Quota was not updated correctly</p>';
            }
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Update failed: ' . htmlspecialchars($quotaResult['error']) . '</p>';
        }
        ?>
    </div>

    <!-- TEST 8: Validate Tenant ID -->
    <div class="test-section">
        <h2>🔒 Test 8: Tenant ID Validation</h2>
        <?php
        $validIds = ['school_001', 'soshigh-demo', 'waterford2025'];
        $invalidIds = ['ab', 'school@123', 'test/../hack', str_repeat('a', 51)];
        
        $allValid = true;
        $allInvalid = true;
        
        // Test valid IDs
        foreach ($validIds as $id) {
            $path = $manager->getTenantRootPath($id);
            if ($path === false) {
                $allValid = false;
                break;
            }
        }
        
        // Test invalid IDs
        foreach ($invalidIds as $id) {
            $path = $manager->getTenantRootPath($id);
            if ($path !== false) {
                $allInvalid = false;
                break;
            }
        }
        
        if ($allValid && $allInvalid) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ Validation working correctly</p>';
            echo '<p><strong>Valid IDs accepted:</strong></p>';
            echo '<div class="file-list">';
            foreach ($validIds as $id) {
                echo '<div>✅ ' . htmlspecialchars($id) . '</div>';
            }
            echo '</div>';
            echo '<p><strong>Invalid IDs rejected:</strong></p>';
            echo '<div class="file-list">';
            foreach ($invalidIds as $id) {
                echo '<div>❌ ' . htmlspecialchars($id) . '</div>';
            }
            echo '</div></div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Validation not working correctly</p>';
            if (!$allValid) {
                echo '<p class="error">Some valid IDs were rejected</p>';
            }
            if (!$allInvalid) {
                echo '<p class="error">Some invalid IDs were accepted</p>';
            }
        }
        ?>
    </div>

    <!-- TEST 9: Security Files -->
    <div class="test-section">
        <h2>🛡️ Test 9: Security Files Created</h2>
        <?php
        $tenantRoot = $manager->getTenantRootPath($test_tenant_id);
        
        $htaccessExists = file_exists($tenantRoot . '/.htaccess');
        $indexExists = file_exists($tenantRoot . '/uploads/documents/index.php');
        $uploadHtaccessExists = file_exists($tenantRoot . '/uploads/documents/.htaccess');
        
        if ($htaccessExists && $indexExists && $uploadHtaccessExists) {
            echo '<span class="test-status pass">✅ PASS</span>';
            echo '<div class="result">';
            echo '<p class="success">✅ All security files created</p>';
            echo '<div class="file-list">';
            echo '<div>✅ Root .htaccess</div>';
            echo '<div>✅ Upload .htaccess</div>';
            echo '<div>✅ Directory index.php</div>';
            echo '</div></div>';
        } else {
            echo '<span class="test-status fail">❌ FAIL</span>';
            echo '<p class="error">❌ Some security files missing:</p>';
            echo '<ul>';
            if (!$htaccessExists) echo '<li>Missing: Root .htaccess</li>';
            if (!$uploadHtaccessExists) echo '<li>Missing: Upload .htaccess</li>';
            if (!$indexExists) echo '<li>Missing: Directory index.php</li>';
            echo '</ul>';
        }
        ?>
    </div>

    <!-- TEST 10: Cleanup -->
    <div class="test-section">
        <h2>🧹 Test 10: Cleanup Test Files</h2>
        <?php
        function deleteDirectoryRecursive($dir) {
            if (!file_exists($dir)) return true;
            if (!is_dir($dir)) return unlink($dir);
            
            foreach (scandir($dir) as $item) {
                if ($item == '.' || $item == '..') continue;
                if (!deleteDirectoryRecursive($dir . DIRECTORY_SEPARATOR . $item)) return false;
            }
            return rmdir($dir);
        }
        
        $testPath = $manager->getTenantRootPath($test_tenant_id);
        if ($testPath && file_exists($testPath)) {
            if (deleteDirectoryRecursive($testPath)) {
                echo '<span class="test-status pass">✅ PASS</span>';
                echo '<p class="success">✅ Test files cleaned up successfully</p>';
            } else {
                echo '<span class="test-status skip">⚠️ WARNING</span>';
                echo '<p class="warning">Failed to cleanup. You may need to manually delete:</p>';
                echo '<p><code>' . htmlspecialchars($testPath) . '</code></p>';
            }
        } else {
            echo '<span class="test-status skip">⚠️ SKIP</span>';
            echo '<p>No test directory to clean up</p>';
        }
        ?>
    </div>

    <!-- Summary -->
    <div class="test-section" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
        <h2 style="color: white;">🎉 Test Suite Complete!</h2>
        <p>If all tests passed, your Tenant Directory Manager is ready for production!</p>
        <h3 style="color: white;">Next Steps:</h3>
        <ul>
            <li>Review <code>TENANT_DIRECTORY_MANAGER_GUIDE.md</code> for full documentation</li>
            <li>Check <code>TENANT_DIRECTORY_MANAGER_EXAMPLES.php</code> for usage examples</li>
            <li>Integrate into your school creation process</li>
            <li>Set up storage monitoring scripts</li>
            <li>Configure backup retention policies</li>
        </ul>
    </div>

    <div style="text-align: center; padding: 20px; color: #6b7280;">
        <p>Tenant Directory Manager v1.0</p>
        <p><em>Last run: <?= date('Y-m-d H:i:s') ?></em></p>
    </div>
</body>
</html>

