<?php
/**
 * Test Parent Registration System
 * Checks all components are working
 */

session_start();
require_once 'includes/functions.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Parent Registration</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="card shadow">
            <div class="card-header bg-primary text-white">
                <h3 class="mb-0"><i class="bi bi-clipboard-check"></i> Parent Registration System Test</h3>
            </div>
            <div class="card-body">
                
                <h5 class="mb-3">System Checks:</h5>
                
                <?php
                $checks = [];
                
                // Check 1: Database connection
                try {
                    $pdo = getDB();
                    $checks[] = ['Database Connection', true, 'Connected successfully'];
                } catch (Exception $e) {
                    $checks[] = ['Database Connection', false, $e->getMessage()];
                }
                
                // Check 2: Students table
                try {
                    $stmt = $pdo->query("SELECT COUNT(*) FROM students");
                    $count = $stmt->fetchColumn();
                    $checks[] = ['Students Table', true, "{$count} students found"];
                } catch (Exception $e) {
                    $checks[] = ['Students Table', false, $e->getMessage()];
                }
                
                // Check 3: Parents table
                try {
                    $stmt = $pdo->query("SELECT COUNT(*) FROM parents");
                    $count = $stmt->fetchColumn();
                    $checks[] = ['Parents Table', true, "{$count} parents found"];
                } catch (Exception $e) {
                    $checks[] = ['Parents Table', false, $e->getMessage()];
                }
                
                // Check 4: Academy References table
                try {
                    $stmt = $pdo->query("SELECT COUNT(*) FROM academy_references WHERE is_active = 1");
                    $count = $stmt->fetchColumn();
                    $checks[] = ['Schools Table', true, "{$count} active schools"];
                } catch (Exception $e) {
                    $checks[] = ['Schools Table', false, $e->getMessage()];
                }
                
                // Check 5: check_student.php exists
                $checks[] = ['check_student.php', file_exists('check_student.php'), 
                    file_exists('check_student.php') ? 'File exists' : 'File missing'];
                
                // Check 6: parent_functions.php exists
                $checks[] = ['parent_functions.php', file_exists('includes/parent_functions.php'), 
                    file_exists('includes/parent_functions.php') ? 'File exists' : 'File missing'];
                
                // Check 7: Session working
                $_SESSION['test'] = 'working';
                $checks[] = ['Session Support', isset($_SESSION['test']), 
                    isset($_SESSION['test']) ? 'Sessions working' : 'Sessions not working'];
                
                // Display results
                foreach ($checks as $check):
                    $icon = $check[1] ? 'check-circle-fill text-success' : 'x-circle-fill text-danger';
                    $badge = $check[1] ? 'success' : 'danger';
                ?>
                <div class="alert alert-<?= $badge ?> d-flex align-items-center mb-2">
                    <i class="bi bi-<?= $icon ?> me-3 fs-4"></i>
                    <div class="flex-grow-1">
                        <strong><?= htmlspecialchars($check[0]) ?>:</strong>
                        <?= htmlspecialchars($check[2]) ?>
                    </div>
                </div>
                <?php endforeach; ?>
                
                <hr class="my-4">
                
                <h5 class="mb-3">Test Student Lookup:</h5>
                <div class="row">
                    <div class="col-md-8">
                        <div class="input-group mb-3">
                            <span class="input-group-text"><i class="bi bi-person-badge"></i></span>
                            <input type="text" class="form-control" id="test_username" placeholder="Enter student username">
                            <button class="btn btn-primary" onclick="testLookup()">
                                <i class="bi bi-search"></i> Test Lookup
                            </button>
                        </div>
                    </div>
                </div>
                
                <div id="lookup_result"></div>
                
                <hr class="my-4">
                
                <h5 class="mb-3">Sample Students:</h5>
                <?php
                try {
                    $stmt = $pdo->query("
                        SELECT 
                            s.username, 
                            s.full_name, 
                            s.academy_reference,
                            ar.academy_name,
                            g.name as grade_name
                        FROM students s
                        LEFT JOIN academy_references ar ON s.academy_reference = ar.reference_code
                        LEFT JOIN grades g ON s.grade_id = g.id
                        ORDER BY s.created_at DESC
                        LIMIT 5
                    ");
                    $students = $stmt->fetchAll();
                    
                    if ($students):
                ?>
                <div class="table-responsive">
                    <table class="table table-sm table-striped">
                        <thead>
                            <tr>
                                <th>Username</th>
                                <th>Name</th>
                                <th>School</th>
                                <th>Grade</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($students as $student): ?>
                            <tr>
                                <td><code><?= htmlspecialchars($student['username']) ?></code></td>
                                <td><?= htmlspecialchars($student['full_name']) ?></td>
                                <td><?= htmlspecialchars($student['academy_name'] ?? 'N/A') ?></td>
                                <td><?= htmlspecialchars($student['grade_name'] ?? 'N/A') ?></td>
                                <td>
                                    <button class="btn btn-sm btn-outline-primary" 
                                            onclick="document.getElementById('test_username').value='<?= htmlspecialchars($student['username']) ?>'; testLookup();">
                                        <i class="bi bi-arrow-up"></i> Test
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php 
                    else:
                        echo '<div class="alert alert-warning">No students found. Create students first in admin panel.</div>';
                    endif;
                } catch (Exception $e) {
                    echo '<div class="alert alert-danger">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
                ?>
                
                <hr class="my-4">
                
                <div class="d-flex gap-2">
                    <a href="parent_register.php" class="btn btn-success">
                        <i class="bi bi-arrow-right"></i> Go to Parent Registration
                    </a>
                    <a href="parent/login.php" class="btn btn-primary">
                        <i class="bi bi-box-arrow-in-right"></i> Parent Login
                    </a>
                    <a href="admin/dashboard.php" class="btn btn-secondary">
                        <i class="bi bi-speedometer2"></i> Admin Dashboard
                    </a>
                </div>
                
            </div>
        </div>
    </div>
    
    <script>
        async function testLookup() {
            const username = document.getElementById('test_username').value.trim();
            const resultDiv = document.getElementById('lookup_result');
            
            if (!username) {
                resultDiv.innerHTML = '<div class="alert alert-warning">Please enter a username</div>';
                return;
            }
            
            resultDiv.innerHTML = '<div class="alert alert-info"><i class="bi bi-hourglass-split"></i> Looking up student...</div>';
            
            try {
                const response = await fetch('check_student.php?username=' + encodeURIComponent(username));
                const data = await response.json();
                
                if (data.success) {
                    resultDiv.innerHTML = `
                        <div class="alert alert-success">
                            <h6><i class="bi bi-check-circle-fill"></i> Student Found!</h6>
                            <table class="table table-sm mb-0 mt-2">
                                <tr><th>Username:</th><td><code>${data.student.username}</code></td></tr>
                                <tr><th>Name:</th><td>${data.student.full_name}</td></tr>
                                <tr><th>School:</th><td>${data.student.school_name} (${data.student.academy_reference})</td></tr>
                                <tr><th>Grade:</th><td>${data.student.grade_name}</td></tr>
                            </table>
                        </div>
                    `;
                } else {
                    resultDiv.innerHTML = `
                        <div class="alert alert-danger">
                            <i class="bi bi-x-circle-fill"></i> <strong>Not Found:</strong> ${data.message}
                        </div>
                    `;
                }
            } catch (error) {
                resultDiv.innerHTML = `
                    <div class="alert alert-danger">
                        <i class="bi bi-exclamation-triangle-fill"></i> <strong>Error:</strong> ${error.message}
                    </div>
                `;
            }
        }
    </script>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

