<?php
/**
 * Moodle Integration Testing Tool
 * Comprehensive diagnostic for all Moodle functions used in the system
 */

session_start();
require_once 'config.php';
require_once 'includes/functions.php';
require_once 'includes/moodle_suspension_handler.php';

// Require admin login for security
if (!isset($_SESSION['admin_id']) && !isset($_SESSION['user_id'])) {
    // Allow access with a secret key for testing
    $secret_key = $_GET['key'] ?? '';
    if ($secret_key !== 'test123') {
        die('Access denied. Login as admin or use ?key=test123');
    }
}

$pdo = getDB();
$test_results = [];
$overall_status = 'success';

// Get a test student
$stmt = $pdo->query("
    SELECT s.*, g.name as grade_name 
    FROM students s 
    LEFT JOIN grades g ON s.grade_id = g.id 
    WHERE s.moodle_user_id IS NOT NULL 
    LIMIT 1
");
$test_student = $stmt->fetch();

// Get Moodle configuration
$moodle_url = MOODLE_API_URL;
$moodle_token = MOODLE_API_TOKEN;

/**
 * Test Moodle API Call
 */
function testMoodleAPI($function, $params = []) {
    global $moodle_url, $moodle_token;
    
    $url = $moodle_url . '?wstoken=' . $moodle_token . 
           '&wsfunction=' . $function . '&moodlewsrestformat=json';
    
    $start_time = microtime(true);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    $duration = round((microtime(true) - $start_time) * 1000, 2);
    
    if ($curl_error) {
        return [
            'success' => false,
            'error' => 'CURL Error: ' . $curl_error,
            'http_code' => $http_code,
            'duration_ms' => $duration
        ];
    }
    
    if ($http_code !== 200) {
        return [
            'success' => false,
            'error' => 'HTTP Error: ' . $http_code,
            'http_code' => $http_code,
            'duration_ms' => $duration
        ];
    }
    
    $result = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            'success' => false,
            'error' => 'JSON Error: ' . json_last_error_msg(),
            'http_code' => $http_code,
            'duration_ms' => $duration,
            'raw_response' => substr($response, 0, 500)
        ];
    }
    
    if (isset($result['exception'])) {
        return [
            'success' => false,
            'error' => 'Moodle Exception: ' . ($result['message'] ?? 'Unknown error'),
            'exception' => $result['exception'],
            'http_code' => $http_code,
            'duration_ms' => $duration
        ];
    }
    
    return [
        'success' => true,
        'data' => $result,
        'http_code' => $http_code,
        'duration_ms' => $duration
    ];
}

// =====================================================
// TEST 1: Connection & Authentication
// =====================================================
$test_results[] = [
    'name' => 'Connection & Authentication',
    'function' => 'core_webservice_get_site_info',
    'description' => 'Tests basic connection and token validity',
    'result' => testMoodleAPI('core_webservice_get_site_info')
];

// =====================================================
// TEST 2: Get Site Information
// =====================================================
$test_results[] = [
    'name' => 'Get Site Info',
    'function' => 'core_webservice_get_site_info',
    'description' => 'Retrieves Moodle site information',
    'result' => testMoodleAPI('core_webservice_get_site_info')
];

// =====================================================
// TEST 3: List All Courses
// =====================================================
$test_results[] = [
    'name' => 'Get All Courses',
    'function' => 'core_course_get_courses',
    'description' => 'Lists all available courses',
    'result' => testMoodleAPI('core_course_get_courses')
];

// =====================================================
// TEST 4: Get Users by Field (if test student exists)
// =====================================================
if ($test_student && $test_student['moodle_user_id']) {
    $test_results[] = [
        'name' => 'Get User by ID',
        'function' => 'core_user_get_users_by_field',
        'description' => "Get user info for student: {$test_student['full_name']}",
        'result' => testMoodleAPI('core_user_get_users_by_field', [
            'field' => 'id',
            'values' => [(int)$test_student['moodle_user_id']]
        ])
    ];
}

// =====================================================
// TEST 5: Create User (Dry Run - Create Test User)
// =====================================================
$test_username = 'testuser_' . time();
$test_results[] = [
    'name' => 'Create User',
    'function' => 'core_user_create_users',
    'description' => 'Tests user creation capability',
    'result' => testMoodleAPI('core_user_create_users', [
        'users' => [[
            'username' => $test_username,
            'password' => 'TestPass123!',
            'firstname' => 'Test',
            'lastname' => 'User',
            'email' => 'test_' . time() . '@example.com'
        ]]
    ])
];

// =====================================================
// TEST 6: Update User (Suspend/Unsuspend)
// =====================================================
if ($test_student && $test_student['moodle_user_id']) {
    $test_results[] = [
        'name' => 'Update User (Suspend)',
        'function' => 'core_user_update_users',
        'description' => 'Tests suspension capability (sets suspended=1)',
        'result' => testMoodleAPI('core_user_update_users', [
            'users' => [[
                'id' => (int)$test_student['moodle_user_id'],
                'suspended' => 1
            ]]
        ])
    ];
    
    // Unsuspend immediately
    $test_results[] = [
        'name' => 'Update User (Unsuspend)',
        'function' => 'core_user_update_users',
        'description' => 'Tests unsuspension capability (sets suspended=0)',
        'result' => testMoodleAPI('core_user_update_users', [
            'users' => [[
                'id' => (int)$test_student['moodle_user_id'],
                'suspended' => 0
            ]]
        ])
    ];
}

// =====================================================
// TEST 7: Get Course Categories
// =====================================================
$test_results[] = [
    'name' => 'Get Course Categories',
    'function' => 'core_course_get_categories',
    'description' => 'Lists all course categories',
    'result' => testMoodleAPI('core_course_get_categories')
];

// =====================================================
// TEST 8: Enroll User in Course
// =====================================================
if ($test_student && $test_student['moodle_user_id']) {
    // Get a course first
    $courses_result = testMoodleAPI('core_course_get_courses');
    if ($courses_result['success'] && !empty($courses_result['data'])) {
        $first_course = $courses_result['data'][0];
        
        $test_results[] = [
            'name' => 'Enroll User in Course',
            'function' => 'enrol_manual_enrol_users',
            'description' => "Enroll student in course: {$first_course['fullname']}",
            'result' => testMoodleAPI('enrol_manual_enrol_users', [
                'enrolments' => [[
                    'roleid' => 5, // Student role
                    'userid' => (int)$test_student['moodle_user_id'],
                    'courseid' => (int)$first_course['id']
                ]]
            ])
        ];
    }
}

// =====================================================
// TEST 9: Get Enrolled Users in Course
// =====================================================
$courses_result = testMoodleAPI('core_course_get_courses');
if ($courses_result['success'] && !empty($courses_result['data'])) {
    $first_course = $courses_result['data'][0];
    
    $test_results[] = [
        'name' => 'Get Enrolled Users',
        'function' => 'core_enrol_get_enrolled_users',
        'description' => "List users enrolled in: {$first_course['fullname']}",
        'result' => testMoodleAPI('core_enrol_get_enrolled_users', [
            'courseid' => (int)$first_course['id']
        ])
    ];
}

// =====================================================
// TEST 10: Delete Created Test User
// =====================================================
if (isset($test_results[4]) && $test_results[4]['result']['success']) {
    $created_user = $test_results[4]['result']['data'][0] ?? null;
    if ($created_user && isset($created_user['id'])) {
        $test_results[] = [
            'name' => 'Delete Test User',
            'function' => 'core_user_delete_users',
            'description' => 'Cleanup: Delete the test user created',
            'result' => testMoodleAPI('core_user_delete_users', [
                'userids' => [(int)$created_user['id']]
            ])
        ];
    }
}

// Calculate overall status
foreach ($test_results as $test) {
    if (!$test['result']['success']) {
        $overall_status = 'warning';
        break;
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Moodle Integration Test - <?= APP_NAME ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px 0;
        }
        .test-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            padding: 30px;
            margin: 20px auto;
            max-width: 1200px;
        }
        .test-header {
            text-align: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 3px solid #667eea;
        }
        .config-box {
            background: #f8f9fa;
            border-left: 4px solid #667eea;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
        }
        .test-item {
            background: #ffffff;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 15px;
            transition: all 0.3s;
        }
        .test-item:hover {
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .test-success {
            border-left: 5px solid #28a745;
            background: #f1f9f3;
        }
        .test-failed {
            border-left: 5px solid #dc3545;
            background: #fdf4f4;
        }
        .badge-success { background: #28a745; }
        .badge-danger { background: #dc3545; }
        .badge-warning { background: #ffc107; color: #000; }
        .code-box {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 10px;
            font-family: monospace;
            font-size: 12px;
            max-height: 200px;
            overflow-y: auto;
        }
        .duration {
            color: #666;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="test-container">
            
            <!-- Header -->
            <div class="test-header">
                <h1><i class="bi bi-mortarboard-fill" style="color: #667eea;"></i> Moodle Integration Test</h1>
                <p class="text-muted mb-0">Comprehensive diagnostic of all Moodle functions</p>
            </div>
            
            <!-- Overall Status -->
            <div class="alert alert-<?= $overall_status === 'success' ? 'success' : 'warning' ?> text-center">
                <?php if ($overall_status === 'success'): ?>
                    <h4><i class="bi bi-check-circle-fill"></i> All Tests Passed!</h4>
                    <p class="mb-0">Your Moodle integration is working perfectly.</p>
                <?php else: ?>
                    <h4><i class="bi bi-exclamation-triangle-fill"></i> Some Tests Failed</h4>
                    <p class="mb-0">Please review the results below.</p>
                <?php endif; ?>
            </div>
            
            <!-- Configuration Info -->
            <div class="config-box">
                <h5><i class="bi bi-gear"></i> Configuration</h5>
                <div class="row">
                    <div class="col-md-6">
                        <strong>Moodle URL:</strong><br>
                        <code><?= htmlspecialchars($moodle_url) ?></code>
                    </div>
                    <div class="col-md-6">
                        <strong>API Token:</strong><br>
                        <code><?= htmlspecialchars(substr($moodle_token, 0, 20)) ?>...</code>
                    </div>
                </div>
                <?php if ($test_student): ?>
                <div class="mt-2">
                    <strong>Test Student:</strong> <?= htmlspecialchars($test_student['full_name']) ?> 
                    (Moodle ID: <?= $test_student['moodle_user_id'] ?>)
                </div>
                <?php else: ?>
                <div class="mt-2 text-warning">
                    <i class="bi bi-exclamation-triangle"></i> No students synced to Moodle yet. Some tests will be skipped.
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Test Results -->
            <h4 class="mb-3"><i class="bi bi-list-check"></i> Test Results</h4>
            
            <?php foreach ($test_results as $index => $test): ?>
            <div class="test-item <?= $test['result']['success'] ? 'test-success' : 'test-failed' ?>">
                <div class="d-flex justify-content-between align-items-start mb-2">
                    <div>
                        <h5 class="mb-1">
                            <?php if ($test['result']['success']): ?>
                                <i class="bi bi-check-circle-fill text-success"></i>
                            <?php else: ?>
                                <i class="bi bi-x-circle-fill text-danger"></i>
                            <?php endif; ?>
                            Test <?= $index + 1 ?>: <?= htmlspecialchars($test['name']) ?>
                        </h5>
                        <p class="text-muted mb-1"><?= htmlspecialchars($test['description']) ?></p>
                        <small class="text-muted">
                            <strong>Function:</strong> <code><?= htmlspecialchars($test['function']) ?></code>
                        </small>
                    </div>
                    <div class="text-end">
                        <span class="badge badge-<?= $test['result']['success'] ? 'success' : 'danger' ?>">
                            <?= $test['result']['success'] ? 'PASS' : 'FAIL' ?>
                        </span>
                        <br>
                        <span class="duration">
                            <i class="bi bi-clock"></i> <?= $test['result']['duration_ms'] ?>ms
                        </span>
                    </div>
                </div>
                
                <?php if ($test['result']['success']): ?>
                    <!-- Success Details -->
                    <div class="mt-3">
                        <button class="btn btn-sm btn-outline-primary" type="button" 
                                data-bs-toggle="collapse" 
                                data-bs-target="#response-<?= $index ?>">
                            <i class="bi bi-code"></i> View Response
                        </button>
                        <div class="collapse mt-2" id="response-<?= $index ?>">
                            <div class="code-box">
                                <?= htmlspecialchars(json_encode($test['result']['data'], JSON_PRETTY_PRINT)) ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Extract useful info -->
                    <?php if ($test['function'] === 'core_webservice_get_site_info' && isset($test['result']['data']['sitename'])): ?>
                    <div class="alert alert-info mt-2 mb-0">
                        <strong>Site Name:</strong> <?= htmlspecialchars($test['result']['data']['sitename']) ?><br>
                        <strong>Version:</strong> <?= htmlspecialchars($test['result']['data']['release'] ?? 'Unknown') ?><br>
                        <strong>Users:</strong> <?= number_format($test['result']['data']['usercount'] ?? 0) ?>
                    </div>
                    <?php elseif ($test['function'] === 'core_course_get_courses' && is_array($test['result']['data'])): ?>
                    <div class="alert alert-info mt-2 mb-0">
                        <strong>Total Courses:</strong> <?= count($test['result']['data']) ?> courses found
                    </div>
                    <?php elseif ($test['function'] === 'core_user_create_users' && is_array($test['result']['data'])): ?>
                    <div class="alert alert-success mt-2 mb-0">
                        <i class="bi bi-check-circle"></i> User created successfully! 
                        ID: <?= $test['result']['data'][0]['id'] ?? 'Unknown' ?>
                    </div>
                    <?php endif; ?>
                    
                <?php else: ?>
                    <!-- Error Details -->
                    <div class="alert alert-danger mt-2 mb-0">
                        <strong>Error:</strong> <?= htmlspecialchars($test['result']['error']) ?>
                        <?php if (isset($test['result']['exception'])): ?>
                        <br><strong>Exception:</strong> <?= htmlspecialchars($test['result']['exception']) ?>
                        <?php endif; ?>
                        <?php if (isset($test['result']['raw_response'])): ?>
                        <br>
                        <button class="btn btn-sm btn-outline-danger mt-2" type="button" 
                                data-bs-toggle="collapse" 
                                data-bs-target="#error-<?= $index ?>">
                            <i class="bi bi-bug"></i> View Raw Response
                        </button>
                        <div class="collapse mt-2" id="error-<?= $index ?>">
                            <div class="code-box">
                                <?= htmlspecialchars($test['result']['raw_response']) ?>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            <?php endforeach; ?>
            
            <!-- Summary -->
            <div class="card mt-4">
                <div class="card-body">
                    <h5><i class="bi bi-clipboard-check"></i> Test Summary</h5>
                    <?php
                    $passed = 0;
                    $failed = 0;
                    foreach ($test_results as $test) {
                        if ($test['result']['success']) {
                            $passed++;
                        } else {
                            $failed++;
                        }
                    }
                    ?>
                    <div class="row text-center mt-3">
                        <div class="col-md-4">
                            <h2 class="text-primary"><?= count($test_results) ?></h2>
                            <p class="text-muted">Total Tests</p>
                        </div>
                        <div class="col-md-4">
                            <h2 class="text-success"><?= $passed ?></h2>
                            <p class="text-muted">Passed</p>
                        </div>
                        <div class="col-md-4">
                            <h2 class="text-danger"><?= $failed ?></h2>
                            <p class="text-muted">Failed</p>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Functions Available -->
            <div class="card mt-4">
                <div class="card-body">
                    <h5><i class="bi bi-tools"></i> Functions Tested</h5>
                    <p class="text-muted">These are the Moodle functions your system uses:</p>
                    <ul>
                        <li><code>core_webservice_get_site_info</code> - Get site information</li>
                        <li><code>core_course_get_courses</code> - List courses</li>
                        <li><code>core_user_get_users_by_field</code> - Get user details</li>
                        <li><code>core_user_create_users</code> - Create new users</li>
                        <li><code>core_user_update_users</code> - Update users (suspend/unsuspend)</li>
                        <li><code>enrol_manual_enrol_users</code> - Enroll users in courses</li>
                        <li><code>core_enrol_get_enrolled_users</code> - Get enrolled users</li>
                        <li><code>core_user_delete_users</code> - Delete users</li>
                    </ul>
                </div>
            </div>
            
            <!-- Actions -->
            <div class="text-center mt-4">
                <a href="?" class="btn btn-primary">
                    <i class="bi bi-arrow-clockwise"></i> Run Tests Again
                </a>
                <?php if (isset($_SESSION['admin_id'])): ?>
                <a href="admin/dashboard.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back to Dashboard
                </a>
                <?php endif; ?>
            </div>
            
            <!-- Warning Note -->
            <div class="alert alert-warning mt-4">
                <i class="bi bi-shield-exclamation"></i> <strong>Security Note:</strong> 
                This test page should be deleted or protected after testing. 
                It contains sensitive information about your Moodle configuration.
                <hr>
                <small>To delete: Remove <code>test_moodle_functions.php</code> from your server.</small>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

