<?php
/**
 * Test Clean URLs
 * Check if .htaccess rewriting is working
 */

require_once 'includes/functions.php';
$pdo = getDB();

// Get a sample school
$stmt = $pdo->query("SELECT reference_code, academy_name FROM academy_references WHERE is_active = 1 LIMIT 1");
$school = $stmt->fetch();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Clean URLs</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="card shadow">
            <div class="card-header bg-primary text-white">
                <h3 class="mb-0"><i class="bi bi-speedometer me-2"></i>Clean URLs Test</h3>
            </div>
            <div class="card-body">
                
                <h5 class="mb-3">Server Configuration Check:</h5>
                
                <!-- mod_rewrite Check -->
                <div class="mb-3">
                    <strong>1. Apache mod_rewrite Module:</strong><br>
                    <?php if (function_exists('apache_get_modules')): ?>
                        <?php $modules = apache_get_modules(); ?>
                        <?php if (in_array('mod_rewrite', $modules)): ?>
                            <span class="badge bg-success"><i class="bi bi-check-circle"></i> Enabled</span>
                        <?php else: ?>
                            <span class="badge bg-danger"><i class="bi bi-x-circle"></i> Not Enabled</span>
                            <p class="text-danger small mt-1">mod_rewrite is required for clean URLs</p>
                        <?php endif; ?>
                    <?php else: ?>
                        <span class="badge bg-warning">Cannot detect (apache_get_modules not available)</span>
                        <p class="small mt-1">This is common on shared hosting. Test URLs below to verify.</p>
                    <?php endif; ?>
                </div>
                
                <!-- .htaccess Check -->
                <div class="mb-3">
                    <strong>2. .htaccess File:</strong><br>
                    <?php if (file_exists('.htaccess')): ?>
                        <span class="badge bg-success"><i class="bi bi-check-circle"></i> Exists</span>
                        <p class="small mt-1">File size: <?= filesize('.htaccess') ?> bytes</p>
                    <?php else: ?>
                        <span class="badge bg-danger"><i class="bi bi-x-circle"></i> Not Found</span>
                        <p class="text-danger small mt-1">Upload .htaccess to enable clean URLs</p>
                    <?php endif; ?>
                </div>
                
                <!-- URL Generator Check -->
                <div class="mb-3">
                    <strong>3. URL Generator Class:</strong><br>
                    <?php if (file_exists('includes/school_url_generator.php')): ?>
                        <span class="badge bg-success"><i class="bi bi-check-circle"></i> Exists</span>
                        <?php 
                        require_once 'includes/school_url_generator.php';
                        if (class_exists('SchoolURLGenerator')) {
                            echo '<p class="small text-success mt-1">Class loaded successfully</p>';
                        }
                        ?>
                    <?php else: ?>
                        <span class="badge bg-danger"><i class="bi bi-x-circle"></i> Not Found</span>
                        <p class="text-danger small mt-1">Upload school_url_generator.php to includes/</p>
                    <?php endif; ?>
                </div>
                
                <hr class="my-4">
                
                <h5 class="mb-3">Test Clean URLs:</h5>
                
                <?php if ($school): ?>
                    <?php 
                    $school_code_lower = strtolower($school['reference_code']);
                    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . 
                                '://' . $_SERVER['HTTP_HOST'] . '/Multi-Tanent';
                    
                    $test_urls = [
                        'Student Registration' => [
                            'clean' => $base_url . '/' . $school_code_lower . '/register',
                            'legacy' => $base_url . '/register_student.php?school=' . $school['reference_code'],
                            'icon' => 'person-plus',
                            'color' => 'primary'
                        ],
                        'Parent Registration' => [
                            'clean' => $base_url . '/' . $school_code_lower . '/parent/register',
                            'legacy' => $base_url . '/parent_register.php?school=' . $school['reference_code'],
                            'icon' => 'people',
                            'color' => 'success'
                        ],
                        'Student Login' => [
                            'clean' => $base_url . '/' . $school_code_lower . '/login',
                            'legacy' => $base_url . '/admin/login.php?school=' . $school['reference_code'],
                            'icon' => 'box-arrow-in-right',
                            'color' => 'info'
                        ],
                        'Parent Login' => [
                            'clean' => $base_url . '/' . $school_code_lower . '/parent/login',
                            'legacy' => $base_url . '/parent/login.php?school=' . $school['reference_code'],
                            'icon' => 'shield-check',
                            'color' => 'warning'
                        ]
                    ];
                    ?>
                    
                    <div class="alert alert-info">
                        <strong>Testing with school:</strong> <?= htmlspecialchars($school['academy_name']) ?> 
                        (<?= htmlspecialchars($school['reference_code']) ?>)
                    </div>
                    
                    <?php foreach ($test_urls as $name => $urls): ?>
                    <div class="card mb-3">
                        <div class="card-header bg-<?= $urls['color'] ?> text-white">
                            <h6 class="mb-0">
                                <i class="bi bi-<?= $urls['icon'] ?> me-2"></i>
                                <?= $name ?>
                            </h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <strong class="d-block mb-2">Clean URL:</strong>
                                    <div class="input-group input-group-sm mb-2">
                                        <input type="text" class="form-control" 
                                               value="<?= htmlspecialchars($urls['clean']) ?>" readonly>
                                        <button class="btn btn-outline-secondary" 
                                                onclick="navigator.clipboard.writeText('<?= htmlspecialchars($urls['clean']) ?>')">
                                            <i class="bi bi-clipboard"></i>
                                        </button>
                                    </div>
                                    <a href="<?= htmlspecialchars($urls['clean']) ?>" 
                                       target="_blank" class="btn btn-sm btn-primary">
                                        <i class="bi bi-box-arrow-up-right me-1"></i>Test Clean URL
                                    </a>
                                    <span id="status_clean_<?= $name ?>" class="ms-2"></span>
                                </div>
                                <div class="col-md-6">
                                    <strong class="d-block mb-2">Legacy URL (Fallback):</strong>
                                    <div class="input-group input-group-sm mb-2">
                                        <input type="text" class="form-control" 
                                               value="<?= htmlspecialchars($urls['legacy']) ?>" readonly>
                                        <button class="btn btn-outline-secondary" 
                                                onclick="navigator.clipboard.writeText('<?= htmlspecialchars($urls['legacy']) ?>')">
                                            <i class="bi bi-clipboard"></i>
                                        </button>
                                    </div>
                                    <a href="<?= htmlspecialchars($urls['legacy']) ?>" 
                                       target="_blank" class="btn btn-sm btn-secondary">
                                        <i class="bi bi-box-arrow-up-right me-1"></i>Test Legacy URL
                                    </a>
                                    <span id="status_legacy_<?= $name ?>" class="ms-2"></span>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    
                <?php else: ?>
                    <div class="alert alert-warning">
                        No active schools found. Create a school first.
                    </div>
                <?php endif; ?>
                
                <hr class="my-4">
                
                <h5 class="mb-3">Troubleshooting:</h5>
                
                <div class="accordion" id="troubleshootingAccordion">
                    <!-- Clean URLs not working -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#issue1">
                                Clean URLs return 404 error
                            </button>
                        </h2>
                        <div id="issue1" class="accordion-collapse collapse show" data-bs-parent="#troubleshootingAccordion">
                            <div class="accordion-body">
                                <strong>Solution 1: Enable mod_rewrite</strong>
                                <ol>
                                    <li>Login to cPanel</li>
                                    <li>Go to: Software → Select PHP Version</li>
                                    <li>Click: Extensions</li>
                                    <li>Enable: mod_rewrite</li>
                                    <li>Save</li>
                                </ol>
                                
                                <strong>Solution 2: Use legacy URLs instead</strong>
                                <p>If mod_rewrite cannot be enabled, use the "Legacy URL" format shown above. 
                                   They work exactly the same, just aren't as clean.</p>
                                
                                <strong>Solution 3: Contact hosting support</strong>
                                <p>Ask them to enable mod_rewrite for your account.</p>
                            </div>
                        </div>
                    </div>
                    
                    <!-- .htaccess causes errors -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#issue2">
                                Site shows 500 error after uploading .htaccess
                            </button>
                        </h2>
                        <div id="issue2" class="accordion-collapse collapse" data-bs-parent="#troubleshootingAccordion">
                            <div class="accordion-body">
                                <strong>Solution: Delete .htaccess</strong>
                                <ol>
                                    <li>Go to cPanel File Manager</li>
                                    <li>Navigate to Multi-Tanent/</li>
                                    <li>Delete .htaccess file</li>
                                    <li>Use legacy URLs instead</li>
                                </ol>
                                <p class="text-info">
                                    <i class="bi bi-info-circle me-1"></i>
                                    The system works perfectly without .htaccess, you just won't have clean URLs.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
                
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

