<?php
/**
 * .htaccess Template Generator
 * 
 * Generates secure .htaccess configurations for different tenant directory types
 * 
 * @package Multi-Tenant System
 * @version 1.0
 */

class HtaccessGenerator {
    
    /**
     * Generate .htaccess for upload directories (documents, photos, receipts, reports)
     * 
     * @param array $allowed_types Array of allowed file extensions (e.g., ['pdf', 'jpg', 'png'])
     * @param string $directory_name Name of directory (for comments)
     * @return string .htaccess content
     */
    public static function forUploads($allowed_types = ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'], $directory_name = 'uploads') {
        $extensions = implode('|', array_map('preg_quote', $allowed_types));
        $extensionList = implode(', ', $allowed_types);
        
        $htaccess = "# .htaccess for {$directory_name}\n";
        $htaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n";
        $htaccess .= "# Security configuration for tenant upload directory\n";
        $htaccess .= str_repeat("#", 60) . "\n\n";
        
        // Prevent directory listing
        $htaccess .= "# Prevent directory listing\n";
        $htaccess .= "Options -Indexes\n";
        $htaccess .= "DirectoryIndex disabled\n\n";
        
        // Disable PHP execution
        $htaccess .= "# Disable PHP execution in upload directories\n";
        $htaccess .= "# Prevents uploaded PHP files from being executed\n";
        $htaccess .= "php_flag engine off\n\n";
        
        // Alternative PHP execution block (for servers without php_flag)
        $htaccess .= "# Block PHP files (alternative method)\n";
        $htaccess .= "<FilesMatch \"\\.php$\">\n";
        $htaccess .= "    Order Deny,Allow\n";
        $htaccess .= "    Deny from all\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Allow only specific file types
        $htaccess .= "# Allow only safe file types\n";
        $htaccess .= "# Allowed: {$extensionList}\n";
        $htaccess .= "<FilesMatch \"\\.({$extensions})$\">\n";
        $htaccess .= "    Order Allow,Deny\n";
        $htaccess .= "    Allow from all\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Force download for certain file types
        $htaccess .= "# Force download for documents\n";
        $htaccess .= "<FilesMatch \"\\.(pdf|doc|docx|xls|xlsx)$\">\n";
        $htaccess .= "    Header set Content-Disposition \"attachment\"\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Security headers
        $htaccess .= "# Security headers\n";
        $htaccess .= "<IfModule mod_headers.c>\n";
        $htaccess .= "    Header set X-Content-Type-Options \"nosniff\"\n";
        $htaccess .= "    Header set X-Frame-Options \"SAMEORIGIN\"\n";
        $htaccess .= "    Header set X-XSS-Protection \"1; mode=block\"\n";
        $htaccess .= "</IfModule>\n\n";
        
        // Cache control for images
        $htaccess .= "# Cache control for images\n";
        $htaccess .= "<FilesMatch \"\\.(jpg|jpeg|png|gif|webp)$\">\n";
        $htaccess .= "    Header set Cache-Control \"max-age=2592000, public\"\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Deny access to hidden files
        $htaccess .= "# Deny access to hidden files\n";
        $htaccess .= "<FilesMatch \"^\\..+$\">\n";
        $htaccess .= "    Order Deny,Allow\n";
        $htaccess .= "    Deny from all\n";
        $htaccess .= "</FilesMatch>\n";
        
        return $htaccess;
    }
    
    /**
     * Generate .htaccess for backup directories
     * 
     * @return string .htaccess content
     */
    public static function forBackups() {
        $htaccess = "# .htaccess for backups\n";
        $htaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n";
        $htaccess .= "# Security configuration for tenant backup directory\n";
        $htaccess .= str_repeat("#", 60) . "\n\n";
        
        // Deny all direct access
        $htaccess .= "# Deny all direct access to backups\n";
        $htaccess .= "# Backups should only be accessed via authenticated PHP scripts\n";
        $htaccess .= "Order Deny,Allow\n";
        $htaccess .= "Deny from all\n\n";
        
        // Prevent directory listing
        $htaccess .= "# Prevent directory listing\n";
        $htaccess .= "Options -Indexes\n\n";
        
        // Disable PHP execution
        $htaccess .= "# Disable PHP execution\n";
        $htaccess .= "php_flag engine off\n\n";
        
        // Security headers
        $htaccess .= "# Security headers\n";
        $htaccess .= "<IfModule mod_headers.c>\n";
        $htaccess .= "    Header set X-Robots-Tag \"noindex, nofollow\"\n";
        $htaccess .= "</IfModule>\n\n";
        
        // Additional protection
        $htaccess .= "# Additional protection\n";
        $htaccess .= "<Files *>\n";
        $htaccess .= "    Order Deny,Allow\n";
        $htaccess .= "    Deny from all\n";
        $htaccess .= "</Files>\n";
        
        return $htaccess;
    }
    
    /**
     * Generate .htaccess for log directories
     * 
     * @return string .htaccess content
     */
    public static function forLogs() {
        $htaccess = "# .htaccess for logs\n";
        $htaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n";
        $htaccess .= "# Security configuration for tenant log directory\n";
        $htaccess .= str_repeat("#", 60) . "\n\n";
        
        // Deny all direct access
        $htaccess .= "# Deny all direct access to log files\n";
        $htaccess .= "# Logs should only be accessed via authenticated admin panel\n";
        $htaccess .= "Order Deny,Allow\n";
        $htaccess .= "Deny from all\n\n";
        
        // Prevent directory listing
        $htaccess .= "# Prevent directory listing\n";
        $htaccess .= "Options -Indexes\n";
        $htaccess .= "DirectoryIndex disabled\n\n";
        
        // Disable PHP execution
        $htaccess .= "# Disable PHP execution\n";
        $htaccess .= "php_flag engine off\n\n";
        
        // Block access to all log files
        $htaccess .= "# Block access to all log files\n";
        $htaccess .= "<FilesMatch \"\\.(log|txt)$\">\n";
        $htaccess .= "    Order Deny,Allow\n";
        $htaccess .= "    Deny from all\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Security headers
        $htaccess .= "# Security headers\n";
        $htaccess .= "<IfModule mod_headers.c>\n";
        $htaccess .= "    Header set X-Robots-Tag \"noindex, nofollow, noarchive\"\n";
        $htaccess .= "</IfModule>\n";
        
        return $htaccess;
    }
    
    /**
     * Generate .htaccess for temp directories
     * 
     * @return string .htaccess content
     */
    public static function forTemp() {
        $htaccess = "# .htaccess for temp\n";
        $htaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n";
        $htaccess .= "# Security configuration for tenant temporary directory\n";
        $htaccess .= str_repeat("#", 60) . "\n\n";
        
        // Deny all direct access
        $htaccess .= "# Deny all direct access to temporary files\n";
        $htaccess .= "# Temp files are for processing only, not public access\n";
        $htaccess .= "Order Deny,Allow\n";
        $htaccess .= "Deny from all\n\n";
        
        // Prevent directory listing
        $htaccess .= "# Prevent directory listing\n";
        $htaccess .= "Options -Indexes\n";
        $htaccess .= "DirectoryIndex disabled\n\n";
        
        // Disable PHP execution
        $htaccess .= "# Disable PHP execution\n";
        $htaccess .= "php_flag engine off\n\n";
        
        // Block all files
        $htaccess .= "# Block all direct file access\n";
        $htaccess .= "<Files *>\n";
        $htaccess .= "    Order Deny,Allow\n";
        $htaccess .= "    Deny from all\n";
        $htaccess .= "</Files>\n\n";
        
        // Auto-cleanup recommendation (comment only)
        $htaccess .= "# NOTE: Temp files should be automatically cleaned up\n";
        $htaccess .= "# Set up a cron job to delete files older than 7 days\n";
        $htaccess .= "# Example: find /path/to/temp -type f -mtime +7 -delete\n";
        
        return $htaccess;
    }
    
    /**
     * Generate .htaccess for root tenant directory
     * 
     * @return string .htaccess content
     */
    public static function forRoot() {
        $htaccess = "# .htaccess for tenant root\n";
        $htaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n";
        $htaccess .= "# Security configuration for tenant root directory\n";
        $htaccess .= str_repeat("#", 60) . "\n\n";
        
        // Deny access by default
        $htaccess .= "# Deny direct access by default\n";
        $htaccess .= "Order Deny,Allow\n";
        $htaccess .= "Deny from all\n\n";
        
        // Allow only images and PDFs through application
        $htaccess .= "# Allow safe file types to be served\n";
        $htaccess .= "<FilesMatch \"\\.(jpg|jpeg|png|gif|webp|pdf)$\">\n";
        $htaccess .= "    Allow from all\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Prevent access to config files
        $htaccess .= "# Protect configuration files\n";
        $htaccess .= "<FilesMatch \"\\.(json|txt|log)$\">\n";
        $htaccess .= "    Order Deny,Allow\n";
        $htaccess .= "    Deny from all\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Disable directory listing
        $htaccess .= "# Disable directory listing\n";
        $htaccess .= "Options -Indexes\n";
        
        return $htaccess;
    }
    
    /**
     * Generate .htaccess for profile photos (more permissive)
     * 
     * @return string .htaccess content
     */
    public static function forProfilePhotos() {
        $htaccess = "# .htaccess for profile photos\n";
        $htaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n";
        $htaccess .= str_repeat("#", 60) . "\n\n";
        
        // Allow images
        $htaccess .= "# Allow image files\n";
        $htaccess .= "Options -Indexes\n";
        $htaccess .= "php_flag engine off\n\n";
        
        $htaccess .= "<FilesMatch \"\\.(jpg|jpeg|png|gif|webp)$\">\n";
        $htaccess .= "    Order Allow,Deny\n";
        $htaccess .= "    Allow from all\n";
        $htaccess .= "</FilesMatch>\n\n";
        
        // Cache headers for better performance
        $htaccess .= "# Cache images for 30 days\n";
        $htaccess .= "<IfModule mod_headers.c>\n";
        $htaccess .= "    Header set Cache-Control \"max-age=2592000, public\"\n";
        $htaccess .= "    Header set X-Content-Type-Options \"nosniff\"\n";
        $htaccess .= "</IfModule>\n\n";
        
        // Deny everything else
        $htaccess .= "# Deny non-image files\n";
        $htaccess .= "<FilesMatch \"\\.(php|html|js|css|sh|sql)$\">\n";
        $htaccess .= "    Order Deny,Allow\n";
        $htaccess .= "    Deny from all\n";
        $htaccess .= "</FilesMatch>\n";
        
        return $htaccess;
    }
    
    /**
     * Generate .htaccess for documents
     * 
     * @return string .htaccess content
     */
    public static function forDocuments() {
        return self::forUploads(
            ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv'],
            'documents'
        );
    }
    
    /**
     * Generate .htaccess for payment receipts
     * 
     * @return string .htaccess content
     */
    public static function forPaymentReceipts() {
        return self::forUploads(
            ['pdf', 'jpg', 'jpeg', 'png'],
            'payment_receipts'
        );
    }
    
    /**
     * Generate .htaccess for reports
     * 
     * @return string .htaccess content
     */
    public static function forReports() {
        return self::forUploads(
            ['pdf', 'xlsx', 'csv', 'html'],
            'reports'
        );
    }
    
    /**
     * Get all templates as array
     * 
     * @return array Associative array of directory => htaccess content
     */
    public static function getAllTemplates() {
        return [
            'root' => self::forRoot(),
            'uploads/documents' => self::forDocuments(),
            'uploads/profile_photos' => self::forProfilePhotos(),
            'uploads/payment_receipts' => self::forPaymentReceipts(),
            'uploads/reports' => self::forReports(),
            'backups' => self::forBackups(),
            'backups/database' => self::forBackups(),
            'backups/files' => self::forBackups(),
            'logs' => self::forLogs(),
            'logs/access' => self::forLogs(),
            'logs/errors' => self::forLogs(),
            'logs/payments' => self::forLogs(),
            'temp' => self::forTemp()
        ];
    }
    
    /**
     * Write .htaccess files to tenant directory
     * 
     * @param string $tenant_root Path to tenant root directory
     * @return array Result with success status
     */
    public static function writeAllToDirectory($tenant_root) {
        $templates = self::getAllTemplates();
        $created = 0;
        
        foreach ($templates as $subdir => $content) {
            $path = $tenant_root . '/' . $subdir;
            
            // Create directory if doesn't exist
            if (!file_exists($path)) {
                mkdir($path, 0755, true);
            }
            
            // Write .htaccess
            $htaccess_path = $path . '/.htaccess';
            if (file_put_contents($htaccess_path, $content) !== false) {
                chmod($htaccess_path, 0644);
                $created++;
            }
        }
        
        return [
            'success' => true,
            'created' => $created,
            'total' => count($templates)
        ];
    }
}

