<?php
/**
 * Super Admin - Storage Overview
 * 
 * Shows storage usage across all tenants
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/tenant_directory_manager.php';
requireAdminLogin();

// Check if user is super admin
if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

// Database connection
$conn = getDB();

// Initialize directory manager
$dirManager = new TenantDirectoryManager($conn);

// Get all tenants (academy_references table)
$stmt = $conn->query("
    SELECT reference_code, academy_name, contact_email 
    FROM academy_references 
    WHERE is_active = 1
    ORDER BY academy_name
");
$tenants = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate storage for each tenant
$storageData = [];
$totalSystemStorage = 0;
$totalQuota = 0;
$alertTenants = [];

foreach ($tenants as $tenant) {
    $usage = $dirManager->calculateTenantStorageUsage($tenant['reference_code']);
    
    if ($usage['success']) {
        $storageData[] = [
            'tenant_id' => $tenant['reference_code'],
            'tenant_name' => $tenant['academy_name'],
            'email' => $tenant['contact_email'],
            'usage' => $usage
        ];
        
        $totalSystemStorage += $usage['total_bytes'];
        $totalQuota += $usage['quota_bytes'];
        
        // Check for alerts (>80%)
        if ($usage['percentage'] > 80) {
            $alertTenants[] = [
                'tenant_id' => $tenant['reference_code'],
                'tenant_name' => $tenant['academy_name'],
                'percentage' => $usage['percentage']
            ];
        }
    }
}

// Get top 10 largest files across all tenants
$stmt = $conn->query("
    SELECT 
        tf.tenant_id,
        ar.academy_name as tenant_name,
        tf.original_filename,
        tf.file_category,
        tf.file_size,
        tf.upload_date
    FROM tenant_files tf
    LEFT JOIN academy_references ar 
        ON tf.tenant_id COLLATE utf8mb4_general_ci = ar.reference_code COLLATE utf8mb4_general_ci
    WHERE tf.is_deleted = 0
    ORDER BY tf.file_size DESC
    LIMIT 10
");
$largestFiles = $stmt->fetchAll(PDO::FETCH_ASSOC);

// formatBytes() function is available from includes/security_headers.php

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Storage Overview - Super Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        .stat-card {
            border-left: 4px solid;
            transition: transform 0.2s;
        }
        .stat-card:hover {
            transform: translateY(-5px);
        }
        .progress-thin {
            height: 8px;
        }
        .alert-badge {
            position: absolute;
            top: 10px;
            right: 10px;
        }
    </style>
</head>
<body class="bg-light">
    
    <!-- Navbar -->
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-hdd-stack"></i> Storage Management
            </a>
            <a href="../dashboard.php" class="btn btn-outline-light btn-sm">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
        </div>
    </nav>

    <div class="container-fluid py-4">
        
        <!-- System Overview -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card stat-card border-primary">
                    <div class="card-body">
                        <h6 class="text-muted">Total Tenants</h6>
                        <h3><?= count($tenants) ?></h3>
                    </div>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="card stat-card border-success">
                    <div class="card-body">
                        <h6 class="text-muted">System Storage Used</h6>
                        <h3><?= formatBytes($totalSystemStorage) ?></h3>
                    </div>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="card stat-card border-info">
                    <div class="card-body">
                        <h6 class="text-muted">Total Quota</h6>
                        <h3><?= formatBytes($totalQuota) ?></h3>
                    </div>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="card stat-card border-warning">
                    <div class="card-body">
                        <h6 class="text-muted">Storage Alerts</h6>
                        <h3><?= count($alertTenants) ?></h3>
                    </div>
                </div>
            </div>
        </div>

        <!-- Alerts -->
        <?php if (!empty($alertTenants)): ?>
        <div class="row mb-4">
            <div class="col-12">
                <div class="alert alert-warning">
                    <h5><i class="bi bi-exclamation-triangle"></i> Storage Alerts</h5>
                    <ul class="mb-0">
                        <?php foreach ($alertTenants as $alert): ?>
                        <li>
                            <strong><?= htmlspecialchars($alert['tenant_name']) ?></strong> 
                            - <?= $alert['percentage'] ?>% used
                        </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Storage Chart -->
        <div class="row mb-4">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-pie-chart"></i> Storage by Tenant</h5>
                    </div>
                    <div class="card-body">
                        <canvas id="storageChart" height="100"></canvas>
                    </div>
                </div>
            </div>
            
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-trophy"></i> Top Storage Users</h5>
                    </div>
                    <div class="card-body">
                        <ol class="list-group list-group-numbered">
                            <?php 
                            usort($storageData, function($a, $b) {
                                return $b['usage']['total_bytes'] - $a['usage']['total_bytes'];
                            });
                            
                            foreach (array_slice($storageData, 0, 5) as $data): 
                            ?>
                            <li class="list-group-item d-flex justify-content-between align-items-start">
                                <div class="ms-2 me-auto">
                                    <div class="fw-bold"><?= htmlspecialchars($data['tenant_name']) ?></div>
                                    <?= $data['usage']['total_formatted'] ?>
                                </div>
                                <span class="badge bg-primary rounded-pill"><?= $data['usage']['percentage'] ?>%</span>
                            </li>
                            <?php endforeach; ?>
                        </ol>
                    </div>
                </div>
            </div>
        </div>

        <!-- Tenant Storage Table -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="bi bi-table"></i> All Tenants Storage</h5>
                        <button class="btn btn-sm btn-primary" onclick="location.reload()">
                            <i class="bi bi-arrow-clockwise"></i> Refresh
                        </button>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Tenant</th>
                                        <th>Storage Used</th>
                                        <th>Quota</th>
                                        <th>Usage %</th>
                                        <th>Available</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($storageData as $data): ?>
                                    <tr>
                                        <td>
                                            <strong><?= htmlspecialchars($data['tenant_name']) ?></strong><br>
                                            <small class="text-muted"><?= htmlspecialchars($data['tenant_id']) ?></small>
                                        </td>
                                        <td><?= $data['usage']['total_formatted'] ?></td>
                                        <td><?= $data['usage']['quota_formatted'] ?></td>
                                        <td>
                                            <div class="progress progress-thin">
                                                <?php 
                                                $percentage = $data['usage']['percentage'];
                                                $color = $percentage > 90 ? 'danger' : ($percentage > 70 ? 'warning' : 'success');
                                                ?>
                                                <div class="progress-bar bg-<?= $color ?>" 
                                                     style="width: <?= $percentage ?>%">
                                                </div>
                                            </div>
                                            <small><?= $percentage ?>%</small>
                                        </td>
                                        <td><?= $data['usage']['available_formatted'] ?></td>
                                        <td>
                                            <?php if ($percentage > 90): ?>
                                                <span class="badge bg-danger">Critical</span>
                                            <?php elseif ($percentage > 80): ?>
                                                <span class="badge bg-warning">Warning</span>
                                            <?php else: ?>
                                                <span class="badge bg-success">Good</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <a href="details.php?tenant_id=<?= urlencode($data['tenant_id']) ?>" 
                                               class="btn btn-sm btn-outline-primary">
                                                <i class="bi bi-info-circle"></i> Details
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Largest Files -->
        <?php if (!empty($largestFiles)): ?>
        <div class="row mt-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-file-earmark-bar-graph"></i> Top 10 Largest Files</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Tenant</th>
                                        <th>Filename</th>
                                        <th>Category</th>
                                        <th>Size</th>
                                        <th>Upload Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($largestFiles as $file): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($file['tenant_name']) ?></td>
                                        <td><?= htmlspecialchars($file['original_filename']) ?></td>
                                        <td><span class="badge bg-secondary"><?= $file['file_category'] ?></span></td>
                                        <td><?= formatBytes($file['file_size']) ?></td>
                                        <td><?= date('Y-m-d', strtotime($file['upload_date'])) ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>

    </div>

    <!-- Chart.js Script -->
    <script>
    // Prepare chart data
    const chartData = {
        labels: <?= json_encode(array_column($storageData, 'tenant_name')) ?>,
        datasets: [{
            label: 'Storage Used (MB)',
            data: <?= json_encode(array_map(function($d) {
                return round($d['usage']['total_bytes'] / 1024 / 1024, 2);
            }, $storageData)) ?>,
            backgroundColor: [
                'rgba(255, 99, 132, 0.5)',
                'rgba(54, 162, 235, 0.5)',
                'rgba(255, 206, 86, 0.5)',
                'rgba(75, 192, 192, 0.5)',
                'rgba(153, 102, 255, 0.5)',
                'rgba(255, 159, 64, 0.5)'
            ],
            borderColor: [
                'rgba(255, 99, 132, 1)',
                'rgba(54, 162, 235, 1)',
                'rgba(255, 206, 86, 1)',
                'rgba(75, 192, 192, 1)',
                'rgba(153, 102, 255, 1)',
                'rgba(255, 159, 64, 1)'
            ],
            borderWidth: 1
        }]
    };

    // Create chart
    const ctx = document.getElementById('storageChart').getContext('2d');
    new Chart(ctx, {
        type: 'bar',
        data: chartData,
        options: {
            responsive: true,
            maintainAspectRatio: true,
            plugins: {
                legend: {
                    display: false
                },
                title: {
                    display: true,
                    text: 'Storage Usage by Tenant'
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Storage (MB)'
                    }
                }
            }
        }
    });
    </script>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

