<?php
session_start();
require_once '../../config.php';
require_once '../../includes/functions.php';

// Check if user is super admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'super_admin') {
    header('Location: ../login.php');
    exit;
}

$pdo = getDB();
$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_sms_settings'])) {
    try {
        // Save default SMS gateway settings
        $settings = [
            'default_sms_enabled' => isset($_POST['default_sms_enabled']) ? '1' : '0',
            'default_sms_provider' => $_POST['default_sms_provider'] ?? 'africastalking',
            'default_at_username' => trim($_POST['default_at_username'] ?? ''),
            'default_at_api_key' => trim($_POST['default_at_api_key'] ?? ''),
            'default_at_sender_id' => trim($_POST['default_at_sender_id'] ?? ''),
            'default_twilio_account_sid' => trim($_POST['default_twilio_account_sid'] ?? ''),
            'default_twilio_auth_token' => trim($_POST['default_twilio_auth_token'] ?? ''),
            'default_twilio_from_number' => trim($_POST['default_twilio_from_number'] ?? ''),
        ];
        
        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("
                INSERT INTO super_admin_settings (setting_key, setting_value, updated_at)
                VALUES (?, ?, NOW())
                ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()
            ");
            $stmt->execute([$key, $value, $value]);
        }
        
        $success = 'Default SMS gateway settings saved successfully!';
    } catch (Exception $e) {
        $error = 'Error saving settings: ' . $e->getMessage();
    }
}

// Load current settings
$stmt = $pdo->query("SELECT setting_key, setting_value FROM super_admin_settings WHERE setting_key LIKE 'default_sms_%' OR setting_key LIKE 'default_at_%' OR setting_key LIKE 'default_twilio_%'");
$current_settings = [];
while ($row = $stmt->fetch()) {
    $current_settings[$row['setting_key']] = $row['setting_value'];
}

// Get count of schools using default vs custom gateway
$stmt = $pdo->query("
    SELECT 
        SUM(CASE WHEN ts.setting_value = '1' AND ts2.setting_value IS NULL THEN 1 ELSE 0 END) as using_default,
        SUM(CASE WHEN ts.setting_value = '1' AND ts2.setting_value IS NOT NULL THEN 1 ELSE 0 END) as using_custom,
        SUM(CASE WHEN ts.setting_value = '0' OR ts.setting_value IS NULL THEN 1 ELSE 0 END) as sms_disabled
    FROM academy_references ar
    LEFT JOIN tenant_settings ts ON ar.reference_code = ts.academy_reference AND ts.setting_key = 'sms_enabled'
    LEFT JOIN tenant_settings ts2 ON ar.reference_code = ts2.academy_reference AND ts2.setting_key = 'sms_api_key'
    WHERE ar.is_active = 1
");
$gateway_stats = $stmt->fetch();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Default SMS Gateway Settings</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <style>
        body { background-color: #f8f9fa; }
        .settings-card { border-radius: 15px; box-shadow: 0 0 20px rgba(0,0,0,0.1); }
        .provider-option { cursor: pointer; border: 2px solid #dee2e6; border-radius: 10px; padding: 20px; transition: all 0.3s; }
        .provider-option:hover { border-color: #0d6efd; background-color: #f8f9ff; }
        .provider-option.active { border-color: #0d6efd; background-color: #e7f1ff; }
        .password-toggle { cursor: pointer; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-phone-fill text-primary"></i> Default SMS Gateway Settings</h2>
                <p class="text-muted mb-0">Configure the default SMS gateway for all schools</p>
            </div>
            <a href="../dashboard.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
        </div>

        <?php if ($success): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bi bi-check-circle-fill"></i> <?= htmlspecialchars($success) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <?php if ($error): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle-fill"></i> <?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="row mb-4">
            <div class="col-md-4">
                <div class="card text-center">
                    <div class="card-body">
                        <h3 class="text-primary"><?= $gateway_stats['using_default'] ?></h3>
                        <small class="text-muted">Schools Using Default Gateway</small>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card text-center">
                    <div class="card-body">
                        <h3 class="text-success"><?= $gateway_stats['using_custom'] ?></h3>
                        <small class="text-muted">Schools Using Custom Gateway</small>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card text-center">
                    <div class="card-body">
                        <h3 class="text-secondary"><?= $gateway_stats['sms_disabled'] ?></h3>
                        <small class="text-muted">Schools with SMS Disabled</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Settings Form -->
        <div class="card settings-card">
            <div class="card-body p-4">
                <form method="POST" action="">
                    
                    <!-- Enable/Disable Default Gateway -->
                    <div class="mb-4">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" id="default_sms_enabled" 
                                   name="default_sms_enabled" value="1"
                                   <?= ($current_settings['default_sms_enabled'] ?? '0') == '1' ? 'checked' : '' ?>>
                            <label class="form-check-label" for="default_sms_enabled">
                                <strong>Enable Default SMS Gateway</strong>
                                <br><small class="text-muted">Schools without custom gateway will use these settings</small>
                            </label>
                        </div>
                    </div>

                    <hr>

                    <!-- Provider Selection -->
                    <h5 class="mb-3">Select SMS Provider</h5>
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <div class="provider-option <?= ($current_settings['default_sms_provider'] ?? 'africastalking') == 'africastalking' ? 'active' : '' ?>" 
                                 onclick="selectProvider('africastalking')">
                                <input type="radio" name="default_sms_provider" value="africastalking" 
                                       id="provider_at" class="form-check-input me-2"
                                       <?= ($current_settings['default_sms_provider'] ?? 'africastalking') == 'africastalking' ? 'checked' : '' ?>>
                                <label for="provider_at" class="d-block">
                                    <h6 class="mb-2"><i class="bi bi-phone"></i> Africa's Talking</h6>
                                    <small class="text-muted">Recommended for Africa</small><br>
                                    <small class="text-muted">~E0.20-0.30 per SMS</small>
                                </label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="provider-option <?= ($current_settings['default_sms_provider'] ?? '') == 'twilio' ? 'active' : '' ?>" 
                                 onclick="selectProvider('twilio')">
                                <input type="radio" name="default_sms_provider" value="twilio" 
                                       id="provider_twilio" class="form-check-input me-2"
                                       <?= ($current_settings['default_sms_provider'] ?? '') == 'twilio' ? 'checked' : '' ?>>
                                <label for="provider_twilio" class="d-block">
                                    <h6 class="mb-2"><i class="bi bi-phone"></i> Twilio</h6>
                                    <small class="text-muted">Global provider</small><br>
                                    <small class="text-muted">Varies by country</small>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Africa's Talking Settings -->
                    <div id="africastalking_settings" style="<?= ($current_settings['default_sms_provider'] ?? 'africastalking') == 'africastalking' ? '' : 'display:none' ?>">
                        <h5 class="mb-3"><i class="bi bi-phone-fill text-primary"></i> Africa's Talking Settings</h5>
                        
                        <div class="alert alert-info">
                            <strong><i class="bi bi-info-circle"></i> Setup Instructions:</strong>
                            <ol class="mb-0 mt-2">
                                <li>Sign up at <a href="https://africastalking.com/" target="_blank">africastalking.com</a></li>
                                <li>Get your Username and API Key from dashboard</li>
                                <li>Enter credentials below</li>
                            </ol>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Username</label>
                                <input type="text" class="form-control" name="default_at_username" 
                                       value="<?= htmlspecialchars($current_settings['default_at_username'] ?? '') ?>"
                                       placeholder="sandbox or your username">
                                <small class="text-muted">From Africa's Talking dashboard</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Sender ID</label>
                                <input type="text" class="form-control" name="default_at_sender_id" 
                                       value="<?= htmlspecialchars($current_settings['default_at_sender_id'] ?? '') ?>"
                                       placeholder="SKOLO" maxlength="11">
                                <small class="text-muted">Max 11 characters (optional)</small>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">API Key</label>
                            <div class="input-group">
                                <input type="password" class="form-control" name="default_at_api_key" 
                                       id="at_api_key"
                                       value="<?= htmlspecialchars($current_settings['default_at_api_key'] ?? '') ?>"
                                       placeholder="atsk_...">
                                <button class="btn btn-outline-secondary password-toggle" type="button" 
                                        onclick="togglePassword('at_api_key')">
                                    <i class="bi bi-eye"></i>
                                </button>
                            </div>
                            <small class="text-muted">Keep this secret!</small>
                        </div>
                    </div>

                    <!-- Twilio Settings -->
                    <div id="twilio_settings" style="<?= ($current_settings['default_sms_provider'] ?? '') == 'twilio' ? '' : 'display:none' ?>">
                        <h5 class="mb-3"><i class="bi bi-phone-fill text-primary"></i> Twilio Settings</h5>
                        
                        <div class="alert alert-info">
                            <strong><i class="bi bi-info-circle"></i> Setup Instructions:</strong>
                            <ol class="mb-0 mt-2">
                                <li>Sign up at <a href="https://www.twilio.com/" target="_blank">twilio.com</a></li>
                                <li>Get your Account SID and Auth Token</li>
                                <li>Get a Twilio phone number</li>
                            </ol>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Account SID</label>
                                <input type="text" class="form-control" name="default_twilio_account_sid" 
                                       value="<?= htmlspecialchars($current_settings['default_twilio_account_sid'] ?? '') ?>"
                                       placeholder="ACxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">From Number</label>
                                <input type="text" class="form-control" name="default_twilio_from_number" 
                                       value="<?= htmlspecialchars($current_settings['default_twilio_from_number'] ?? '') ?>"
                                       placeholder="+268XXXXXXXX">
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Auth Token</label>
                            <div class="input-group">
                                <input type="password" class="form-control" name="default_twilio_auth_token" 
                                       id="twilio_auth_token"
                                       value="<?= htmlspecialchars($current_settings['default_twilio_auth_token'] ?? '') ?>"
                                       placeholder="Your auth token">
                                <button class="btn btn-outline-secondary password-toggle" type="button" 
                                        onclick="togglePassword('twilio_auth_token')">
                                    <i class="bi bi-eye"></i>
                                </button>
                            </div>
                            <small class="text-muted">Keep this secret!</small>
                        </div>
                    </div>

                    <hr>

                    <!-- Actions -->
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <a href="/Multi-Tanent/check_sms_config.php" target="_blank" class="btn btn-outline-info">
                                <i class="bi bi-gear"></i> Test SMS Configuration
                            </a>
                        </div>
                        <div>
                            <button type="submit" name="save_sms_settings" class="btn btn-primary btn-lg">
                                <i class="bi bi-save"></i> Save Default Gateway Settings
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>

        <!-- Help Section -->
        <div class="card mt-4">
            <div class="card-body">
                <h5><i class="bi bi-question-circle text-info"></i> How It Works</h5>
                <ul class="mb-0">
                    <li><strong>Default Gateway:</strong> Used by schools that don't configure their own SMS gateway</li>
                    <li><strong>Custom Gateway:</strong> Schools can override default with their own credentials in their settings</li>
                    <li><strong>Fallback:</strong> If school's custom gateway fails, system tries default gateway</li>
                    <li><strong>Disable:</strong> Schools can disable SMS entirely for their school</li>
                </ul>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function selectProvider(provider) {
            // Update radio buttons
            document.querySelectorAll('input[name="default_sms_provider"]').forEach(radio => {
                radio.checked = radio.value === provider;
            });
            
            // Update UI
            document.querySelectorAll('.provider-option').forEach(option => {
                option.classList.remove('active');
            });
            event.currentTarget.closest('.provider-option').classList.add('active');
            
            // Show/hide settings
            document.getElementById('africastalking_settings').style.display = provider === 'africastalking' ? 'block' : 'none';
            document.getElementById('twilio_settings').style.display = provider === 'twilio' ? 'block' : 'none';
        }
        
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = event.currentTarget.querySelector('i');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('bi-eye');
                icon.classList.add('bi-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('bi-eye-slash');
                icon.classList.add('bi-eye');
            }
        }
    </script>
</body>
</html>

