<?php
/**
 * View All School URLs
 * Super Admin - Quick reference for all school links
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/school_url_generator.php';
require_once '../../includes/qr_generator.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$pdo = getDB();

// Get all schools
$stmt = $pdo->query("
    SELECT 
        ar.id,
        ar.reference_code,
        ar.academy_name,
        ar.is_active,
        ar.created_at,
        ts.setting_value as contact_email
    FROM academy_references ar
    LEFT JOIN tenant_settings ts ON ar.reference_code = ts.academy_reference AND ts.setting_key = 'school_email'
    ORDER BY ar.academy_name
");
$schools = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>School URLs - Super Admin</title>
    
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        .url-box {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 12px;
            margin-bottom: 10px;
            font-family: monospace;
            font-size: 0.9rem;
        }
        .url-label {
            font-weight: 600;
            color: #495057;
            margin-bottom: 5px;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            font-size: 0.85rem;
            text-transform: uppercase;
        }
        .url-link {
            color: #0d6efd;
            text-decoration: none;
            word-break: break-all;
        }
        .url-link:hover {
            text-decoration: underline;
        }
        .copy-btn {
            font-size: 0.75rem;
            padding: 2px 8px;
        }
        .school-card {
            border-left: 4px solid #0d6efd;
        }
        .school-card.inactive {
            border-left-color: #6c757d;
            opacity: 0.7;
        }
        .qr-code {
            max-width: 150px;
            border: 1px solid #dee2e6;
            padding: 8px;
            border-radius: 4px;
            background: white;
        }
        @media print {
            .no-print {
                display: none;
            }
            .school-card {
                page-break-inside: avoid;
                margin-bottom: 20px;
            }
        }
    </style>
</head>
<body class="bg-light">
    <nav class="navbar navbar-dark bg-dark no-print">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </a>
            <div class="d-flex gap-2">
                <button class="btn btn-sm btn-outline-light" onclick="window.print()">
                    <i class="bi bi-printer me-1"></i>Print All
                </button>
                <a class="btn btn-sm btn-light" href="list.php">
                    <i class="bi bi-arrow-left me-1"></i>Back to Schools
                </a>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <div class="row mb-4 no-print">
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <h4 class="card-title mb-3">
                            <i class="bi bi-link-45deg me-2"></i>All School URLs
                        </h4>
                        <p class="mb-0 text-muted">
                            <i class="bi bi-info-circle me-1"></i>
                            View and copy registration/login URLs for all schools. Share these links with schools for student and parent registration.
                        </p>
                    </div>
                </div>
            </div>
        </div>

        <?php if (empty($schools)): ?>
            <div class="alert alert-info">
                <i class="bi bi-info-circle me-2"></i>
                No schools found. <a href="create.php">Create your first school</a>.
            </div>
        <?php else: ?>
            <div class="row">
                <?php foreach ($schools as $school): ?>
                    <?php
                    $urlGenerator = new SchoolURLGenerator($school['reference_code']);
                    $urls = $urlGenerator->getAllURLs();
                    ?>
                    
                    <div class="col-12 mb-4">
                        <div class="card shadow-sm school-card <?= $school['is_active'] ? '' : 'inactive' ?>">
                            <div class="card-header bg-white border-bottom">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h5 class="mb-1">
                                            <?= htmlspecialchars($school['academy_name']) ?>
                                            <?php if (!$school['is_active']): ?>
                                                <span class="badge bg-secondary">Inactive</span>
                                            <?php endif; ?>
                                        </h5>
                                        <small class="text-muted">
                                            Code: <strong><?= htmlspecialchars($school['reference_code']) ?></strong>
                                            <?php if ($school['contact_email']): ?>
                                                | Email: <?= htmlspecialchars($school['contact_email']) ?>
                                            <?php endif; ?>
                                        </small>
                                    </div>
                                    <button class="btn btn-sm btn-outline-primary no-print" onclick="copyAllURLs('<?= $school['reference_code'] ?>')">
                                        <i class="bi bi-clipboard me-1"></i>Copy All
                                    </button>
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <!-- URLs Column -->
                                    <div class="col-md-8">
                                        <!-- Student Registration -->
                                        <div class="url-box">
                                            <div class="url-label">
                                                <i class="bi bi-person-plus text-primary"></i> Student Registration
                                            </div>
                                            <div class="d-flex justify-content-between align-items-center">
                                                <a href="<?= htmlspecialchars($urls['student_registration']) ?>" target="_blank" class="url-link">
                                                    <?= htmlspecialchars($urls['student_registration']) ?>
                                                </a>
                                                <button class="btn btn-sm btn-outline-secondary copy-btn no-print ms-2" 
                                                        onclick="copyToClipboard('<?= htmlspecialchars($urls['student_registration']) ?>', this)">
                                                    <i class="bi bi-clipboard"></i>
                                                </button>
                                            </div>
                                        </div>

                                        <!-- Student Login -->
                                        <div class="url-box">
                                            <div class="url-label">
                                                <i class="bi bi-box-arrow-in-right text-success"></i> Student Login
                                            </div>
                                            <div class="d-flex justify-content-between align-items-center">
                                                <a href="<?= htmlspecialchars($urls['student_login']) ?>" target="_blank" class="url-link">
                                                    <?= htmlspecialchars($urls['student_login']) ?>
                                                </a>
                                                <button class="btn btn-sm btn-outline-secondary copy-btn no-print ms-2" 
                                                        onclick="copyToClipboard('<?= htmlspecialchars($urls['student_login']) ?>', this)">
                                                    <i class="bi bi-clipboard"></i>
                                                </button>
                                            </div>
                                        </div>

                                        <!-- Parent Registration -->
                                        <div class="url-box">
                                            <div class="url-label">
                                                <i class="bi bi-people text-info"></i> Parent Registration
                                            </div>
                                            <div class="d-flex justify-content-between align-items-center">
                                                <a href="<?= htmlspecialchars($urls['parent_registration']) ?>" target="_blank" class="url-link">
                                                    <?= htmlspecialchars($urls['parent_registration']) ?>
                                                </a>
                                                <button class="btn btn-sm btn-outline-secondary copy-btn no-print ms-2" 
                                                        onclick="copyToClipboard('<?= htmlspecialchars($urls['parent_registration']) ?>', this)">
                                                    <i class="bi bi-clipboard"></i>
                                                </button>
                                            </div>
                                        </div>

                                        <!-- Parent Login -->
                                        <div class="url-box">
                                            <div class="url-label">
                                                <i class="bi bi-box-arrow-in-right text-warning"></i> Parent Login
                                            </div>
                                            <div class="d-flex justify-content-between align-items-center">
                                                <a href="<?= htmlspecialchars($urls['parent_login']) ?>" target="_blank" class="url-link">
                                                    <?= htmlspecialchars($urls['parent_login']) ?>
                                                </a>
                                                <button class="btn btn-sm btn-outline-secondary copy-btn no-print ms-2" 
                                                        onclick="copyToClipboard('<?= htmlspecialchars($urls['parent_login']) ?>', this)">
                                                    <i class="bi bi-clipboard"></i>
                                                </button>
                                            </div>
                                        </div>

                                        <!-- Admin Login -->
                                        <div class="url-box">
                                            <div class="url-label">
                                                <i class="bi bi-person-lock text-danger"></i> Admin Login
                                            </div>
                                            <div class="d-flex justify-content-between align-items-center">
                                                <a href="<?= htmlspecialchars($urls['admin_login']) ?>" target="_blank" class="url-link">
                                                    <?= htmlspecialchars($urls['admin_login']) ?>
                                                </a>
                                                <button class="btn btn-sm btn-outline-secondary copy-btn no-print ms-2" 
                                                        onclick="copyToClipboard('<?= htmlspecialchars($urls['admin_login']) ?>', this)">
                                                    <i class="bi bi-clipboard"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- QR Codes Column -->
                                    <div class="col-md-4 text-center no-print">
                                        <p class="text-muted mb-2"><small><strong>Student Registration QR Code</strong></small></p>
                                        <?php
                                            try {
                                                // Get URL from the generator
                                                $studentRegUrl = isset($urls['student_registration']) ? $urls['student_registration'] : '';
                                                
                                                // Fallback: construct URL manually if missing
                                                if (empty($studentRegUrl)) {
                                                    $schoolCode = isset($school['reference_code']) ? strtolower($school['reference_code']) : '';
                                                    if (!empty($schoolCode)) {
                                                        $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
                                                        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
                                                        $studentRegUrl = $protocol . '://' . $host . '/Multi-Tanent/' . $schoolCode . '/register';
                                                    }
                                                }
                                                
                                                if (!empty($studentRegUrl)) {
                                                    // Build QR code URL using qrserver.com API
                                                    $encodedUrl = urlencode($studentRegUrl);
                                                    $qrCodeSrc = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&margin=2&data=' . $encodedUrl;
                                                    ?>
                                                    <div class="mb-2">
                                                        <img src="<?= htmlspecialchars($qrCodeSrc) ?>" 
                                                             alt="Student Registration QR Code" 
                                                             class="qr-code" 
                                                             style="max-width: 250px; width: 100%; height: auto; display: block; margin: 0 auto; border: 1px solid #ddd; padding: 10px; background: white;"
                                                             loading="lazy">
                                                    </div>
                                                    <small class="text-muted d-block mb-2" style="font-size: 0.7rem; word-break: break-all; max-width: 250px; margin: 0 auto;">
                                                        <?= htmlspecialchars($studentRegUrl) ?>
                                                    </small>
                                                    <?php
                                                } else {
                                                    ?>
                                                    <div class="alert alert-warning mb-2">
                                                        <i class="bi bi-exclamation-triangle me-1"></i> Registration URL not available.
                                                        <br><small>School Code: <?= htmlspecialchars($school['reference_code'] ?? 'N/A') ?></small>
                                                        <br><small>URLs array keys: <?= implode(', ', array_keys($urls ?? [])) ?></small>
                                                    </div>
                                                    <?php
                                                }
                                            } catch (Exception $e) {
                                                error_log("QR code generation error: " . $e->getMessage());
                                                ?>
                                                <div class="alert alert-danger mb-2">
                                                    <i class="bi bi-exclamation-triangle me-1"></i> Error: <?= htmlspecialchars($e->getMessage()) ?>
                                                </div>
                                                <?php
                                            }
                                        ?>
                                        <br>
                                        <a href="view.php?id=<?= $school['id'] ?>" class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-eye me-1"></i>View All QR Codes
                                        </a>
                                    </div>
                                </div>

                                <!-- Hidden textarea for copy all functionality -->
                                <textarea id="urls-<?= $school['reference_code'] ?>" style="position:absolute;left:-9999px;">
Student Registration: <?= $urls['student_registration'] ?>

Student Login: <?= $urls['student_login'] ?>

Parent Registration: <?= $urls['parent_registration'] ?>

Parent Login: <?= $urls['parent_login'] ?>

Admin Login: <?= $urls['admin_login'] ?>
</textarea>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function copyToClipboard(text, button) {
            navigator.clipboard.writeText(text).then(function() {
                // Change icon to checkmark
                const icon = button.querySelector('i');
                const originalClass = icon.className;
                icon.className = 'bi bi-check2';
                button.classList.remove('btn-outline-secondary');
                button.classList.add('btn-success');
                
                // Reset after 2 seconds
                setTimeout(function() {
                    icon.className = originalClass;
                    button.classList.remove('btn-success');
                    button.classList.add('btn-outline-secondary');
                }, 2000);
            }).catch(function(err) {
                alert('Failed to copy: ' + err);
            });
        }

        function copyAllURLs(schoolCode) {
            const textarea = document.getElementById('urls-' + schoolCode);
            textarea.select();
            document.execCommand('copy');
            
            // Show feedback
            const btn = event.target.closest('button');
            const originalHTML = btn.innerHTML;
            btn.innerHTML = '<i class="bi bi-check2 me-1"></i>Copied!';
            btn.classList.remove('btn-outline-primary');
            btn.classList.add('btn-success');
            
            setTimeout(function() {
                btn.innerHTML = originalHTML;
                btn.classList.remove('btn-success');
                btn.classList.add('btn-outline-primary');
            }, 2000);
        }
    </script>
</body>
</html>

