<?php
/**
 * View School Details
 * Super Admin Only
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/qr_generator.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$schoolId = (int)($_GET['id'] ?? 0);
$pdo = getDB();

// Get school details
$stmt = $pdo->prepare("
    SELECT ar.*, a.username as admin_username, a.momo_number, aw.balance, aw.total_earnings, aw.withdrawn
    FROM academy_references ar
    LEFT JOIN admins a ON ar.admin_id = a.id
    LEFT JOIN admin_wallet aw ON aw.admin_id = ar.admin_id
    WHERE ar.id = ?
");
$stmt->execute([$schoolId]);
$school = $stmt->fetch();

if (!$school) {
    header('Location: list.php');
    exit;
}

// Get school settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$school['reference_code']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$softDeleted = ($settings['soft_deleted'] ?? '0') === '1';
$softDeletedAt = $settings['soft_deleted_at'] ?? null;
$csrfToken = generateCSRFToken();

// Get school stats
$stats = getSchoolStats($school['reference_code']);

// Generate URLs for this school
require_once '../../includes/school_url_generator.php';
$urlGenerator = new SchoolURLGenerator($school['reference_code']);
$schoolURLs = $urlGenerator->getAllURLs();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($school['academy_name']) ?> - Details</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="list.php">
                    <i class="bi bi-arrow-left me-1"></i>Back to Schools
                </a>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <div class="row mb-4">
            <div class="col-md-8">
                <h2><?= htmlspecialchars($school['academy_name']) ?></h2>
                <p class="text-muted"><?= htmlspecialchars($school['description'] ?? '') ?></p>
            </div>
            <div class="col-md-4">
                <div class="d-flex flex-wrap gap-2 justify-content-end">
                    <a href="edit.php?id=<?= $school['id'] ?>" class="btn btn-primary">
                        <i class="bi bi-pencil me-1"></i>Edit
                    </a>
                    <a href="../../select_school.php?school=<?= urlencode($school['reference_code']) ?>" 
                       class="btn btn-success">
                        <i class="bi bi-box-arrow-in-right me-1"></i>Access School
                    </a>
                    <?php if (!$softDeleted): ?>
                        <form method="post" action="actions.php"
                              class="d-inline"
                              onsubmit="return confirm('Soft delete this school? Admin access will be disabled, but data will remain for restore.');">
                            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                            <input type="hidden" name="action" value="soft_delete">
                            <input type="hidden" name="school_id" value="<?= (int)$school['id'] ?>">
                            <input type="hidden" name="redirect" value="view">
                            <button type="submit" class="btn btn-outline-warning">
                                <i class="bi bi-archive me-1"></i>Soft Delete
                            </button>
                        </form>
                    <?php else: ?>
                        <form method="post" action="actions.php"
                              class="d-inline"
                              onsubmit="return confirm('Restore this school and re-enable access?');">
                            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                            <input type="hidden" name="action" value="restore">
                            <input type="hidden" name="school_id" value="<?= (int)$school['id'] ?>">
                            <input type="hidden" name="redirect" value="view">
                            <button type="submit" class="btn btn-outline-success">
                                <i class="bi bi-arrow-clockwise me-1"></i>Restore
                            </button>
                        </form>
                        <form method="post" action="actions.php"
                              class="d-inline"
                              onsubmit="return confirm('Permanently delete this school? All related data – including students and Moodle enrollments – will be removed. This cannot be undone.');">
                            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                            <input type="hidden" name="action" value="permanent_delete">
                            <input type="hidden" name="school_id" value="<?= (int)$school['id'] ?>">
                            <input type="hidden" name="redirect" value="list">
                            <button type="submit" class="btn btn-outline-danger">
                                <i class="bi bi-trash me-1"></i>Delete Permanently
                            </button>
                        </form>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <?php if (isset($_GET['status'])): ?>
            <?php
                $status = $_GET['status'];
                $alertClass = 'info';
                $message = '';
                switch ($status) {
                    case 'soft_deleted':
                        $alertClass = 'warning';
                        $message = 'School soft deleted. Admin access disabled but data remains for restore.';
                        break;
                    case 'restored':
                        $alertClass = 'success';
                        $message = 'School restored successfully.';
                        break;
                    case 'deleted':
                        $alertClass = 'success';
                        $message = 'School permanently deleted.';
                        break;
                    case 'invalid_csrf':
                        $alertClass = 'danger';
                        $message = 'Security verification failed. Please try again.';
                        break;
                    case 'invalid_action':
                        $alertClass = 'danger';
                        $message = 'Invalid action requested.';
                        break;
                    case 'error':
                        $alertClass = 'danger';
                        $message = 'Operation failed. Check logs for details.';
                        break;
                }
            ?>
            <?php if ($message): ?>
                <div class="alert alert-<?= $alertClass ?> alert-dismissible fade show">
                    <i class="bi bi-info-circle me-2"></i><?= htmlspecialchars($message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="row g-4 mb-4">
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <i class="bi bi-people fs-1 text-primary"></i>
                        <h3 class="mt-2"><?= number_format($stats['total_students']) ?></h3>
                        <p class="text-muted mb-0">Total Students</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <i class="bi bi-check-circle fs-1 text-success"></i>
                        <h3 class="mt-2"><?= number_format($stats['total_payments']) ?></h3>
                        <p class="text-muted mb-0">Payments</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <i class="bi bi-currency-exchange fs-1 text-info"></i>
                        <h3 class="mt-2">E<?= number_format($stats['total_revenue'], 2) ?></h3>
                        <p class="text-muted mb-0">Total Revenue</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body text-center">
                        <i class="bi bi-wallet2 fs-1 text-warning"></i>
                        <h3 class="mt-2">E<?= number_format($school['balance'] ?? 0, 2) ?></h3>
                        <p class="text-muted mb-0">Wallet Balance</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- School URLs -->
        <div class="card shadow-sm mb-4">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0">
                    <i class="bi bi-link-45deg me-2"></i>
                    School Registration & Access URLs
                </h5>
            </div>
            <div class="card-body">
                <p class="text-muted mb-3">
                    <i class="bi bi-info-circle me-1"></i>
                    Clean URLs for this school. Share with students and parents.
                </p>
                
                <div class="row">
                    <div class="col-md-6">
                        <!-- Student Registration -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">
                                <i class="bi bi-person-plus text-primary me-1"></i>
                                Student Registration
                            </label>
                            <div class="input-group input-group-sm">
                                <input type="text" class="form-control" 
                                       value="<?= htmlspecialchars($schoolURLs['student_registration']) ?>" 
                                       readonly id="url_student_reg">
                                <button class="btn btn-outline-secondary" onclick="copyURL('url_student_reg')">
                                    <i class="bi bi-clipboard"></i>
                                </button>
                            </div>
                        </div>
                        
                        <!-- Parent Registration -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">
                                <i class="bi bi-people text-success me-1"></i>
                                Parent Registration
                            </label>
                            <div class="input-group input-group-sm">
                                <input type="text" class="form-control" 
                                       value="<?= htmlspecialchars($schoolURLs['parent_registration']) ?>" 
                                       readonly id="url_parent_reg">
                                <button class="btn btn-outline-secondary" onclick="copyURL('url_parent_reg')">
                                    <i class="bi bi-clipboard"></i>
                                </button>
                            </div>
                        </div>
                        
                        <!-- Student Login -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">
                                <i class="bi bi-box-arrow-in-right text-info me-1"></i>
                                Student Login
                            </label>
                            <div class="input-group input-group-sm">
                                <input type="text" class="form-control" 
                                       value="<?= htmlspecialchars($schoolURLs['student_login']) ?>" 
                                       readonly id="url_student_login">
                                <button class="btn btn-outline-secondary" onclick="copyURL('url_student_login')">
                                    <i class="bi bi-clipboard"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <!-- Parent Login -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">
                                <i class="bi bi-shield-check text-warning me-1"></i>
                                Parent Login
                            </label>
                            <div class="input-group input-group-sm">
                                <input type="text" class="form-control" 
                                       value="<?= htmlspecialchars($schoolURLs['parent_login']) ?>" 
                                       readonly id="url_parent_login">
                                <button class="btn btn-outline-secondary" onclick="copyURL('url_parent_login')">
                                    <i class="bi bi-clipboard"></i>
                                </button>
                            </div>
                        </div>
                        
                        <!-- QR Code -->
                        <div class="text-center p-3 bg-light rounded">
                            <h6 class="mb-2">Student Registration QR Code</h6>
                            <?php
                                try {
                                    $studentRegUrl = $urlGenerator->getStudentRegistrationURL();
                                    if (!empty($studentRegUrl)) {
                                        // Build QR code URL using qrserver.com API
                                        $encodedUrl = urlencode($studentRegUrl);
                                        $qrCodeSrc = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&margin=2&data=' . $encodedUrl;
                                        ?>
                                        <div class="mb-2">
                                            <img src="<?= htmlspecialchars($qrCodeSrc) ?>" 
                                                 alt="Student Registration QR Code"
                                                 id="qr_student_registration"
                                                 class="img-fluid" 
                                                 style="max-width: 250px; width: 100%; height: auto; display: block; margin: 0 auto; border: 1px solid #ddd; padding: 10px; background: white;"
                                                 loading="lazy">
                                        </div>
                                        <small class="text-muted d-block mb-2" style="font-size: 0.7rem; word-break: break-all; max-width: 250px; margin: 0 auto;">
                                            <?= htmlspecialchars($studentRegUrl) ?>
                                        </small>
                                        <?php
                                    } else {
                                        ?>
                                        <div class="alert alert-warning">
                                            <i class="bi bi-exclamation-triangle me-1"></i> Registration URL not available.
                                        </div>
                                        <?php
                                    }
                                } catch (Exception $e) {
                                    error_log("QR code generation error: " . $e->getMessage());
                                    ?>
                                    <div class="alert alert-danger">
                                        <i class="bi bi-exclamation-triangle me-1"></i> Error: <?= htmlspecialchars($e->getMessage()) ?>
                                    </div>
                                    <?php
                                }
                            ?>
                            <div>
                                <button class="btn btn-sm btn-outline-primary" onclick="printQR()">
                                    <i class="bi bi-printer me-1"></i>Print
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Details -->
        <div class="row">
            <div class="col-md-6 mb-4">
                <div class="card h-100">
                    <div class="card-header bg-white">
                        <h5 class="mb-0"><i class="bi bi-info-circle me-2"></i>School Information</h5>
                    </div>
                    <div class="card-body">
                        <table class="table table-sm">
                            <tr>
                                <th width="40%">Reference Code:</th>
                                <td><span class="badge bg-secondary"><?= htmlspecialchars($school['reference_code']) ?></span></td>
                            </tr>
                            <tr>
                                <th>School Name:</th>
                                <td><?= htmlspecialchars($settings['school_name'] ?? $school['academy_name']) ?></td>
                            </tr>
                            <tr>
                                <th>Email:</th>
                                <td><?= htmlspecialchars($settings['school_email'] ?? 'Not set') ?></td>
                            </tr>
                            <tr>
                                <th>Phone:</th>
                                <td><?= htmlspecialchars($settings['school_phone'] ?? 'Not set') ?></td>
                            </tr>
                            <tr>
                                <th>Address:</th>
                                <td><?= htmlspecialchars($settings['school_address'] ?? 'Not set') ?></td>
                            </tr>
                            <tr>
                                <th>Academic Year:</th>
                                <td><?= htmlspecialchars($settings['academic_year'] ?? 'Not set') ?></td>
                            </tr>
                            <tr>
                                <th>Max Students:</th>
                                <td>
                                    <span class="badge bg-info"><?= number_format($settings['max_students'] ?? 500) ?></span>
                                    <?php 
                                    $currentCount = $stats['total_students'] ?? 0;
                                    $maxStudents = (int)($settings['max_students'] ?? 500);
                                    $remaining = max(0, $maxStudents - $currentCount);
                                    $percentUsed = $maxStudents > 0 ? ($currentCount / $maxStudents) * 100 : 0;
                                    ?>
                                    <small class="text-muted d-block mt-1">
                                        <?= number_format($currentCount) ?> enrolled, <?= number_format($remaining) ?> remaining 
                                        (<?= number_format($percentUsed, 1) ?>% capacity)
                                    </small>
                                </td>
                            </tr>
                            <tr>
                                <th>Filtering Mode:</th>
                                <td>
                                    <?php 
                                    $filterMode = $settings['filtering_mode'] ?? 'auto';
                                    $badgeClass = $filterMode === 'auto' ? 'bg-primary' : ($filterMode === 'client' ? 'bg-success' : 'bg-warning');
                                    ?>
                                    <span class="badge <?= $badgeClass ?>"><?= ucfirst($filterMode) ?></span>
                                    <small class="text-muted d-block mt-1">
                                        <?php if ($filterMode === 'auto'): ?>
                                            Auto-selects: <?= $currentCount < 2000 ? 'Client-side (Fast)' : 'Server-side (Efficient)' ?>
                                        <?php elseif ($filterMode === 'client'): ?>
                                            JavaScript filtering (Best for <2,000 students)
                                        <?php else: ?>
                                            PHP/AJAX filtering (Best for 3,000+ students)
                                        <?php endif; ?>
                                    </small>
                                </td>
                            </tr>
                            <tr>
                                <th>Status:</th>
                                <td>
                                    <?php if ($softDeleted): ?>
                                        <span class="badge bg-dark">Soft Deleted</span>
                                        <?php if ($softDeletedAt): ?>
                                            <small class="text-muted d-block">Since <?= date('F d, Y H:i', strtotime($softDeletedAt)) ?></small>
                                        <?php endif; ?>
                                    <?php elseif ($school['is_active']): ?>
                                        <span class="badge bg-success">Active</span>
                                    <?php else: ?>
                                        <span class="badge bg-danger">Suspended</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <tr>
                                <th>Created:</th>
                                <td><?= date('F d, Y', strtotime($school['created_at'])) ?></td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>

            <div class="col-md-6 mb-4">
                <div class="card h-100">
                    <div class="card-header bg-white">
                        <h5 class="mb-0"><i class="bi bi-person-badge me-2"></i>Administrator</h5>
                    </div>
                    <div class="card-body">
                        <table class="table table-sm">
                            <tr>
                                <th width="40%">Username:</th>
                                <td><?= htmlspecialchars($school['admin_username'] ?? 'Not assigned') ?></td>
                            </tr>
                            <tr>
                                <th>Admin ID:</th>
                                <td><?= $school['admin_id'] ? '#' . $school['admin_id'] : 'Not assigned' ?></td>
                            </tr>
                            <tr>
                                <th>MoMo Number:</th>
                                <td><?= htmlspecialchars($school['momo_number'] ?? 'Not set') ?></td>
                            </tr>
                        </table>

                        <hr>

                        <h6>Wallet Information</h6>
                        <table class="table table-sm">
                            <tr>
                                <th width="40%">Balance:</th>
                                <td>E<?= number_format($school['balance'] ?? 0, 2) ?></td>
                            </tr>
                            <tr>
                                <th>Total Earnings:</th>
                                <td>E<?= number_format($school['total_earnings'] ?? 0, 2) ?></td>
                            </tr>
                            <tr>
                                <th>Total Withdrawn:</th>
                                <td>E<?= number_format($school['withdrawn'] ?? 0, 2) ?></td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Branding Settings -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header bg-white">
                        <h5 class="mb-0"><i class="bi bi-palette me-2"></i>Branding & Settings</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6>Brand Colors</h6>
                                <div class="d-flex gap-3 mb-3">
                                    <div>
                                        <small class="text-muted d-block">Primary</small>
                                        <div style="width: 50px; height: 50px; background: <?= htmlspecialchars($settings['primary_color'] ?? '#4F46E5') ?>; border-radius: 8px;"></div>
                                        <code class="small"><?= htmlspecialchars($settings['primary_color'] ?? '#4F46E5') ?></code>
                                    </div>
                                    <div>
                                        <small class="text-muted d-block">Secondary</small>
                                        <div style="width: 50px; height: 50px; background: <?= htmlspecialchars($settings['secondary_color'] ?? '#10B981') ?>; border-radius: 8px;"></div>
                                        <code class="small"><?= htmlspecialchars($settings['secondary_color'] ?? '#10B981') ?></code>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h6>Logo</h6>
                                <?php if (isset($settings['logo_url'])): ?>
                                    <img src="<?= htmlspecialchars($settings['logo_url']) ?>" 
                                         alt="School Logo" 
                                         style="max-height: 60px;"
                                         onerror="this.src='/images/img-melanesys-logo.svg'">
                                <?php else: ?>
                                    <p class="text-muted">No logo uploaded</p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Copy URL to clipboard
        function copyURL(inputId) {
            const input = document.getElementById(inputId);
            input.select();
            input.setSelectionRange(0, 99999); // For mobile
            
            navigator.clipboard.writeText(input.value).then(function() {
                // Visual feedback
                const btn = event.target.closest('button');
                const originalHTML = btn.innerHTML;
                btn.innerHTML = '<i class="bi bi-check"></i>';
                btn.classList.add('btn-success');
                btn.classList.remove('btn-outline-secondary');
                
                setTimeout(() => {
                    btn.innerHTML = originalHTML;
                    btn.classList.remove('btn-success');
                    btn.classList.add('btn-outline-secondary');
                }, 2000);
            }).catch(function(err) {
                alert('Failed to copy: ' + err);
            });
        }
        
        // Print QR Code
        function printQR() {
            const printWindow = window.open('', '_blank');
            const qrImage = document.getElementById('qr_student_registration');
            const studentRegInput = document.getElementById('url_student_reg');

            if (!qrImage || !studentRegInput) {
                alert('QR code not available.');
                return;
            }
            
            printWindow.document.write(`
                <html>
                <head>
                    <title>Student Registration QR Code</title>
                    <style>
                        body { 
                            text-align: center; 
                            font-family: Arial, sans-serif; 
                            padding: 40px;
                        }
                        h1 { color: #333; }
                        img { max-width: 400px; margin: 30px 0; }
                        .url { 
                            font-size: 18px; 
                            background: #f0f0f0; 
                            padding: 15px; 
                            border-radius: 8px;
                            word-break: break-all;
                        }
                        @media print {
                            button { display: none; }
                        }
                    </style>
                </head>
                <body>
                    <h1><?= htmlspecialchars($school['academy_name']) ?></h1>
                    <h2>Student Registration</h2>
                    <img src="${qrImage.src}" alt="QR Code">
                    <div class="url">${studentRegInput.value}</div>
                    <p style="margin-top:30px; color:#666;">Scan the QR code or visit the URL above to register</p>
                    <button onclick="window.print()" style="padding:10px 30px; font-size:16px; margin-top:20px;">Print</button>
                </body>
                </html>
            `);
            printWindow.document.close();
        }
    </script>
</body>
</html>

