<?php
/**
 * List All Schools
 * Super Admin Only
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$pdo = getDB();

// Get all schools (including inactive)
$stmt = $pdo->query("
    SELECT 
        ar.*,
        a.username as admin_username,
        aw.balance as wallet_balance,
        aw.total_earnings as total_earned,
        COUNT(DISTINCT s.id) as student_count,
        COUNT(DISTINCT p.id) as payment_count,
        SUM(p.amount) as total_revenue
    FROM academy_references ar
    LEFT JOIN admins a ON ar.admin_id = a.id
    LEFT JOIN admin_wallet aw ON aw.admin_id = ar.admin_id
    LEFT JOIN students s ON s.academy_reference = ar.reference_code
    LEFT JOIN payments p ON p.academy_reference = ar.reference_code AND p.status = 'completed'
    GROUP BY ar.id
    ORDER BY ar.created_at DESC
");
$schools = $stmt->fetchAll();

$softDeletedMap = [];
$softStmt = $pdo->query("
    SELECT academy_reference, setting_value 
    FROM tenant_settings 
    WHERE setting_key = 'soft_deleted'
");
foreach ($softStmt as $row) {
    $softDeletedMap[$row['academy_reference']] = $row['setting_value'];
}

$csrfToken = generateCSRFToken();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Schools - Super Admin</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="../dashboard.php">
                    <i class="bi bi-arrow-left me-1"></i>Dashboard
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid py-4">
        <?php if (isset($_GET['status'])): ?>
            <?php
                $status = $_GET['status'];
                $alertClass = 'info';
                $message = '';
                switch ($status) {
                    case 'soft_deleted':
                        $alertClass = 'warning';
                        $message = 'School soft deleted successfully.';
                        break;
                    case 'restored':
                        $alertClass = 'success';
                        $message = 'School restored successfully.';
                        break;
                    case 'deleted':
                        $alertClass = 'success';
                        $message = 'School permanently deleted.';
                        break;
                    case 'invalid_csrf':
                        $alertClass = 'danger';
                        $message = 'Security verification failed. Please try again.';
                        break;
                    case 'invalid_action':
                        $alertClass = 'danger';
                        $message = 'Invalid action requested.';
                        break;
                    case 'not_found':
                        $alertClass = 'danger';
                        $message = 'School not found.';
                        break;
                    case 'error':
                        $alertClass = 'danger';
                        $message = 'Operation failed. Check logs for details.';
                        break;
                }
            ?>
            <?php if ($message): ?>
                <div class="alert alert-<?= $alertClass ?> alert-dismissible fade show">
                    <i class="bi bi-info-circle me-2"></i><?= htmlspecialchars($message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
        <?php endif; ?>

        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2><i class="bi bi-building me-2"></i>All Schools</h2>
            <div class="btn-group">
                <a href="view_urls.php" class="btn btn-outline-primary">
                    <i class="bi bi-link-45deg me-1"></i>View All URLs
                </a>
                <a href="create.php" class="btn btn-primary">
                    <i class="bi bi-plus-circle me-1"></i>Add New School
                </a>
            </div>
        </div>

        <div class="card shadow">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>School</th>
                                <th>Code</th>
                                <th>Admin</th>
                                <th>Students</th>
                                <th>Payments</th>
                                <th>Revenue</th>
                                <th>Wallet</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($schools as $school): ?>
                            <?php 
                                $ref = $school['reference_code'];
                                $softDeleted = ($softDeletedMap[$ref] ?? '0') === '1';
                                $rowClass = '';
                                if ($softDeleted) {
                                    $rowClass = 'table-warning';
                                } elseif (!$school['is_active']) {
                                    $rowClass = 'table-secondary';
                                }
                            ?>
                            <tr class="<?= $rowClass ?>">
                                <td>
                                    <strong><?= htmlspecialchars($school['academy_name']) ?></strong>
                                    <?php if (!empty($school['address'])): ?>
                                        <br><small class="text-muted"><?= htmlspecialchars($school['address']) ?></small>
                                    <?php endif; ?>
                                </td>
                                <td><span class="badge bg-secondary"><?= htmlspecialchars($school['reference_code']) ?></span></td>
                                <td><small><?= htmlspecialchars($school['admin_username'] ?? 'N/A') ?></small></td>
                                <td><?= number_format($school['student_count']) ?></td>
                                <td><?= number_format($school['payment_count']) ?></td>
                                <td>E<?= number_format($school['total_revenue'] ?? 0, 2) ?></td>
                                <td>E<?= number_format($school['wallet_balance'] ?? 0, 2) ?></td>
                                <td>
                                    <?php if ($softDeleted): ?>
                                        <span class="badge bg-dark">Soft Deleted</span>
                                    <?php elseif ($school['is_active']): ?>
                                        <span class="badge bg-success">Active</span>
                                    <?php else: ?>
                                        <span class="badge bg-danger">Suspended</span>
                                    <?php endif; ?>
                                </td>
                                <td><small><?= date('M d, Y', strtotime($school['created_at'])) ?></small></td>
                                <td>
                                    <div class="d-flex flex-wrap gap-1">
                                        <a href="view.php?id=<?= $school['id'] ?>" 
                                           class="btn btn-outline-info btn-sm" 
                                           title="View">
                                            <i class="bi bi-eye"></i>
                                        </a>
                                        <a href="edit.php?id=<?= $school['id'] ?>" 
                                           class="btn btn-outline-secondary btn-sm" 
                                           title="Edit">
                                            <i class="bi bi-pencil"></i>
                                        </a>
                                        <?php if (!$softDeleted): ?>
                                            <form method="post" action="actions.php" class="d-inline"
                                                  onsubmit="return confirm('Soft delete this school? Admin access will be disabled, but data will remain for restore.');">
                                                <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                                                <input type="hidden" name="action" value="soft_delete">
                                                <input type="hidden" name="school_id" value="<?= (int)$school['id'] ?>">
                                                <input type="hidden" name="redirect" value="list">
                                                <button type="submit" class="btn btn-outline-warning btn-sm" title="Soft Delete">
                                                    <i class="bi bi-archive"></i>
                                                </button>
                                            </form>
                                        <?php else: ?>
                                            <form method="post" action="actions.php" class="d-inline"
                                                  onsubmit="return confirm('Restore this school and re-enable access?');">
                                                <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                                                <input type="hidden" name="action" value="restore">
                                                <input type="hidden" name="school_id" value="<?= (int)$school['id'] ?>">
                                                <input type="hidden" name="redirect" value="list">
                                                <button type="submit" class="btn btn-outline-success btn-sm" title="Restore">
                                                    <i class="bi bi-arrow-clockwise"></i>
                                                </button>
                                            </form>
                                            <form method="post" action="actions.php" class="d-inline"
                                                  onsubmit="return confirm('Permanently delete this school? All related data – including students and Moodle enrollments – will be removed. This cannot be undone.');">
                                                <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                                                <input type="hidden" name="action" value="permanent_delete">
                                                <input type="hidden" name="school_id" value="<?= (int)$school['id'] ?>">
                                                <input type="hidden" name="redirect" value="list">
                                                <button type="submit" class="btn btn-outline-danger btn-sm" title="Delete Permanently">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

