<?php
/**
 * Edit School - Complete
 * Super Admin Only
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$schoolId = (int)($_GET['id'] ?? 0);
$pdo = getDB();

// Get school with admin info
$stmt = $pdo->prepare("
    SELECT ar.*, a.id as admin_user_id, a.username as admin_username, a.momo_number
    FROM academy_references ar
    LEFT JOIN admins a ON ar.admin_id = a.id
    WHERE ar.id = ?
");
$stmt->execute([$schoolId]);
$school = $stmt->fetch();

if (!$school) {
    header('Location: list.php');
    exit;
}

// Get tenant settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$school['reference_code']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate inputs
    $schoolName = trim($_POST['school_name'] ?? '');
    $contactEmail = trim($_POST['contact_email'] ?? '');
    $contactPhone = trim($_POST['contact_phone'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $academicYear = trim($_POST['academic_year'] ?? '');
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $adminMomo = trim($_POST['admin_momo'] ?? '');
    $primaryColor = trim($_POST['primary_color'] ?? '#4F46E5');
    $secondaryColor = trim($_POST['secondary_color'] ?? '#10B981');
    $maxStudents = (int)($_POST['max_students'] ?? 500);
    $filteringMode = trim($_POST['filtering_mode'] ?? 'auto');
    
    if (empty($schoolName)) {
        $errors[] = 'School name is required';
    }
    
    if ($maxStudents < 1 || $maxStudents > 10000) {
        $errors[] = 'Maximum students must be between 1 and 10,000';
    }
    
    if (!empty($contactEmail) && !filter_var($contactEmail, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Valid email address required';
    }
    
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Update academy_references
            $stmt = $pdo->prepare("
                UPDATE academy_references 
                SET academy_name = ?,
                    contact_email = ?,
                    contact_phone = ?,
                    address = ?,
                    is_active = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $schoolName,
                $contactEmail,
                $contactPhone,
                $address,
                $isActive,
                $schoolId
            ]);
            
            // Update admin MoMo number if admin exists
            if ($school['admin_user_id']) {
                $stmt = $pdo->prepare("UPDATE admins SET momo_number = ? WHERE id = ?");
                $stmt->execute([$adminMomo, $school['admin_user_id']]);
            }
            
            // Update tenant settings
            $settingsToUpdate = [
                'school_name' => $schoolName,
                'school_email' => $contactEmail,
                'school_phone' => $contactPhone,
                'school_address' => $address,
                'academic_year' => $academicYear,
                'primary_color' => $primaryColor,
                'secondary_color' => $secondaryColor,
                'max_students' => $maxStudents,
                'filtering_mode' => $filteringMode
            ];
            
            foreach ($settingsToUpdate as $key => $value) {
                // Check if setting exists
                $stmt = $pdo->prepare("
                    SELECT COUNT(*) FROM tenant_settings 
                    WHERE academy_reference = ? AND setting_key = ?
                ");
                $stmt->execute([$school['reference_code'], $key]);
                $exists = $stmt->fetchColumn();
                
                if ($exists) {
                    $stmt = $pdo->prepare("
                        UPDATE tenant_settings 
                        SET setting_value = ? 
                        WHERE academy_reference = ? AND setting_key = ?
                    ");
                    $stmt->execute([$value, $school['reference_code'], $key]);
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
                        VALUES (?, ?, ?)
                    ");
                    $stmt->execute([$school['reference_code'], $key, $value]);
                }
            }
            
            $pdo->commit();
            $success = true;
            $successMessage = 'School information updated successfully!';
            
            // Refresh data
            $stmt = $pdo->prepare("
                SELECT ar.*, a.id as admin_user_id, a.username as admin_username, a.momo_number
                FROM academy_references ar
                LEFT JOIN admins a ON ar.admin_id = a.id
                WHERE ar.id = ?
            ");
            $stmt->execute([$schoolId]);
            $school = $stmt->fetch();
            
            $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
            $stmt->execute([$school['reference_code']]);
            $settings = [];
            while ($row = $stmt->fetch()) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = 'Error updating school: ' . $e->getMessage();
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit School - <?= htmlspecialchars($school['academy_name']) ?></title>
    
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="view.php?id=<?= $schoolId ?>">
                    <i class="bi bi-arrow-left me-1"></i>Back to School
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid py-4">
        <div class="row">
            <div class="col-lg-10 mx-auto">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">
                            <i class="bi bi-pencil-square me-2"></i>
                            Edit School: <?= htmlspecialchars($school['academy_name']) ?>
                        </h4>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success alert-dismissible fade show">
                                <i class="bi bi-check-circle me-2"></i>
                                <strong><?= htmlspecialchars($successMessage) ?></strong>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger alert-dismissible fade show">
                                <i class="bi bi-exclamation-triangle me-2"></i>
                                <strong>Please fix the following errors:</strong>
                                <ul class="mb-0 mt-2">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?= htmlspecialchars($error) ?></li>
                                    <?php endforeach; ?>
                                </ul>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <form method="POST" class="needs-validation" novalidate>
                            <!-- School Information -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-building me-2"></i>School Information
                                    </h5>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Reference Code</label>
                                    <input type="text" class="form-control" value="<?= htmlspecialchars($school['reference_code']) ?>" disabled>
                                    <small class="text-muted">Cannot be changed after creation</small>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="school_name" class="form-label">
                                        School Name <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" class="form-control" id="school_name" name="school_name" 
                                           value="<?= htmlspecialchars($_POST['school_name'] ?? $school['academy_name']) ?>" required>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="contact_email" class="form-label">
                                        <i class="bi bi-envelope me-1"></i>Contact Email
                                    </label>
                                    <input type="email" class="form-control" id="contact_email" name="contact_email" 
                                           value="<?= htmlspecialchars($_POST['contact_email'] ?? $school['contact_email'] ?? '') ?>"
                                           placeholder="school@example.com">
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="contact_phone" class="form-label">
                                        <i class="bi bi-telephone me-1"></i>Contact Phone
                                    </label>
                                    <div class="input-group">
                                        <span class="input-group-text">+268</span>
                                        <input type="text" class="form-control" id="contact_phone" name="contact_phone" 
                                               value="<?= htmlspecialchars($_POST['contact_phone'] ?? $school['contact_phone'] ?? '') ?>"
                                               placeholder="7687 3207"
                                               pattern="[0-9\s]{8,15}"
                                               maxlength="15">
                                    </div>
                                </div>

                                <div class="col-md-8 mb-3">
                                    <label for="address" class="form-label">
                                        <i class="bi bi-geo-alt me-1"></i>Address
                                    </label>
                                    <textarea class="form-control" id="address" name="address" rows="2" 
                                              placeholder="School physical address"><?= htmlspecialchars($_POST['address'] ?? $school['address'] ?? '') ?></textarea>
                                </div>

                                <div class="col-md-4 mb-3">
                                    <label for="academic_year" class="form-label">
                                        <i class="bi bi-calendar me-1"></i>Academic Year
                                    </label>
                                    <input type="text" class="form-control" id="academic_year" name="academic_year" 
                                           value="<?= htmlspecialchars($_POST['academic_year'] ?? $settings['academic_year'] ?? date('Y') . '/' . (date('Y') + 1)) ?>"
                                           placeholder="2024/2025">
                                </div>

                                <div class="col-md-4 mb-3">
                                    <label for="max_students" class="form-label">
                                        <i class="bi bi-people me-1"></i>Maximum Students
                                    </label>
                                    <input type="number" class="form-control" id="max_students" name="max_students" 
                                           value="<?= htmlspecialchars($_POST['max_students'] ?? $settings['max_students'] ?? 500) ?>"
                                           min="1" max="10000" placeholder="500">
                                    <small class="text-muted">Student enrollment limit for this school</small>
                                </div>

                                <div class="col-md-4 mb-3">
                                    <label for="filtering_mode" class="form-label">
                                        <i class="bi bi-funnel me-1"></i>Table Filtering Mode
                                    </label>
                                    <select class="form-select" id="filtering_mode" name="filtering_mode">
                                        <option value="auto" <?= ($settings['filtering_mode'] ?? 'auto') === 'auto' ? 'selected' : '' ?>>
                                            Auto (Recommended)
                                        </option>
                                        <option value="client" <?= ($settings['filtering_mode'] ?? 'auto') === 'client' ? 'selected' : '' ?>>
                                            Client-side (Fast, Small Tables)
                                        </option>
                                        <option value="server" <?= ($settings['filtering_mode'] ?? 'auto') === 'server' ? 'selected' : '' ?>>
                                            Server-side (Efficient, Large Tables)
                                        </option>
                                    </select>
                                    <small class="text-muted" id="filtering-mode-help">Auto selects best method based on student count</small>
                                </div>

                                <div class="col-md-12 mb-3">
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                               <?= ($school['is_active'] ?? 1) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="is_active">
                                            <strong>School is Active</strong>
                                            <small class="d-block text-muted">Inactive schools are hidden and cannot be accessed</small>
                                        </label>
                                    </div>
                                </div>
                            </div>

                            <!-- Administrator Information -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-person-badge me-2"></i>Administrator
                                    </h5>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Admin Username</label>
                                    <input type="text" class="form-control" 
                                           value="<?= htmlspecialchars($school['admin_username'] ?? 'Not assigned') ?>" disabled>
                                    <small class="text-muted">Username cannot be changed</small>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="admin_momo" class="form-label">
                                        <i class="bi bi-phone me-1"></i>Admin MoMo Number
                                    </label>
                                    <div class="input-group">
                                        <span class="input-group-text">+268</span>
                                        <input type="text" class="form-control" id="admin_momo" name="admin_momo" 
                                               value="<?= htmlspecialchars($_POST['admin_momo'] ?? $school['momo_number'] ?? '') ?>"
                                               placeholder="7687 3207"
                                               pattern="[0-9\s]{8,15}"
                                               maxlength="15">
                                    </div>
                                    <small class="text-muted">For receiving payments (MTN or Eswatini Mobile)</small>
                                </div>
                            </div>

                            <!-- Branding & Settings -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-palette me-2"></i>Branding & Design
                                    </h5>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="primary_color" class="form-label">
                                        <i class="bi bi-circle-fill me-1"></i>Primary Color
                                    </label>
                                    <div class="input-group">
                                        <input type="color" class="form-control form-control-color" id="primary_color" name="primary_color" 
                                               value="<?= htmlspecialchars($_POST['primary_color'] ?? $settings['primary_color'] ?? '#4F46E5') ?>">
                                        <input type="text" class="form-control" 
                                               value="<?= htmlspecialchars($_POST['primary_color'] ?? $settings['primary_color'] ?? '#4F46E5') ?>"
                                               readonly>
                                    </div>
                                    <small class="text-muted">Main brand color</small>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="secondary_color" class="form-label">
                                        <i class="bi bi-circle-fill me-1"></i>Secondary Color
                                    </label>
                                    <div class="input-group">
                                        <input type="color" class="form-control form-control-color" id="secondary_color" name="secondary_color" 
                                               value="<?= htmlspecialchars($_POST['secondary_color'] ?? $settings['secondary_color'] ?? '#10B981') ?>">
                                        <input type="text" class="form-control" 
                                               value="<?= htmlspecialchars($_POST['secondary_color'] ?? $settings['secondary_color'] ?? '#10B981') ?>"
                                               readonly>
                                    </div>
                                    <small class="text-muted">Accent color</small>
                                </div>

                                <div class="col-12">
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle me-2"></i>
                                        <strong>Logo Upload:</strong> To upload or change the school logo, 
                                        log in to the school admin dashboard and go to Settings → Branding.
                                    </div>
                                </div>
                            </div>

                            <!-- Action Buttons -->
                            <div class="row">
                                <div class="col-12">
                                    <hr>
                                    <div class="d-flex gap-2">
                                        <button type="submit" class="btn btn-primary btn-lg">
                                            <i class="bi bi-save me-1"></i>
                                            Save All Changes
                                        </button>
                                        <a href="view.php?id=<?= $schoolId ?>" class="btn btn-secondary btn-lg">
                                            <i class="bi bi-x-circle me-1"></i>
                                            Cancel
                                        </a>
                                        <a href="list.php" class="btn btn-outline-secondary btn-lg ms-auto">
                                            <i class="bi bi-arrow-left me-1"></i>
                                            Back to List
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Additional Actions Card -->
                <div class="card shadow mt-4">
                    <div class="card-header bg-light">
                        <h5 class="mb-0">
                            <i class="bi bi-gear me-2"></i>Additional Actions
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-4">
                                <a href="?id=<?= $schoolId ?>&action=reset_password" 
                                   class="btn btn-warning w-100 mb-2"
                                   onclick="return confirm('Reset admin password for this school?')">
                                    <i class="bi bi-key me-1"></i>
                                    Reset Admin Password
                                </a>
                            </div>
                            <div class="col-md-4">
                                <a href="../dashboard.php?select_school=<?= urlencode($school['reference_code']) ?>" 
                                   class="btn btn-info w-100 mb-2">
                                    <i class="bi bi-box-arrow-in-right me-1"></i>
                                    Access School Dashboard
                                </a>
                            </div>
                            <div class="col-md-4">
                                <?php if ($school['is_active']): ?>
                                    <a href="list.php?action=suspend&id=<?= $schoolId ?>" 
                                       class="btn btn-danger w-100 mb-2"
                                       onclick="return confirm('Suspend this school?')">
                                        <i class="bi bi-pause-circle me-1"></i>
                                        Suspend School
                                    </a>
                                <?php else: ?>
                                    <a href="list.php?action=activate&id=<?= $schoolId ?>" 
                                       class="btn btn-success w-100 mb-2">
                                        <i class="bi bi-play-circle me-1"></i>
                                        Activate School
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Sync color picker with text input
        document.getElementById('primary_color').addEventListener('input', function(e) {
            this.nextElementSibling.value = e.target.value;
        });
        document.getElementById('secondary_color').addEventListener('input', function(e) {
            this.nextElementSibling.value = e.target.value;
        });

        // Update filtering mode help text
        document.getElementById('filtering_mode').addEventListener('change', function() {
            const helpText = document.getElementById('filtering-mode-help');
            const mode = this.value;
            
            const helpMessages = {
                'auto': 'Auto selects best method based on student count (Client: <2,000 | Server: 3,000+)',
                'client': 'JavaScript filtering - Fast but uses browser memory. Best for small to medium tables (under ~2,000 students)',
                'server': 'PHP/AJAX filtering - Memory efficient, handles large datasets. Best for tables with 3,000+ students or more'
            };
            
            helpText.textContent = helpMessages[mode] || 'Auto selects best method based on student count';
        });
    </script>
</body>
</html>
