<?php
/**
 * Create New School
 * Super Admin Only
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/school_registration_helper.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $schoolName = trim($_POST['school_name'] ?? '');
    $referenceCode = strtoupper(trim($_POST['reference_code'] ?? ''));
    $adminEmail = trim($_POST['admin_email'] ?? '');
    $adminPassword = trim($_POST['admin_password'] ?? '');
    $description = trim($_POST['description'] ?? '');
    
    // Validation
    if (empty($schoolName)) $errors[] = 'School name is required';
    if (empty($referenceCode)) $errors[] = 'Reference code is required';
    if (!preg_match('/^[A-Z0-9]{2,10}$/', $referenceCode)) {
        $errors[] = 'Reference code must be 2-10 uppercase letters/numbers';
    }
    if (empty($adminEmail) || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Valid admin email is required';
    }
    if (empty($adminPassword) || strlen($adminPassword) < 8) {
        $errors[] = 'Admin password must be at least 8 characters';
    }
    
    if (empty($errors)) {
        $pdo = getDB();
        
        try {
            $pdo->beginTransaction();
            
            // Check if reference code already exists
            $stmt = $pdo->prepare("SELECT id FROM academy_references WHERE reference_code = ?");
            $stmt->execute([$referenceCode]);
            if ($stmt->fetch()) {
                throw new Exception('Reference code already exists');
            }
            
            // Create admin user for this school
            $adminUsername = strtolower($referenceCode) . '_admin';
            $adminPasswordHash = password_hash($adminPassword, PASSWORD_DEFAULT);
            
            $stmt = $pdo->prepare("
                INSERT INTO admins (username, password_hash, academy_reference, created_at)
                VALUES (?, ?, ?, NOW())
            ");
            $stmt->execute([$adminUsername, $adminPasswordHash, $referenceCode]);
            $adminId = $pdo->lastInsertId();
            
            // Create academy reference
            $stmt = $pdo->prepare("
                INSERT INTO academy_references 
                (admin_id, academy_name, reference_code, is_active, created_at)
                VALUES (?, ?, ?, 1, NOW())
            ");
            $stmt->execute([$adminId, $schoolName, $referenceCode]);
            
            // Create wallet for admin
            $stmt = $pdo->prepare("
                INSERT INTO admin_wallet (admin_id, balance, total_earnings, withdrawn)
                VALUES (?, 0.00, 0.00, 0.00)
            ");
            $stmt->execute([$adminId]);
            
            // Create default settings
            $defaultSettings = [
                'school_name' => $schoolName,
                'school_email' => $adminEmail,
                'primary_color' => '#4F46E5',
                'secondary_color' => '#10B981',
                'academic_year' => date('Y') . '/' . (date('Y') + 1),
                'email_from_name' => $schoolName,
                'max_students' => 500,  // Default limit
                'filtering_mode' => 'auto',  // Auto-select best filtering method
                'subject_fee' => 350,  // Default subject fee (E350/month)
                'yearly_discount' => 10,  // Default yearly discount (10%)
                'currency_symbol' => 'E'  // Default currency (Emalangeni)
            ];
            
            foreach ($defaultSettings as $key => $value) {
                $stmt = $pdo->prepare("
                    INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$referenceCode, $key, $value]);
            }
            
            $pdo->commit();
            
            // Create tenant folder structure
            require_once '../../includes/tenant_directory_manager.php';
            $tenantManager = new TenantDirectoryManager($pdo);
            $folderResult = $tenantManager->initializeTenant($referenceCode, $schoolName);
            
            if (!$folderResult['success']) {
                error_log("Warning: Could not create tenant folders for $referenceCode: " . $folderResult['message']);
            }
            
            $success = true;
            $successMessage = "School created successfully! Admin username: $adminUsername";
            
            // Generate URLs for the new school
            require_once '../../includes/school_url_generator.php';
            $urlGenerator = new SchoolURLGenerator($referenceCode);
            $schoolURLs = $urlGenerator->getAllURLs();
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = 'Error creating school: ' . $e->getMessage();
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create New School - Super Admin</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="../dashboard.php">
                    <i class="bi bi-arrow-left me-1"></i>Back
                </a>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <div class="row">
            <div class="col-md-8 mx-auto">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">
                            <i class="bi bi-plus-circle me-2"></i>
                            Create New School
                        </h4>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle me-2"></i>
                                <strong><?= htmlspecialchars($successMessage) ?></strong>
                            </div>
                            
                            <!-- School URLs Card -->
                            <div class="card shadow-sm mb-4">
                                <div class="card-header bg-primary text-white">
                                    <h5 class="mb-0">
                                        <i class="bi bi-link-45deg me-2"></i>
                                        School Registration URLs
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <p class="text-muted mb-3">
                                        <i class="bi bi-info-circle me-1"></i>
                                        Share these URLs with students and parents. Each URL is unique to this school.
                                    </p>
                                    
                                    <?php if (isset($schoolURLs)): ?>
                                    
                                    <!-- Student Registration -->
                                    <div class="mb-3">
                                        <label class="form-label fw-bold">
                                            <i class="bi bi-person-plus text-primary me-1"></i>
                                            Student Registration
                                        </label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" 
                                                   value="<?= htmlspecialchars($schoolURLs['student_registration']) ?>" 
                                                   readonly id="url_student_reg">
                                            <button class="btn btn-outline-secondary" onclick="copyURL('url_student_reg')">
                                                <i class="bi bi-clipboard"></i>
                                            </button>
                                            <a href="<?= htmlspecialchars($schoolURLs['student_registration']) ?>" 
                                               target="_blank" class="btn btn-outline-primary">
                                                <i class="bi bi-box-arrow-up-right"></i>
                                            </a>
                                        </div>
                                    </div>
                                    
                                    <!-- Parent Registration -->
                                    <div class="mb-3">
                                        <label class="form-label fw-bold">
                                            <i class="bi bi-people text-success me-1"></i>
                                            Parent Registration
                                        </label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" 
                                                   value="<?= htmlspecialchars($schoolURLs['parent_registration']) ?>" 
                                                   readonly id="url_parent_reg">
                                            <button class="btn btn-outline-secondary" onclick="copyURL('url_parent_reg')">
                                                <i class="bi bi-clipboard"></i>
                                            </button>
                                            <a href="<?= htmlspecialchars($schoolURLs['parent_registration']) ?>" 
                                               target="_blank" class="btn btn-outline-primary">
                                                <i class="bi bi-box-arrow-up-right"></i>
                                            </a>
                                        </div>
                                    </div>
                                    
                                    <!-- Student Login -->
                                    <div class="mb-3">
                                        <label class="form-label fw-bold">
                                            <i class="bi bi-box-arrow-in-right text-info me-1"></i>
                                            Student Login
                                        </label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" 
                                                   value="<?= htmlspecialchars($schoolURLs['student_login']) ?>" 
                                                   readonly id="url_student_login">
                                            <button class="btn btn-outline-secondary" onclick="copyURL('url_student_login')">
                                                <i class="bi bi-clipboard"></i>
                                            </button>
                                            <a href="<?= htmlspecialchars($schoolURLs['student_login']) ?>" 
                                               target="_blank" class="btn btn-outline-primary">
                                                <i class="bi bi-box-arrow-up-right"></i>
                                            </a>
                                        </div>
                                    </div>
                                    
                                    <!-- Parent Login -->
                                    <div class="mb-3">
                                        <label class="form-label fw-bold">
                                            <i class="bi bi-shield-check text-warning me-1"></i>
                                            Parent Login
                                        </label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" 
                                                   value="<?= htmlspecialchars($schoolURLs['parent_login']) ?>" 
                                                   readonly id="url_parent_login">
                                            <button class="btn btn-outline-secondary" onclick="copyURL('url_parent_login')">
                                                <i class="bi bi-clipboard"></i>
                                            </button>
                                            <a href="<?= htmlspecialchars($schoolURLs['parent_login']) ?>" 
                                               target="_blank" class="btn btn-outline-primary">
                                                <i class="bi bi-box-arrow-up-right"></i>
                                            </a>
                                        </div>
                                    </div>
                                    
                                    <!-- QR Code for Student Registration -->
                                    <div class="text-center mt-4 p-3 bg-light rounded">
                                        <h6>Student Registration QR Code</h6>
                                        <img src="<?= $urlGenerator->getQRCodeURL('student_registration') ?>" 
                                             alt="QR Code" class="img-fluid mb-2" style="max-width: 200px;">
                                        <p class="small text-muted mb-0">Students can scan this to register</p>
                                        <button class="btn btn-sm btn-outline-primary mt-2" onclick="printQR()">
                                            <i class="bi bi-printer me-1"></i>Print QR Code
                                        </button>
                                    </div>
                                    
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="d-flex gap-2">
                                <a href="../dashboard.php" class="btn btn-success">
                                    <i class="bi bi-speedometer2 me-2"></i>Go to Dashboard
                                </a>
                                <a href="view.php?code=<?= urlencode($referenceCode) ?>" class="btn btn-primary">
                                    <i class="bi bi-eye me-2"></i>View School Details
                                </a>
                                <a href="create.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-plus-circle me-2"></i>Create Another School
                                </a>
                            </div>
                        <?php else: ?>
                            <?php if (!empty($errors)): ?>
                                <div class="alert alert-danger">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <ul class="mb-0">
                                        <?php foreach ($errors as $error): ?>
                                            <li><?= htmlspecialchars($error) ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <form method="POST" action="">
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="school_name" class="form-label">
                                            School Name <span class="text-danger">*</span>
                                        </label>
                                        <input type="text" class="form-control" id="school_name" 
                                               name="school_name" required
                                               value="<?= htmlspecialchars($_POST['school_name'] ?? '') ?>"
                                               placeholder="e.g., St. Mary's High School">
                                        <small class="text-muted">Full official name of the school</small>
                                    </div>

                                    <div class="col-md-6 mb-3">
                                        <label for="reference_code" class="form-label">
                                            Reference Code <span class="text-danger">*</span>
                                        </label>
                                        <input type="text" class="form-control" id="reference_code" 
                                               name="reference_code" required
                                               value="<?= htmlspecialchars($_POST['reference_code'] ?? '') ?>"
                                               placeholder="e.g., STMARY"
                                               pattern="[A-Z0-9]{2,10}"
                                               style="text-transform: uppercase;">
                                        <small class="text-muted">2-10 characters, uppercase only</small>
                                    </div>
                                </div>

                                <div class="mb-3">
                                    <label for="description" class="form-label">Description</label>
                                    <textarea class="form-control" id="description" name="description" rows="3"
                                              placeholder="Brief description of the school..."><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea>
                                </div>

                                <hr>
                                <h5 class="mb-3">School Administrator Account</h5>

                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="admin_email" class="form-label">
                                            Admin Email <span class="text-danger">*</span>
                                        </label>
                                        <input type="email" class="form-control" id="admin_email" 
                                               name="admin_email" required
                                               value="<?= htmlspecialchars($_POST['admin_email'] ?? '') ?>"
                                               placeholder="admin@school.com">
                                        <small class="text-muted">This will be used for login and notifications</small>
                                    </div>

                                    <div class="col-md-6 mb-3">
                                        <label for="admin_password" class="form-label">
                                            Admin Password <span class="text-danger">*</span>
                                        </label>
                                        <input type="password" class="form-control" id="admin_password" 
                                               name="admin_password" required
                                               minlength="8"
                                               placeholder="Minimum 8 characters">
                                        <small class="text-muted">Minimum 8 characters</small>
                                    </div>
                                </div>

                                <div class="alert alert-info">
                                    <i class="bi bi-info-circle me-2"></i>
                                    <strong>Admin Login Credentials:</strong><br>
                                    Username will be: <code id="preview_username">[reference_code]_admin</code><br>
                                    Password: As entered above
                                </div>

                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-save me-1"></i>
                                        Create School
                                    </button>
                                    <a href="../dashboard.php" class="btn btn-secondary">
                                        <i class="bi bi-x-circle me-1"></i>
                                        Cancel
                                    </a>
                                </div>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Update preview username
        document.getElementById('reference_code').addEventListener('input', function(e) {
            const code = e.target.value.toUpperCase();
            document.getElementById('preview_username').textContent = code ? code.toLowerCase() + '_admin' : '[reference_code]_admin';
        });
        
        // Copy URL to clipboard
        function copyURL(inputId) {
            const input = document.getElementById(inputId);
            input.select();
            input.setSelectionRange(0, 99999); // For mobile
            
            navigator.clipboard.writeText(input.value).then(function() {
                // Visual feedback
                const btn = event.target.closest('button');
                const originalHTML = btn.innerHTML;
                btn.innerHTML = '<i class="bi bi-check"></i> Copied!';
                btn.classList.add('btn-success');
                btn.classList.remove('btn-outline-secondary');
                
                setTimeout(() => {
                    btn.innerHTML = originalHTML;
                    btn.classList.remove('btn-success');
                    btn.classList.add('btn-outline-secondary');
                }, 2000);
            }).catch(function(err) {
                alert('Failed to copy: ' + err);
            });
        }
        
        // Print QR Code
        function printQR() {
            const printWindow = window.open('', '_blank');
            const qrImage = document.querySelector('img[alt="QR Code"]');
            const studentRegInput = document.getElementById('url_student_reg');
            
            printWindow.document.write(`
                <html>
                <head>
                    <title>Student Registration QR Code</title>
                    <style>
                        body { 
                            text-align: center; 
                            font-family: Arial, sans-serif; 
                            padding: 40px;
                        }
                        h1 { color: #333; }
                        img { max-width: 400px; margin: 30px 0; }
                        .url { 
                            font-size: 18px; 
                            background: #f0f0f0; 
                            padding: 15px; 
                            border-radius: 8px;
                            word-break: break-all;
                        }
                        @media print {
                            button { display: none; }
                        }
                    </style>
                </head>
                <body>
                    <h1><?= htmlspecialchars($school['academy_name'] ?? 'School') ?></h1>
                    <h2>Student Registration</h2>
                    <img src="${qrImage.src}" alt="QR Code">
                    <div class="url">${studentRegInput.value}</div>
                    <p style="margin-top:30px; color:#666;">Scan the QR code or visit the URL above to register</p>
                    <button onclick="window.print()" style="padding:10px 30px; font-size:16px; margin-top:20px;">Print</button>
                </body>
                </html>
            `);
            printWindow.document.close();
        }
    </script>
</body>
</html>

