<?php
/**
 * System Health Dashboard
 * Monitor overall system performance and health
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$pdo = getDB();

// Database health check
function checkDatabaseHealth() {
    global $pdo;
    try {
        $pdo->query("SELECT 1");
        return ['status' => 'healthy', 'message' => 'Database connection OK'];
    } catch (Exception $e) {
        return ['status' => 'error', 'message' => $e->getMessage()];
    }
}

// Get system statistics
$stats = [];

// Total records by table
$tables = ['students', 'payments', 'academy_references', 'admins', 'grades'];
foreach ($tables as $table) {
    try {
        $count = $pdo->query("SELECT COUNT(*) FROM $table")->fetchColumn();
        $stats[$table] = $count;
    } catch (Exception $e) {
        $stats[$table] = 'Error';
    }
}

// Recent activity (last 24 hours)
$stmt = $pdo->query("
    SELECT COUNT(*) FROM students 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
");
$new_students_24h = $stmt->fetchColumn();

$stmt = $pdo->query("
    SELECT COUNT(*) FROM payments 
    WHERE payment_date >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
    AND status = 'completed'
");
$new_payments_24h = $stmt->fetchColumn();

// Payment success rate (last 7 days)
$stmt = $pdo->query("
    SELECT 
        status,
        COUNT(*) as count
    FROM payments 
    WHERE payment_date >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    GROUP BY status
");
$payment_stats = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$total_attempts = array_sum($payment_stats);
$successful = $payment_stats['completed'] ?? 0;
$success_rate = $total_attempts > 0 ? ($successful / $total_attempts) * 100 : 0;

// Database size
try {
    $stmt = $pdo->query("
        SELECT 
            table_schema AS 'Database',
            ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) AS 'Size_MB'
        FROM information_schema.TABLES 
        WHERE table_schema = DATABASE()
        GROUP BY table_schema
    ");
    $db_size = $stmt->fetch();
} catch (Exception $e) {
    $db_size = ['Size_MB' => 'N/A'];
}

// Active schools
$active_schools = $pdo->query("SELECT COUNT(*) FROM academy_references WHERE is_active = 1")->fetchColumn();
$total_schools = $pdo->query("SELECT COUNT(*) FROM academy_references")->fetchColumn();

// Database health
$db_health = checkDatabaseHealth();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Health - Super Admin</title>
    
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-shield-check me-2"></i>Super Admin
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="../dashboard.php">
                    <i class="bi bi-arrow-left me-1"></i>Back
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col-12">
                <h1 class="h3">
                    <i class="bi bi-heart-pulse me-2"></i>
                    System Health Monitor
                </h1>
                <p class="text-muted">Real-time system status and performance metrics</p>
            </div>
        </div>

        <!-- Health Status Cards -->
        <div class="row g-4 mb-4">
            <div class="col-md-3">
                <div class="card shadow-sm <?= $db_health['status'] === 'healthy' ? 'bg-success' : 'bg-danger' ?> text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-1 opacity-75">Database</h6>
                                <h4 class="mb-0"><?= ucfirst($db_health['status']) ?></h4>
                            </div>
                            <div>
                                <i class="bi bi-database-check fs-1"></i>
                            </div>
                        </div>
                        <small class="opacity-75"><?= htmlspecialchars($db_health['message']) ?></small>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card shadow-sm bg-info text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-1 opacity-75">Active Schools</h6>
                                <h4 class="mb-0"><?= $active_schools ?> / <?= $total_schools ?></h4>
                            </div>
                            <div>
                                <i class="bi bi-building-check fs-1"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card shadow-sm <?= $success_rate >= 90 ? 'bg-success' : ($success_rate >= 70 ? 'bg-warning' : 'bg-danger') ?> text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-1 opacity-75">Payment Success Rate</h6>
                                <h4 class="mb-0"><?= number_format($success_rate, 1) ?>%</h4>
                            </div>
                            <div>
                                <i class="bi bi-cash-coin fs-1"></i>
                            </div>
                        </div>
                        <small class="opacity-75">Last 7 days</small>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card shadow-sm bg-primary text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-1 opacity-75">Database Size</h6>
                                <h4 class="mb-0"><?= $db_size['Size_MB'] ?? 'N/A' ?> MB</h4>
                            </div>
                            <div>
                                <i class="bi bi-hdd fs-1"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Activity Stats -->
        <div class="row g-4 mb-4">
            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-activity me-2"></i>
                            Recent Activity (24 Hours)
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row text-center">
                            <div class="col-6">
                                <i class="bi bi-person-plus fs-1 text-primary"></i>
                                <h3 class="mt-2"><?= $new_students_24h ?></h3>
                                <p class="text-muted mb-0">New Students</p>
                            </div>
                            <div class="col-6">
                                <i class="bi bi-credit-card fs-1 text-success"></i>
                                <h3 class="mt-2"><?= $new_payments_24h ?></h3>
                                <p class="text-muted mb-0">New Payments</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-database me-2"></i>
                            Database Records
                        </h5>
                    </div>
                    <div class="card-body">
                        <table class="table table-sm mb-0">
                            <tbody>
                                <tr>
                                    <td><i class="bi bi-building me-2 text-primary"></i>Schools</td>
                                    <td class="text-end"><strong><?= number_format($stats['academy_references']) ?></strong></td>
                                </tr>
                                <tr>
                                    <td><i class="bi bi-people me-2 text-success"></i>Students</td>
                                    <td class="text-end"><strong><?= number_format($stats['students']) ?></strong></td>
                                </tr>
                                <tr>
                                    <td><i class="bi bi-credit-card me-2 text-info"></i>Payments</td>
                                    <td class="text-end"><strong><?= number_format($stats['payments']) ?></strong></td>
                                </tr>
                                <tr>
                                    <td><i class="bi bi-person-badge me-2 text-warning"></i>Admins</td>
                                    <td class="text-end"><strong><?= number_format($stats['admins']) ?></strong></td>
                                </tr>
                                <tr>
                                    <td><i class="bi bi-mortarboard me-2 text-secondary"></i>Grades</td>
                                    <td class="text-end"><strong><?= number_format($stats['grades']) ?></strong></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Payment Statistics -->
        <div class="row">
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-graph-up me-2"></i>
                            Payment Statistics (Last 7 Days)
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row text-center">
                            <?php foreach ($payment_stats as $status => $count): ?>
                            <div class="col-md-4 mb-3">
                                <div class="p-3 rounded <?= $status === 'completed' ? 'bg-success' : ($status === 'pending' ? 'bg-warning' : 'bg-danger') ?> bg-opacity-10">
                                    <h3><?= number_format($count) ?></h3>
                                    <p class="mb-0 text-capitalize"><strong><?= htmlspecialchars($status) ?></strong></p>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Refresh Notice -->
        <div class="row mt-4">
            <div class="col-12 text-center">
                <small class="text-muted">
                    <i class="bi bi-arrow-clockwise me-1"></i>
                    Last updated: <?= date('Y-m-d H:i:s') ?> | 
                    <a href="javascript:location.reload()">Refresh Now</a>
                </small>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Auto-refresh every 30 seconds
        setTimeout(function() {
            location.reload();
        }, 30000);
    </script>
</body>
</html>

