<?php
/**
 * Super Admin Dashboard
 * Manage all schools and view cross-school analytics
 */

require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();

// Check if user is super admin
if (!isSuperAdmin()) {
    header('Location: ../admin/dashboard.php');
    exit;
}

$commissionMessages = [];
$commissionErrors = [];
$csrfToken = generateCSRFToken();

// Handle quick school selection BEFORE loading page
if (isset($_GET['select_school'])) {
    $ref = $_GET['select_school'];
    if (setCurrentSchool($ref)) {
        header('Location: ../admin/dashboard.php');
        exit;
    } else {
        $selectionError = "Failed to select school: Invalid or inactive school.";
    }
}

$pdo = getDB();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['commission_form'])) {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $commissionErrors[] = 'Security validation failed. Please refresh and try again.';
    } else {
        $academyRef = strtoupper(trim($_POST['academy_reference'] ?? ''));
        $commissionType = $_POST['commission_type'] ?? 'percentage';
        $commissionValue = trim($_POST['commission_value'] ?? '');

        if ($academyRef === '') {
            $commissionErrors[] = 'Invalid school reference.';
        } elseif (!in_array($commissionType, ['percentage', 'per_student'], true)) {
            $commissionErrors[] = 'Unknown commission type selected.';
        } elseif (!is_numeric($commissionValue) || (float)$commissionValue < 0) {
            $commissionErrors[] = 'Commission value must be a positive number.';
        } else {
            if (setSchoolCommissionSettings($academyRef, $commissionType, (float)$commissionValue)) {
                $commissionMessages[] = "Commission settings updated for {$academyRef}.";
            } else {
                $commissionErrors[] = "Unable to save commission settings for {$academyRef}.";
            }
        }
    }
}

// Get all schools
$schools = getAllSchools();
$schoolStatsMap = [];
$commissionSettings = [];

foreach ($schools as $school) {
    $ref = $school['reference_code'];
    $schoolStatsMap[$ref] = getSchoolStats($ref);
    $commissionSettings[$ref] = getSchoolCommissionSettings($ref);
}

$commissionData = [];
$commissionSummary = [
    'schools' => 0,
    'students' => 0,
    'revenue' => 0.0,
    'earned' => 0.0,
    'projected' => 0.0,
    'payments' => 0,
];

foreach ($schools as $school) {
    $ref = $school['reference_code'];
    $stats = $schoolStatsMap[$ref] ?? [
        'total_students' => 0,
        'total_revenue' => 0,
    ];
    $settings = $commissionSettings[$ref] ?? ['type' => 'percentage', 'value' => 10];
    $earned = $commissionTotals[$ref]['total_commission'] ?? 0.0;
    $totalRevenue = (float)($stats['total_revenue'] ?? 0);
    $totalStudents = (int)($stats['total_students'] ?? 0);
    $projected = $settings['type'] === 'percentage'
        ? ($settings['value'] / 100) * $totalRevenue
        : $settings['value'] * $totalStudents;
    $payments = (int)($commissionTotals[$ref]['total_payments'] ?? 0);

    $commissionData[] = [
        'school' => $school,
        'settings' => $settings,
        'earned' => $earned,
        'projected' => $projected,
        'revenue' => $totalRevenue,
        'students' => $totalStudents,
        'payments' => $payments,
    ];

    $commissionSummary['students'] += $totalStudents;
    $commissionSummary['revenue'] += $totalRevenue;
    $commissionSummary['earned'] += $earned;
    $commissionSummary['projected'] += $projected;
    $commissionSummary['payments'] += $payments;
}

$commissionSummary['schools'] = count($commissionData);

usort($commissionData, function ($a, $b) {
    return $b['earned'] <=> $a['earned'];
});

// Commission totals from payments
$commissionTotals = [];
try {
    $commissionStmt = $pdo->query("
        SELECT 
            academy_reference,
            SUM(platform_fee) as total_commission,
            COUNT(*) as total_payments
        FROM payments
        WHERE status = 'completed'
        GROUP BY academy_reference
    ");
    foreach ($commissionStmt as $row) {
        if (!empty($row['academy_reference'])) {
            $commissionTotals[$row['academy_reference']] = [
                'total_commission' => (float)($row['total_commission'] ?? 0),
                'total_payments' => (int)($row['total_payments'] ?? 0),
            ];
        }
    }
} catch (PDOException $e) {
    error_log("Error loading commission totals: " . $e->getMessage());
}

// Get system-wide statistics
$stmt = $pdo->query("
    SELECT 
        COUNT(*) as total_schools
    FROM academy_references 
    WHERE is_active = 1
");
$total_schools = $stmt->fetchColumn();

$stmt = $pdo->query("
    SELECT COUNT(*) as total_students
    FROM students
");
$total_students = $stmt->fetchColumn();

$stmt = $pdo->query("
    SELECT 
        COUNT(*) as total_payments,
        SUM(amount) as total_revenue,
        SUM(platform_fee) as total_platform_fees
    FROM payments 
    WHERE status = 'completed'
");
$payment_stats = $stmt->fetch();

// Get recent activity
$stmt = $pdo->query("
    SELECT 
        s.full_name,
        s.academy_reference,
        s.created_at
    FROM students s
    ORDER BY s.created_at DESC
    LIMIT 10
");
$recent_students = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Super Admin Dashboard - Skolo-Kine</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .stats-card {
            transition: transform 0.3s;
        }
        .stats-card:hover {
            transform: translateY(-5px);
        }
        .commission-wrapper {
            border: none;
            border-radius: 20px;
            overflow: hidden;
        }
        .commission-header {
            background: linear-gradient(135deg, #312e81, #6366f1);
            color: #ffffff;
            padding: 1.75rem;
            display: flex;
            flex-wrap: wrap;
            justify-content: space-between;
            gap: 1.5rem;
            align-items: flex-start;
        }
        .commission-header h4,
        .commission-header h3 {
            margin: 0;
        }
        .commission-header .label {
            font-size: 0.75rem;
            letter-spacing: 0.08em;
            text-transform: uppercase;
            display: inline-block;
            margin-bottom: 0.25rem;
            opacity: 0.8;
        }
        .commission-summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 1.25rem;
            margin-bottom: 1.75rem;
        }
        .summary-chip {
            padding: 1.1rem 1.25rem;
            border-radius: 14px;
            background: linear-gradient(145deg, #ffffff, #f5f7ff);
            border: 1px solid rgba(99, 102, 241, 0.12);
            box-shadow: 0 12px 24px rgba(15, 23, 42, 0.05);
        }
        .summary-chip .label {
            font-size: 0.75rem;
            text-transform: uppercase;
            color: #6366f1;
            letter-spacing: 0.08em;
            display: block;
            margin-bottom: 0.35rem;
        }
        .summary-chip .value {
            font-size: 1.5rem;
            font-weight: 600;
            color: #0f172a;
        }
        .commission-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1.5rem;
        }
        .commission-item {
            background: #ffffff;
            border-radius: 18px;
            padding: 1.5rem;
            border: 1px solid rgba(226, 232, 240, 0.9);
            transition: all 0.25s ease;
            position: relative;
            overflow: hidden;
        }
        .commission-item::after {
            content: '';
            position: absolute;
            inset: 0;
            border-radius: 20px;
            padding: 1px;
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.25), rgba(129, 140, 248, 0.1));
            mask: linear-gradient(#fff 0 0) content-box, linear-gradient(#fff 0 0);
            -webkit-mask: linear-gradient(#fff 0 0) content-box, linear-gradient(#fff 0 0);
            -webkit-mask-composite: xor;
            mask-composite: exclude;
            pointer-events: none;
        }
        .commission-item:hover {
            transform: translateY(-6px);
            box-shadow: 0 18px 30px rgba(15, 23, 42, 0.12);
        }
        .commission-item h6 {
            font-weight: 600;
            color: #1e293b;
        }
        .commission-item .earned-value {
            font-size: 1.35rem;
            font-weight: 700;
            color: #15803d;
        }
        .commission-item .commission-setting {
            display: inline-block;
            background: rgba(99, 102, 241, 0.12);
            color: #3730a3;
            padding: 0.25rem 0.75rem;
            border-radius: 999px;
            font-size: 0.75rem;
            margin-top: 0.4rem;
        }
        .commission-stats {
            display: flex;
            flex-wrap: wrap;
            gap: 0.75rem;
            margin-top: 1.25rem;
        }
        .commission-stats .stat {
            flex: 1 1 110px;
            background: #f8fafc;
            border-radius: 12px;
            padding: 0.75rem 0.85rem;
        }
        .commission-stats .stat small {
            display: block;
            font-size: 0.7rem;
            letter-spacing: 0.06em;
            text-transform: uppercase;
            color: #64748b;
        }
        .commission-stats .stat span {
            display: block;
            font-weight: 600;
            font-size: 1rem;
            color: #0f172a;
        }
        .commission-projection {
            font-size: 0.85rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.35rem;
        }
        .commission-form {
            margin-top: 1.25rem;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
            gap: 0.75rem;
            align-items: end;
        }
        .commission-form .form-select,
        .commission-form .form-control {
            min-height: 44px;
            border-radius: 10px;
        }
        .commission-form .input-group-text {
            border-radius: 10px;
        }
        .commission-item .payments-meta {
            margin-top: 0.75rem;
            font-size: 0.8rem;
            color: #64748b;
        }
        @media (max-width: 767.98px) {
            .commission-header {
                padding: 1.5rem;
            }
            .commission-item {
                padding: 1.25rem;
            }
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="dashboard.php">
                <i class="bi bi-shield-check me-2"></i>
                <strong>Super Admin</strong>
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="schools/list.php">
                    <i class="bi bi-building me-1"></i>Schools
                </a>
                <a class="nav-link" href="schools/view_urls.php">
                    <i class="bi bi-link-45deg me-1"></i>URLs
                </a>
                <a class="nav-link" href="storage/overview.php">
                    <i class="bi bi-hdd-stack me-1"></i>Storage
                </a>
                <a class="nav-link" href="../admin/logout.php">
                    <i class="bi bi-box-arrow-right me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col-12">
                <h1 class="h3">
                    <i class="bi bi-speedometer2 me-2"></i>
                    Super Admin Dashboard
                </h1>
                <p class="text-muted">System-wide overview and management</p>
            </div>
        </div>

        <?php if (isset($selectionError)): ?>
            <div class="row mb-4">
                <div class="col-12">
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-exclamation-triangle me-2"></i>
                        <strong><?= htmlspecialchars($selectionError) ?></strong>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <?php if (!empty($commissionMessages)): ?>
            <div class="row mb-3">
                <div class="col-12">
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="bi bi-check-circle me-2"></i>
                        <strong><?= htmlspecialchars(implode(' ', $commissionMessages)) ?></strong>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <?php if (!empty($commissionErrors)): ?>
            <div class="row mb-3">
                <div class="col-12">
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-x-circle me-2"></i>
                        <strong><?= htmlspecialchars(implode(' ', $commissionErrors)) ?></strong>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="row g-4 mb-4">
            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Active Schools</h6>
                                <h2 class="mb-0"><?= $total_schools ?></h2>
                            </div>
                            <div class="bg-primary bg-opacity-10 p-3 rounded">
                                <i class="bi bi-building fs-2 text-primary"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Total Students</h6>
                                <h2 class="mb-0"><?= number_format($total_students) ?></h2>
                            </div>
                            <div class="bg-success bg-opacity-10 p-3 rounded">
                                <i class="bi bi-people fs-2 text-success"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Total Revenue</h6>
                                <h2 class="mb-0">E<?= number_format($payment_stats['total_revenue'] ?? 0, 2) ?></h2>
                            </div>
                            <div class="bg-info bg-opacity-10 p-3 rounded">
                                <i class="bi bi-currency-exchange fs-2 text-info"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Platform Fees</h6>
                                <h2 class="mb-0">E<?= number_format($payment_stats['total_platform_fees'] ?? 0, 2) ?></h2>
                            </div>
                            <div class="bg-warning bg-opacity-10 p-3 rounded">
                                <i class="bi bi-wallet2 fs-2 text-warning"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Commission Fees Card -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card shadow-sm commission-wrapper">
                    <div class="commission-header">
                        <div>
                            <span class="label">Commission Management</span>
                            <h4>Commission Fees</h4>
                            <p class="mb-0">Configure per-school commission and keep track of what the platform earns from every tenant.</p>
                        </div>
                        <div class="commission-header-summary text-end">
                            <span class="label">Total Earned</span>
                            <h3>E<?= number_format($commissionSummary['earned'], 2) ?></h3>
                            <small class="d-block text-white-50">
                                Projected: E<?= number_format($commissionSummary['projected'], 2) ?>
                            </small>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if (empty($commissionData)): ?>
                            <div class="text-center text-muted py-5">
                                <i class="bi bi-cash-stack fs-2 d-block mb-2"></i>
                                No schools available yet. Add a school to start tracking commissions.
                            </div>
                        <?php else: ?>
                            <div class="commission-summary">
                                <div class="summary-chip">
                                    <span class="label">Active Schools</span>
                                    <h5 class="value"><?= number_format($commissionSummary['schools']) ?></h5>
                                    <small class="text-muted">Configured for commission tracking</small>
                                </div>
                                <div class="summary-chip">
                                    <span class="label">Students Covered</span>
                                    <h5 class="value"><?= number_format($commissionSummary['students']) ?></h5>
                                    <small class="text-muted">Across every tenant</small>
                                </div>
                                <div class="summary-chip">
                                    <span class="label">Revenue Processed</span>
                                    <h5 class="value">E<?= number_format($commissionSummary['revenue'], 2) ?></h5>
                                    <small class="text-muted">Completed payments only</small>
                                </div>
                                <div class="summary-chip">
                                    <span class="label">Payments Logged</span>
                                    <h5 class="value"><?= number_format($commissionSummary['payments']) ?></h5>
                                    <small class="text-muted">With platform fees recorded</small>
                                </div>
                            </div>

                            <div class="commission-grid">
                                <?php foreach ($commissionData as $row):
                                    $school = $row['school'];
                                    $settings = $row['settings'];
                                    $isPercentage = $settings['type'] === 'percentage';
                                    $difference = $row['projected'] - $row['earned'];
                                    $differenceClass = 'text-muted';
                                    $differenceLabel = 'On target';
                                    $differenceIcon = 'bi-slash-circle';
                                    if (abs($difference) >= 0.01) {
                                        if ($difference > 0) {
                                            $differenceClass = 'text-success';
                                            $differenceLabel = 'Projected upside';
                                            $differenceIcon = 'bi-graph-up-arrow';
                                        } else {
                                            $differenceClass = 'text-danger';
                                            $differenceLabel = 'Over target';
                                            $differenceIcon = 'bi-graph-down-arrow';
                                        }
                                    }
                                ?>
                                <div class="commission-item shadow-sm">
                                    <div class="d-flex justify-content-between align-items-start gap-3">
                                        <div>
                                            <h6 class="mb-1"><?= htmlspecialchars($school['academy_name']) ?></h6>
                                            <span class="badge bg-secondary"><?= htmlspecialchars($school['reference_code']) ?></span>
                                            <div class="commission-setting">
                                                <?= $isPercentage
                                                    ? number_format($settings['value'], 2) . '% of revenue'
                                                    : 'E' . number_format($settings['value'], 2) . ' per student' ?>
                                            </div>
                                        </div>
                                        <div class="text-end">
                                            <span class="earned-value">E<?= number_format($row['earned'], 2) ?></span>
                                            <div class="small text-muted">Earned to date</div>
                                        </div>
                                    </div>

                                    <div class="commission-stats">
                                        <div class="stat">
                                            <small>Students</small>
                                            <span><?= number_format($row['students']) ?></span>
                                        </div>
                                        <div class="stat">
                                            <small>Total revenue</small>
                                            <span>E<?= number_format($row['revenue'], 2) ?></span>
                                        </div>
                                        <div class="stat">
                                            <small>Projected commission</small>
                                            <span>E<?= number_format($row['projected'], 2) ?></span>
                                        </div>
                                    </div>

                                    <div class="commission-projection mt-3 <?= $differenceClass ?>">
                                    <i class="bi <?= $differenceIcon ?>"></i>
                                    <?= $differenceLabel ?>
                                    <?php if (abs($difference) >= 0.01): ?>
                                        : E<?= number_format(abs($difference), 2) ?>
                                    <?php else: ?>
                                        (E<?= number_format($row['projected'], 2) ?>)
                                    <?php endif; ?>
                                    </div>

                                    <form method="post" class="commission-form">
                                        <input type="hidden" name="commission_form" value="1">
                                        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                                        <input type="hidden" name="academy_reference" value="<?= htmlspecialchars($school['reference_code']) ?>">

                                        <div>
                                            <label class="form-label small text-muted mb-1">Commission Type</label>
                                            <select name="commission_type" class="form-select">
                                                <option value="percentage" <?= $isPercentage ? 'selected' : '' ?>>
                                                    Percentage of revenue (%)
                                                </option>
                                                <option value="per_student" <?= !$isPercentage ? 'selected' : '' ?>>
                                                    Per student (E)
                                                </option>
                                            </select>
                                        </div>
                                        <div>
                                            <label class="form-label small text-muted mb-1">Value</label>
                                            <div class="input-group">
                                                <input type="number" step="0.01" min="0" class="form-control"
                                                       name="commission_value"
                                                       value="<?= htmlspecialchars(number_format($settings['value'], 2, '.', '')) ?>">
                                                <span class="input-group-text">
                                                    <?= $isPercentage ? '%' : 'E' ?>
                                                </span>
                                            </div>
                                        </div>
                                        <div>
                                            <label class="form-label small text-muted mb-1">Save</label>
                                            <button type="submit" class="btn btn-primary w-100">
                                                <i class="bi bi-save me-1"></i>Update
                                            </button>
                                        </div>
                                    </form>

                                    <div class="payments-meta">
                                        <?php if ($row['payments'] > 0): ?>
                                            <?= number_format($row['payments']) ?> completed payments with platform fees.
                                        <?php else: ?>
                                            No platform fee payments recorded yet.
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>

                            <p class="text-muted small mt-4 mb-0 commission-footnote">
                                <i class="bi bi-info-circle me-1"></i>
                                "Earned" reflects actual platform fees captured in completed payments. "Projected commission" re-calculates in real time based on your current settings and student counts.
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Schools Overview -->
        <div class="row">
            <div class="col-12 mb-4">
                <div class="card shadow-sm">
                    <div class="card-header bg-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">
                            <i class="bi bi-building me-2"></i>
                            Schools Overview
                        </h5>
                        <div class="btn-group">
                            <a href="schools/view_urls.php" class="btn btn-outline-success btn-sm">
                                <i class="bi bi-link-45deg me-1"></i>
                                View URLs
                            </a>
                            <a href="schools/create.php" class="btn btn-primary btn-sm">
                                <i class="bi bi-plus-circle me-1"></i>
                                Add New School
                            </a>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>School Name</th>
                                        <th>Reference Code</th>
                                        <th>Students</th>
                                        <th>Revenue</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($schools as $school): 
                                        $stats = $schoolStatsMap[$school['reference_code']] ?? [
                                            'total_students' => 0,
                                            'total_revenue' => 0,
                                            'academy_reference' => $school['reference_code']
                                        ];
                                    ?>
                                    <tr>
                                        <td>
                                            <strong><?= htmlspecialchars($school['academy_name']) ?></strong>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary"><?= htmlspecialchars($school['reference_code']) ?></span>
                                        </td>
                                        <td><?= number_format($stats['total_students']) ?></td>
                                        <td>E<?= number_format($stats['total_revenue'], 2) ?></td>
                                        <td>
                                            <?php if ($school['is_active']): ?>
                                                <span class="badge bg-success">Active</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">Suspended</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="?select_school=<?= urlencode($school['reference_code']) ?>" 
                                                   class="btn btn-outline-primary" 
                                                   title="Access School">
                                                    <i class="bi bi-box-arrow-in-right"></i>
                                                </a>
                                                <a href="schools/edit.php?id=<?= $school['id'] ?>" 
                                                   class="btn btn-outline-secondary" 
                                                   title="Edit">
                                                    <i class="bi bi-pencil"></i>
                                                </a>
                                                <a href="schools/view.php?id=<?= $school['id'] ?>" 
                                                   class="btn btn-outline-info" 
                                                   title="View Details">
                                                    <i class="bi bi-eye"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Activity -->
        <div class="row">
            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-clock-history me-2"></i>
                            Recent Student Registrations
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="list-group list-group-flush">
                            <?php foreach ($recent_students as $s): ?>
                            <div class="list-group-item px-0">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <strong><?= htmlspecialchars($s['full_name']) ?></strong>
                                        <br>
                                        <small class="text-muted">
                                            <span class="badge bg-info"><?= htmlspecialchars($s['academy_reference']) ?></span>
                                        </small>
                                    </div>
                                    <div>
                                        <small class="text-muted"><?= date('M d, Y', strtotime($s['created_at'])) ?></small>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-gear me-2"></i>
                            Quick Actions
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="d-grid gap-2">
                            <a href="schools/create.php" class="btn btn-primary btn-lg">
                                <i class="bi bi-plus-circle me-2"></i>
                                Create New School
                            </a>
                            <a href="schools/view_urls.php" class="btn btn-outline-success btn-lg">
                                <i class="bi bi-link-45deg me-2"></i>
                                View All School URLs
                            </a>
                            <a href="schools/list.php" class="btn btn-outline-secondary btn-lg">
                                <i class="bi bi-building me-2"></i>
                                Manage All Schools
                            </a>
                            <a href="admins/list.php" class="btn btn-outline-primary btn-lg">
                                <i class="bi bi-people me-2"></i>
                                Manage School Admins
                            </a>
                            <a href="reports/platform_fees.php" class="btn btn-outline-info btn-lg">
                                <i class="bi bi-graph-up me-2"></i>
                                View Platform Earnings
                            </a>
                            <a href="reports/system_health.php" class="btn btn-outline-success btn-lg">
                                <i class="bi bi-heart-pulse me-2"></i>
                                System Health
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

