<?php
/**
 * Student Profile
 * View and update profile information
 */

session_start();
require_once '../includes/functions.php';
requireStudentLogin();

$student = getCurrentStudent();
$pdo = getDB();

// Get student details
$stmt = $pdo->prepare("SELECT s.*, g.name as grade_name FROM students s LEFT JOIN grades g ON s.grade_id = g.id WHERE s.id = ?");
$stmt->execute([$student['id']]);
$student_details = $stmt->fetch();

// Get school
$stmt = $pdo->prepare("SELECT academy_name FROM academy_references WHERE reference_code = ?");
$stmt->execute([$student['academy_reference']]);
$school = ['academy_name' => $stmt->fetchColumn() ?: 'School'];

// Get settings
$stmt = $pdo->prepare("SELECT setting_value FROM tenant_settings WHERE academy_reference = ? AND setting_key = 'moodle_url'");
$stmt->execute([$student['academy_reference']]);
$moodle_url = $stmt->fetchColumn();

$error = '';
$success = false;

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        $error = 'All password fields are required.';
    } elseif ($new_password !== $confirm_password) {
        $error = 'New passwords do not match.';
    } elseif (strlen($new_password) < 8) {
        $error = 'New password must be at least 8 characters.';
    } elseif (!password_verify($current_password, $student_details['password_hash'])) {
        $error = 'Current password is incorrect.';
    } else {
        // Update password
        $new_hash = password_hash($new_password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE students SET password_hash = ? WHERE id = ?");
        if ($stmt->execute([$new_hash, $student['id']])) {
            $success = true;
        } else {
            $error = 'Failed to update password.';
        }
    }
}

$pageTitle = "My Profile - Student Portal";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?></title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
        }
        .nav-link {
            color: rgba(255,255,255,0.8);
            transition: all 0.3s;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255,255,255,0.1);
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0 sidebar text-white">
                <div class="p-3">
                    <h5 class="mb-0">
                        <i class="bi bi-mortarboard-fill me-2"></i>
                        Student Portal
                    </h5>
                    <small class="text-white-50"><?= htmlspecialchars($school['academy_name']) ?></small>
                </div>
                
                <nav class="nav flex-column px-2">
                    <a class="nav-link rounded" href="dashboard.php">
                        <i class="bi bi-house me-2"></i>Dashboard
                    </a>
                    <a class="nav-link rounded" href="subjects.php">
                        <i class="bi bi-book me-2"></i>My Subjects
                    </a>
                    <a class="nav-link rounded" href="make_payment.php">
                        <i class="bi bi-credit-card me-2"></i>Make Payment
                    </a>
                    <a class="nav-link rounded" href="payment_history.php">
                        <i class="bi bi-receipt me-2"></i>Payment History
                    </a>
                    <?php if ($moodle_url): ?>
                    <a class="nav-link rounded" href="<?= htmlspecialchars($moodle_url) ?>" target="_blank">
                        <i class="bi bi-mortarboard me-2"></i>Moodle Portal
                        <i class="bi bi-box-arrow-up-right ms-1 small"></i>
                    </a>
                    <?php endif; ?>
                    <a class="nav-link active rounded" href="profile.php">
                        <i class="bi bi-person me-2"></i>My Profile
                    </a>
                    <hr class="text-white-50">
                    <a class="nav-link rounded" href="logout.php">
                        <i class="bi bi-box-arrow-right me-2"></i>Logout
                    </a>
                </nav>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 px-4 py-3">
                <h2 class="mb-4"><i class="bi bi-person me-2"></i>My Profile</h2>

                <?php if ($success): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="bi bi-check-circle me-2"></i>
                    Password changed successfully!
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="bi bi-exclamation-triangle me-2"></i>
                    <?= htmlspecialchars($error) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <div class="row">
                    <!-- Profile Information -->
                    <div class="col-md-8">
                        <div class="card mb-4">
                            <div class="card-header bg-primary text-white">
                                <h5 class="mb-0"><i class="bi bi-person-badge me-2"></i>Personal Information</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="text-muted small">Full Name</label>
                                        <div class="fs-5"><strong><?= htmlspecialchars($student_details['full_name'] ?? $student['name']) ?></strong></div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label class="text-muted small">Username</label>
                                        <div><code class="fs-6"><?= htmlspecialchars($student['username']) ?></code></div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label class="text-muted small">Email</label>
                                        <div>
                                            <?= htmlspecialchars($student_details['email']) ?>
                                            <?php if ($student_details['email_verified']): ?>
                                                <span class="badge bg-success ms-2"><i class="bi bi-check-circle"></i> Verified</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label class="text-muted small">Grade</label>
                                        <div><span class="badge bg-info fs-6"><?= htmlspecialchars($student_details['grade_name']) ?></span></div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label class="text-muted small">Phone</label>
                                        <div><?= htmlspecialchars($student_details['phone'] ?? 'Not provided') ?></div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label class="text-muted small">Account Status</label>
                                        <div>
                                            <?php
                                            $status = $student_details['status'] ?? 'active';
                                            $badge = $status === 'active' ? 'success' : 'warning';
                                            ?>
                                            <span class="badge bg-<?= $badge ?> fs-6"><?= ucfirst($status) ?></span>
                                        </div>
                                    </div>
                                </div>

                                <div class="alert alert-info mt-3 mb-0">
                                    <i class="bi bi-info-circle me-2"></i>
                                    To update your personal information, please contact your school administration.
                                </div>
                            </div>
                        </div>

                        <!-- Change Password -->
                        <div class="card">
                            <div class="card-header bg-warning">
                                <h5 class="mb-0"><i class="bi bi-key me-2"></i>Change Password</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST">
                                    <div class="mb-3">
                                        <label for="current_password" class="form-label">Current Password</label>
                                        <input type="password" class="form-control" id="current_password" 
                                               name="current_password" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="new_password" class="form-label">New Password</label>
                                        <input type="password" class="form-control" id="new_password" 
                                               name="new_password" required minlength="8">
                                        <small class="text-muted">Minimum 8 characters. This will also update your Moodle password.</small>
                                    </div>
                                    <div class="mb-3">
                                        <label for="confirm_password" class="form-label">Confirm New Password</label>
                                        <input type="password" class="form-control" id="confirm_password" 
                                               name="confirm_password" required>
                                    </div>
                                    <button type="submit" name="change_password" class="btn btn-warning">
                                        <i class="bi bi-key me-2"></i>Change Password
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>

                    <!-- Sidebar Info -->
                    <div class="col-md-4">
                        <div class="card mb-4">
                            <div class="card-header bg-info text-white">
                                <h6 class="mb-0"><i class="bi bi-mortarboard me-2"></i>Moodle Access</h6>
                            </div>
                            <div class="card-body">
                                <p class="small mb-2">
                                    <strong>Username:</strong><br>
                                    <code><?= htmlspecialchars($student['username']) ?></code>
                                </p>
                                <p class="small mb-3">Use the same credentials for Moodle login.</p>
                                <?php if ($moodle_url): ?>
                                <a href="<?= htmlspecialchars($moodle_url) ?>" target="_blank" class="btn btn-info btn-sm w-100">
                                    <i class="bi bi-box-arrow-up-right me-2"></i>Open Moodle
                                </a>
                                <?php else: ?>
                                <p class="text-muted small mb-0">Moodle URL not configured</p>
                                <?php endif; ?>
                            </div>
                        </div>

                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="bi bi-info-circle me-2"></i>Need Help?</h6>
                            </div>
                            <div class="card-body">
                                <p class="small mb-2">If you have any questions or need assistance:</p>
                                <ul class="small mb-0">
                                    <li>Contact your school administration</li>
                                    <li>Reach out to your teachers</li>
                                    <li>Check the help section in Moodle</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

