<?php
/**
 * Student Payment History
 * View all payment transactions
 */

session_start();
require_once '../includes/functions.php';
requireStudentLogin();

$student = getCurrentStudent();
$pdo = getDB();

// Get school settings
$stmt = $pdo->prepare("SELECT setting_value FROM tenant_settings WHERE academy_reference = ? AND setting_key IN ('currency_symbol', 'moodle_url')");
$stmt->execute([$student['academy_reference']]);
$results = $stmt->fetchAll();
$settings = [];
foreach ($results as $row) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$currency_symbol = $settings['currency_symbol'] ?? 'E';
$moodle_url = $settings['moodle_url'] ?? null;

// Get school name
$stmt = $pdo->prepare("SELECT academy_name FROM academy_references WHERE reference_code = ?");
$stmt->execute([$student['academy_reference']]);
$school = ['academy_name' => $stmt->fetchColumn() ?: 'School'];

// Get all payments
$stmt = $pdo->prepare("
    SELECT 
        p.*,
        CASE 
            WHEN p.status = 'completed' THEN 'success'
            WHEN p.status = 'pending' THEN 'warning'
            ELSE 'danger'
        END as badge_class
    FROM payments p
    WHERE p.student_id = ?
    ORDER BY p.payment_date DESC
");
$stmt->execute([$student['id']]);
$payments = $stmt->fetchAll();

// Calculate totals
$total_paid = 0;
$total_pending = 0;
foreach ($payments as $payment) {
    if ($payment['status'] === 'completed') {
        $total_paid += $payment['amount'];
    } elseif ($payment['status'] === 'pending') {
        $total_pending += $payment['amount'];
    }
}

$pageTitle = "Payment History - Student Portal";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?></title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
        }
        .nav-link {
            color: rgba(255,255,255,0.8);
            transition: all 0.3s;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255,255,255,0.1);
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0 sidebar text-white">
                <div class="p-3">
                    <h5 class="mb-0">
                        <i class="bi bi-mortarboard-fill me-2"></i>
                        Student Portal
                    </h5>
                    <small class="text-white-50"><?= htmlspecialchars($school['academy_name']) ?></small>
                </div>
                
                <nav class="nav flex-column px-2">
                    <a class="nav-link rounded" href="dashboard.php">
                        <i class="bi bi-house me-2"></i>Dashboard
                    </a>
                    <a class="nav-link rounded" href="subjects.php">
                        <i class="bi bi-book me-2"></i>My Subjects
                    </a>
                    <a class="nav-link rounded" href="make_payment.php">
                        <i class="bi bi-credit-card me-2"></i>Make Payment
                    </a>
                    <a class="nav-link active rounded" href="payment_history.php">
                        <i class="bi bi-receipt me-2"></i>Payment History
                    </a>
                    <?php if ($moodle_url): ?>
                    <a class="nav-link rounded" href="<?= htmlspecialchars($moodle_url) ?>" target="_blank">
                        <i class="bi bi-mortarboard me-2"></i>Moodle Portal
                        <i class="bi bi-box-arrow-up-right ms-1 small"></i>
                    </a>
                    <?php endif; ?>
                    <a class="nav-link rounded" href="profile.php">
                        <i class="bi bi-person me-2"></i>My Profile
                    </a>
                    <hr class="text-white-50">
                    <a class="nav-link rounded" href="logout.php">
                        <i class="bi bi-box-arrow-right me-2"></i>Logout
                    </a>
                </nav>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 px-4 py-3">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="bi bi-receipt me-2"></i>Payment History</h2>
                    <a href="make_payment.php" class="btn btn-primary">
                        <i class="bi bi-plus-circle me-2"></i>Make Payment
                    </a>
                </div>

                <!-- Payment Summary -->
                <div class="row g-3 mb-4">
                    <div class="col-md-6">
                        <div class="card bg-success text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <p class="mb-1 opacity-75">Total Paid</p>
                                        <h3 class="mb-0"><?= $currency_symbol ?><?= number_format($total_paid, 2) ?></h3>
                                    </div>
                                    <div>
                                        <i class="bi bi-check-circle-fill" style="font-size: 3rem; opacity: 0.5;"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card bg-warning text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <p class="mb-1 opacity-75">Pending Payments</p>
                                        <h3 class="mb-0"><?= $currency_symbol ?><?= number_format($total_pending, 2) ?></h3>
                                    </div>
                                    <div>
                                        <i class="bi bi-clock-fill" style="font-size: 3rem; opacity: 0.5;"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Payment History Table -->
                <div class="card">
                    <div class="card-header bg-white">
                        <h5 class="mb-0"><i class="bi bi-clock-history me-2"></i>Transaction History</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($payments)): ?>
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle me-2"></i>
                                No payment transactions yet. <a href="make_payment.php">Make your first payment</a>.
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Reference</th>
                                            <th>Amount</th>
                                            <th>Method</th>
                                            <th>MoMo Number</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($payments as $payment): ?>
                                        <tr>
                                            <td><?= date('M d, Y H:i', strtotime($payment['payment_date'])) ?></td>
                                            <td><code><?= htmlspecialchars($payment['payment_reference']) ?></code></td>
                                            <td><strong><?= $currency_symbol ?><?= number_format($payment['amount'], 2) ?></strong></td>
                                            <td>
                                                <i class="bi bi-phone me-1"></i>
                                                <?= ucfirst($payment['payment_method'] ?? 'MoMo') ?>
                                            </td>
                                            <td><?= htmlspecialchars($payment['momo_number'] ?? 'N/A') ?></td>
                                            <td>
                                                <span class="badge bg-<?= $payment['badge_class'] ?>">
                                                    <?= ucfirst($payment['status']) ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

