<?php
/**
 * Student Payment Confirmation
 * Confirm payment before processing
 */

session_start();
require_once '../includes/functions.php';
requireStudentLogin();

if (!isset($_SESSION['student_payment_data'])) {
    header('Location: make_payment.php');
    exit;
}

$payment_data = $_SESSION['student_payment_data'];
$student = getCurrentStudent();
$pdo = getDB();

// Get school settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$student['academy_reference']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$currency_symbol = $settings['currency_symbol'] ?? 'E';

// Get subject details
$placeholders = str_repeat('?,', count($payment_data['subjects']) - 1) . '?';
$stmt = $pdo->prepare("SELECT id, name, code FROM subjects WHERE id IN ($placeholders)");
$stmt->execute($payment_data['subjects']);
$selected_subjects = $stmt->fetchAll();

// Process payment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_payment'])) {
    try {
        $pdo->beginTransaction();
        
        $payment_reference = 'STU' . time() . rand(1000, 9999);
        
        // Create payment record
        $stmt = $pdo->prepare("
            INSERT INTO payments 
            (student_id, academy_reference, amount, payment_reference, status, payment_method, momo_number, payment_date)
            VALUES (?, ?, ?, ?, 'completed', 'mtn_momo', ?, NOW())
        ");
        $stmt->execute([
            $student['id'],
            $student['academy_reference'],
            $payment_data['total_amount'],
            $payment_reference,
            $payment_data['momo_number']
        ]);
        
        // Enroll in subjects (if not already enrolled)
        $stmt = $pdo->prepare("
            INSERT IGNORE INTO student_subject (student_id, subject_id)
            VALUES (?, ?)
        ");
        foreach ($payment_data['subjects'] as $subject_id) {
            $stmt->execute([$student['id'], $subject_id]);
        }
        
        $pdo->commit();
        
        // Clear payment data
        unset($_SESSION['student_payment_data']);
        
        // Set success message
        $_SESSION['payment_success'] = [
            'reference' => $payment_reference,
            'amount' => $payment_data['total_amount'],
            'subjects' => count($payment_data['subjects'])
        ];
        
        header('Location: payment_success.php');
        exit;
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $error = 'Payment processing failed. Please try again.';
        error_log("Student payment error: " . $e->getMessage());
    }
}

$pageTitle = "Confirm Payment - Student Portal";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?></title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card shadow">
                    <div class="card-header bg-warning text-white">
                        <h4 class="mb-0"><i class="bi bi-exclamation-triangle me-2"></i>Confirm Payment</h4>
                    </div>
                    <div class="card-body">
                        <?php if (isset($error)): ?>
                        <div class="alert alert-danger">
                            <i class="bi bi-exclamation-triangle me-2"></i>
                            <?= htmlspecialchars($error) ?>
                        </div>
                        <?php endif; ?>
                        
                        <h5 class="mb-3">Payment Details</h5>
                        
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <p><strong>MoMo Number:</strong><br>+268 <?= htmlspecialchars($payment_data['momo_number']) ?></p>
                                <p><strong>Payment Period:</strong><br><?= ucfirst($payment_data['payment_period']) ?></p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>Total Amount:</strong><br>
                                    <span class="text-success h4"><?= $currency_symbol ?><?= number_format($payment_data['total_amount'], 2) ?></span>
                                </p>
                            </div>
                        </div>

                        <h5 class="mb-3">Selected Subjects (<?= count($selected_subjects) ?>)</h5>
                        <div class="row">
                            <?php foreach ($selected_subjects as $subject): ?>
                            <div class="col-md-6 mb-2">
                                <div class="card bg-light">
                                    <div class="card-body py-2">
                                        <i class="bi bi-check-circle text-success me-2"></i>
                                        <strong><?= htmlspecialchars($subject['name']) ?></strong>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>

                        <div class="alert alert-warning mt-4">
                            <i class="bi bi-info-circle me-2"></i>
                            <strong>Important:</strong> By clicking "Confirm Payment", you authorize the payment of 
                            <?= $currency_symbol ?><?= number_format($payment_data['total_amount'], 2) ?> 
                            from your MoMo account.
                        </div>

                        <form method="POST" class="mt-4">
                            <div class="d-flex gap-2">
                                <a href="make_payment.php" class="btn btn-outline-secondary flex-fill">
                                    <i class="bi bi-arrow-left me-2"></i>Back
                                </a>
                                <button type="submit" name="confirm_payment" class="btn btn-success flex-fill">
                                    <i class="bi bi-check-circle me-2"></i>Confirm Payment
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

