<?php
/**
 * Student Make Payment
 * Students can pay for their subjects using MoMo
 */

session_start();
require_once '../includes/functions.php';
requireStudentLogin();

$student = getCurrentStudent();
$pdo = getDB();

// Get student details
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student['id']]);
$student_details = $stmt->fetch();

// Get school details
$stmt = $pdo->prepare("
    SELECT ar.academy_name, ts.setting_key, ts.setting_value
    FROM academy_references ar
    LEFT JOIN tenant_settings ts ON ar.reference_code = ts.academy_reference
    WHERE ar.reference_code = ?
");
$stmt->execute([$student['academy_reference']]);
$results = $stmt->fetchAll();
$school = ['academy_name' => $results[0]['academy_name'] ?? 'School'];
$settings = [];
foreach ($results as $row) {
    if ($row['setting_key']) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

$currency_symbol = $settings['currency_symbol'] ?? 'E';
$subject_fee = (float)($settings['subject_fee'] ?? 350);

// Get available subjects for grade
$stmt = $pdo->prepare("
    SELECT s.*
    FROM subjects s
    INNER JOIN grade_subject gs ON s.id = gs.subject_id
    WHERE gs.grade_id = ?
    ORDER BY s.name
");
$stmt->execute([$student_details['grade_id']]);
$available_subjects = $stmt->fetchAll();

// Get currently enrolled subjects
$stmt = $pdo->prepare("
    SELECT subject_id FROM student_subject WHERE student_id = ?
");
$stmt->execute([$student['id']]);
$enrolled_subject_ids = array_column($stmt->fetchAll(), 'subject_id');

$error = '';
$success = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['process_payment'])) {
    $selected_subjects = $_POST['subjects'] ?? [];
    $momo_number = trim($_POST['momo_number'] ?? '');
    $payment_period = $_POST['payment_period'] ?? 'monthly';
    
    if (empty($selected_subjects)) {
        $error = 'Please select at least one subject to pay for.';
    } elseif (empty($momo_number)) {
        $error = 'Please enter MoMo number.';
    } else {
        // Calculate amount
        $subject_count = count($selected_subjects);
        $monthly_total = $subject_count * $subject_fee;
        
        if ($payment_period === 'yearly') {
            $discount = (float)($settings['yearly_discount'] ?? 10);
            $yearly_total = $monthly_total * 12;
            $total_amount = $yearly_total * (1 - ($discount / 100));
        } else {
            $total_amount = $monthly_total;
        }
        
        // Store payment data in session
        $_SESSION['student_payment_data'] = [
            'student_id' => $student['id'],
            'academy_reference' => $student['academy_reference'],
            'subjects' => $selected_subjects,
            'momo_number' => $momo_number,
            'payment_period' => $payment_period,
            'total_amount' => $total_amount
        ];
        
        // Redirect to confirmation
        header('Location: payment_confirm.php');
        exit;
    }
}

$pageTitle = "Make Payment - Student Portal";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?></title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
        }
        .nav-link {
            color: rgba(255,255,255,0.8);
            transition: all 0.3s;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255,255,255,0.1);
        }
        .subject-checkbox {
            cursor: pointer;
            padding: 15px;
            border: 2px solid #dee2e6;
            border-radius: 8px;
            transition: all 0.3s;
        }
        .subject-checkbox:hover {
            border-color: #0d6efd;
            background: #f8f9fa;
        }
        .subject-checkbox.enrolled {
            background: #e7f1ff;
            border-color: #0d6efd;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0 sidebar text-white">
                <div class="p-3">
                    <h5 class="mb-0">
                        <i class="bi bi-mortarboard-fill me-2"></i>
                        Student Portal
                    </h5>
                    <small class="text-white-50"><?= htmlspecialchars($school['academy_name']) ?></small>
                </div>
                
                <nav class="nav flex-column px-2">
                    <a class="nav-link rounded" href="dashboard.php">
                        <i class="bi bi-house me-2"></i>Dashboard
                    </a>
                    <a class="nav-link rounded" href="subjects.php">
                        <i class="bi bi-book me-2"></i>My Subjects
                    </a>
                    <a class="nav-link active rounded" href="make_payment.php">
                        <i class="bi bi-credit-card me-2"></i>Make Payment
                    </a>
                    <a class="nav-link rounded" href="payment_history.php">
                        <i class="bi bi-receipt me-2"></i>Payment History
                    </a>
                    <?php if (!empty($settings['moodle_url'])): ?>
                    <a class="nav-link rounded" href="<?= htmlspecialchars($settings['moodle_url']) ?>" target="_blank">
                        <i class="bi bi-mortarboard me-2"></i>Moodle Portal
                        <i class="bi bi-box-arrow-up-right ms-1 small"></i>
                    </a>
                    <?php endif; ?>
                    <a class="nav-link rounded" href="profile.php">
                        <i class="bi bi-person me-2"></i>My Profile
                    </a>
                    <hr class="text-white-50">
                    <a class="nav-link rounded" href="logout.php">
                        <i class="bi bi-box-arrow-right me-2"></i>Logout
                    </a>
                </nav>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 px-4 py-3">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="bi bi-credit-card me-2"></i>Make Payment</h2>
                    <a href="dashboard.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left me-1"></i>Back
                    </a>
                </div>

                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="bi bi-exclamation-triangle me-2"></i>
                    <?= htmlspecialchars($error) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <form method="POST" id="paymentForm">
                    <div class="row">
                        <div class="col-md-8">
                            <!-- Select Subjects -->
                            <div class="card mb-4">
                                <div class="card-header bg-primary text-white">
                                    <h5 class="mb-0"><i class="bi bi-book me-2"></i>Select Subjects to Pay For</h5>
                                </div>
                                <div class="card-body">
                                    <p class="text-muted">
                                        <i class="bi bi-info-circle me-1"></i>
                                        Select the subjects you want to pay for. Already enrolled subjects are highlighted.
                                    </p>
                                    
                                    <div class="row g-3">
                                        <?php foreach ($available_subjects as $subject): 
                                            $is_enrolled = in_array($subject['id'], $enrolled_subject_ids);
                                        ?>
                                        <div class="col-md-6">
                                            <div class="subject-checkbox <?= $is_enrolled ? 'enrolled' : '' ?>">
                                                <div class="form-check">
                                                    <input class="form-check-input subject-check" type="checkbox" 
                                                           name="subjects[]" value="<?= $subject['id'] ?>" 
                                                           id="subject_<?= $subject['id'] ?>"
                                                           data-fee="<?= $subject_fee ?>">
                                                    <label class="form-check-label" for="subject_<?= $subject['id'] ?>">
                                                        <strong><?= htmlspecialchars($subject['name']) ?></strong>
                                                        <br>
                                                        <small class="text-muted">Code: <?= htmlspecialchars($subject['code']) ?></small>
                                                        <?php if ($is_enrolled): ?>
                                                        <br><span class="badge bg-primary mt-1">Currently Enrolled</span>
                                                        <?php endif; ?>
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>

                            <!-- Payment Details -->
                            <div class="card mb-4">
                                <div class="card-header bg-info text-white">
                                    <h5 class="mb-0"><i class="bi bi-phone me-2"></i>Payment Details</h5>
                                </div>
                                <div class="card-body">
                                    <div class="mb-3">
                                        <label for="momo_number" class="form-label">
                                            <i class="bi bi-phone me-1"></i>MoMo Number *
                                        </label>
                                        <div class="input-group">
                                            <span class="input-group-text">+268</span>
                                            <input type="tel" class="form-control" id="momo_number" name="momo_number" 
                                                   placeholder="76001234" required pattern="[0-9]{8}" maxlength="8">
                                        </div>
                                        <small class="text-muted">Your number or your parent's number for payment</small>
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Payment Period</label>
                                        <div class="btn-group w-100" role="group">
                                            <input type="radio" class="btn-check" name="payment_period" id="monthly" 
                                                   value="monthly" checked onchange="calculateTotal()">
                                            <label class="btn btn-outline-primary" for="monthly">
                                                <i class="bi bi-calendar-month me-1"></i>Monthly
                                            </label>
                                            
                                            <input type="radio" class="btn-check" name="payment_period" id="yearly" 
                                                   value="yearly" onchange="calculateTotal()">
                                            <label class="btn btn-outline-primary" for="yearly">
                                                <i class="bi bi-calendar-range me-1"></i>Yearly 
                                                <span class="badge bg-success ms-1"><?= $settings['yearly_discount'] ?? 10 ?>% OFF</span>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Summary -->
                        <div class="col-md-4">
                            <div class="card sticky-top" style="top: 20px;">
                                <div class="card-header bg-success text-white">
                                    <h5 class="mb-0"><i class="bi bi-calculator me-2"></i>Payment Summary</h5>
                                </div>
                                <div class="card-body">
                                    <div class="mb-3">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Selected Subjects:</span>
                                            <strong id="subjectCount">0</strong>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Fee per Subject:</span>
                                            <strong><?= $currency_symbol ?><?= number_format($subject_fee, 2) ?></strong>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Period:</span>
                                            <strong id="periodText">Monthly</strong>
                                        </div>
                                        <div id="discountRow" class="d-flex justify-content-between mb-2 text-success" style="display: none !important;">
                                            <span>Yearly Discount:</span>
                                            <strong>-<?= $settings['yearly_discount'] ?? 10 ?>%</strong>
                                        </div>
                                    </div>
                                    <hr>
                                    <div class="d-flex justify-content-between mb-3">
                                        <strong>Total Amount:</strong>
                                        <h4 class="mb-0 text-success" id="totalAmount"><?= $currency_symbol ?>0.00</h4>
                                    </div>
                                    
                                    <button type="submit" name="process_payment" class="btn btn-success w-100 btn-lg" id="submitBtn" disabled>
                                        <i class="bi bi-credit-card me-2"></i>Proceed to Payment
                                    </button>
                                    
                                    <div class="mt-3">
                                        <small class="text-muted">
                                            <i class="bi bi-shield-check me-1"></i>
                                            Secure payment via MTN MoMo
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const subjectFee = <?= $subject_fee ?>;
        const yearlyDiscount = <?= $settings['yearly_discount'] ?? 10 ?>;
        const currencySymbol = '<?= $currency_symbol ?>';
        
        function calculateTotal() {
            const checkboxes = document.querySelectorAll('.subject-check:checked');
            const count = checkboxes.length;
            const isYearly = document.getElementById('yearly').checked;
            
            let total = 0;
            if (count > 0) {
                const monthlyTotal = count * subjectFee;
                
                if (isYearly) {
                    const yearlyTotal = monthlyTotal * 12;
                    total = yearlyTotal * (1 - (yearlyDiscount / 100));
                    document.getElementById('discountRow').style.display = 'flex';
                    document.getElementById('periodText').textContent = 'Yearly (12 months)';
                } else {
                    total = monthlyTotal;
                    document.getElementById('discountRow').style.display = 'none';
                    document.getElementById('periodText').textContent = 'Monthly';
                }
            }
            
            document.getElementById('subjectCount').textContent = count;
            document.getElementById('totalAmount').textContent = currencySymbol + total.toFixed(2);
            document.getElementById('submitBtn').disabled = count === 0;
        }
        
        // Add event listeners to checkboxes
        document.querySelectorAll('.subject-check').forEach(checkbox => {
            checkbox.addEventListener('change', calculateTotal);
        });
        
        // Initial calculation
        calculateTotal();
    </script>
</body>
</html>

