<?php
/**
 * Student Dashboard
 * Main dashboard for logged-in students
 */

require_once '../includes/functions.php';
requireStudentLogin();

$student = getCurrentStudent();
$pdo = getDB();

// Get student details
$stmt = $pdo->prepare("
    SELECT s.*, g.name as grade_name
    FROM students s
    LEFT JOIN grades g ON s.grade_id = g.id
    WHERE s.id = ?
");
$stmt->execute([$student['id']]);
$student_details = $stmt->fetch();

// Get school details
$stmt = $pdo->prepare("
    SELECT academy_name, contact_email 
    FROM academy_references 
    WHERE reference_code = ?
");
$stmt->execute([$student['academy_reference']]);
$school = $stmt->fetch();

// Get enrolled subjects
$stmt = $pdo->prepare("
    SELECT 
        sub.id,
        sub.name, 
        sub.code
    FROM student_subject ss
    INNER JOIN subjects sub ON ss.subject_id = sub.id
    WHERE ss.student_id = ?
    ORDER BY sub.name
");
$stmt->execute([$student['id']]);
$subjects = $stmt->fetchAll();

// Get recent payments
$stmt = $pdo->prepare("
    SELECT 
        p.amount,
        p.payment_date,
        p.payment_method,
        p.payment_reference,
        p.status
    FROM payments p
    WHERE p.student_id = ?
    ORDER BY p.payment_date DESC
    LIMIT 5
");
$stmt->execute([$student['id']]);
$payments = $stmt->fetchAll();

// Get payment summary
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_payments,
        SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END) as total_paid
    FROM payments 
    WHERE student_id = ?
");
$stmt->execute([$student['id']]);
$payment_summary = $stmt->fetch();

// Get school settings (for Moodle URL)
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$student['academy_reference']]);
$school_settings = [];
while ($row = $stmt->fetch()) {
    $school_settings[$row['setting_key']] = $row['setting_value'];
}

$moodle_url = $school_settings['moodle_url'] ?? null;
$currency_symbol = $school_settings['currency_symbol'] ?? 'E';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Dashboard - <?= htmlspecialchars($school['academy_name']) ?></title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
        }
        .nav-link {
            color: rgba(255,255,255,0.8);
            transition: all 0.3s;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255,255,255,0.1);
        }
        .stat-card {
            border-left: 4px solid #0d6efd;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0 sidebar text-white">
                <div class="p-3">
                    <h5 class="mb-0">
                        <i class="bi bi-mortarboard-fill me-2"></i>
                        Student Portal
                    </h5>
                    <small class="text-white-50"><?= htmlspecialchars($school['academy_name']) ?></small>
                </div>
                
                <nav class="nav flex-column px-2">
                    <a class="nav-link active rounded" href="dashboard.php">
                        <i class="bi bi-house me-2"></i>Dashboard
                    </a>
                    <a class="nav-link rounded" href="subjects.php">
                        <i class="bi bi-book me-2"></i>My Subjects
                    </a>
                    <a class="nav-link rounded" href="make_payment.php">
                        <i class="bi bi-credit-card me-2"></i>Make Payment
                    </a>
                    <a class="nav-link rounded" href="payment_history.php">
                        <i class="bi bi-receipt me-2"></i>Payment History
                    </a>
                    <?php if ($moodle_url): ?>
                    <a class="nav-link rounded" href="<?= htmlspecialchars($moodle_url) ?>" target="_blank">
                        <i class="bi bi-mortarboard me-2"></i>Moodle Portal
                        <i class="bi bi-box-arrow-up-right ms-1 small"></i>
                    </a>
                    <?php endif; ?>
                    <a class="nav-link rounded" href="profile.php">
                        <i class="bi bi-person me-2"></i>My Profile
                    </a>
                    <hr class="text-white-50">
                    <a class="nav-link rounded" href="logout.php">
                        <i class="bi bi-box-arrow-right me-2"></i>Logout
                    </a>
                </nav>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 px-4 py-3">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2>Welcome, <?= htmlspecialchars($student_details['full_name'] ?? $student['name']) ?>!</h2>
                        <p class="text-muted mb-0">
                            <i class="bi bi-person-badge me-1"></i>
                            <?= htmlspecialchars($student['username']) ?> | 
                            <?= htmlspecialchars($student_details['grade_name'] ?? 'Student') ?>
                        </p>
                    </div>
                    <div>
                        <?php
                        $accountStatus = $student_details['status'] ?? 'active';
                        $badgeClass = $accountStatus === 'active' ? 'bg-success' : 'bg-warning';
                        ?>
                        <span class="badge <?= $badgeClass ?>"><?= ucfirst($accountStatus) ?></span>
                    </div>
                </div>
                
                <!-- Stats Cards -->
                <div class="row g-3 mb-4">
                    <div class="col-md-4">
                        <div class="card stat-card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <p class="text-muted mb-1">Enrolled Subjects</p>
                                        <h3 class="mb-0"><?= count($subjects) ?></h3>
                                    </div>
                                    <div>
                                        <i class="bi bi-book-fill text-primary" style="font-size: 2.5rem;"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="card stat-card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <p class="text-muted mb-1">Total Paid</p>
                                        <h3 class="mb-0"><?= $currency_symbol ?><?= number_format($payment_summary['total_paid'] ?? 0, 2) ?></h3>
                                    </div>
                                    <div>
                                        <i class="bi bi-credit-card-fill text-success" style="font-size: 2.5rem;"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="card stat-card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <p class="text-muted mb-1">Payments Made</p>
                                        <h3 class="mb-0"><?= $payment_summary['total_payments'] ?? 0 ?></h3>
                                    </div>
                                    <div>
                                        <i class="bi bi-receipt-fill text-info" style="font-size: 2.5rem;"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Moodle Access Card -->
                <?php if ($moodle_url): ?>
                <div class="alert alert-info mb-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h5 class="alert-heading mb-2">
                                <i class="bi bi-mortarboard me-2"></i>Access Your Moodle Courses
                            </h5>
                            <p class="mb-0">Use your same login credentials to access learning materials and assignments.</p>
                            <small><strong>Username:</strong> <?= htmlspecialchars($student['username']) ?></small>
                        </div>
                        <div>
                            <a href="<?= htmlspecialchars($moodle_url) ?>" target="_blank" class="btn btn-primary">
                                <i class="bi bi-box-arrow-up-right me-2"></i>Open Moodle
                            </a>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Enrolled Subjects -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-book me-2"></i>My Enrolled Subjects</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($subjects)): ?>
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle me-2"></i>
                                You are not enrolled in any subjects yet. Contact your administrator or parent.
                            </div>
                        <?php else: ?>
                            <div class="row">
                                <?php foreach ($subjects as $subject): ?>
                                <div class="col-md-6 mb-3">
                                    <div class="card h-100">
                                        <div class="card-body">
                                            <h6 class="card-title">
                                                <i class="bi bi-book text-primary me-2"></i>
                                                <?= htmlspecialchars($subject['name']) ?>
                                            </h6>
                                            <p class="mb-0">
                                                <small class="text-muted">Code: <?= htmlspecialchars($subject['code']) ?></small>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <div class="mt-3">
                                <a href="subjects.php" class="btn btn-outline-primary">
                                    <i class="bi bi-plus-circle me-2"></i>Manage Subjects
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Recent Payments -->
                <div class="card">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0"><i class="bi bi-credit-card me-2"></i>Recent Payments</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($payments)): ?>
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle me-2"></i>
                                No payment history available.
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Amount</th>
                                            <th>Method</th>
                                            <th>Reference</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($payments as $payment): ?>
                                        <tr>
                                            <td><?= date('M d, Y', strtotime($payment['payment_date'])) ?></td>
                                            <td><?= $currency_symbol ?><?= number_format($payment['amount'], 2) ?></td>
                                            <td>
                                                <i class="bi bi-phone me-1"></i>
                                                <?= ucfirst($payment['payment_method'] ?? 'MoMo') ?>
                                            </td>
                                            <td><code><?= htmlspecialchars($payment['payment_reference']) ?></code></td>
                                            <td>
                                                <?php if ($payment['status'] === 'completed'): ?>
                                                    <span class="badge bg-success"><i class="bi bi-check-circle"></i> Completed</span>
                                                <?php elseif ($payment['status'] === 'pending'): ?>
                                                    <span class="badge bg-warning"><i class="bi bi-clock"></i> Pending</span>
                                                <?php else: ?>
                                                    <span class="badge bg-danger"><i class="bi bi-x-circle"></i> Failed</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <div class="mt-3">
                                <a href="payment_history.php" class="btn btn-outline-success">
                                    <i class="bi bi-receipt me-2"></i>View All Payments
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

