<?php
/**
 * Secure File Serving Script
 * 
 * Serves files from tenant storage with proper access control
 * Prevents direct access to tenant files
 */

// Start session to check user authentication
session_start();

// Include necessary files
require_once 'config.php';
require_once 'includes/tenant_filesystem.php';

// Initialize file system
$fs = new TenantFileSystem($conn);

// Get parameters
$tenant_id = $_GET['tenant'] ?? null;
$type = $_GET['type'] ?? null;
$filename = $_GET['file'] ?? null;

// Validate parameters
if (!$tenant_id || !$type || !$filename) {
    http_response_code(400);
    die('Missing required parameters');
}

// ========================================================================
// ACCESS CONTROL
// ========================================================================

$hasAccess = false;

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    die('Authentication required');
}

// Determine user role and tenant access
$user_role = $_SESSION['role'] ?? null;
$user_tenant = $_SESSION['academy_reference'] ?? null;

// Super admin can access all tenants
if ($user_role === 'super_admin') {
    $hasAccess = true;
}
// Admin can access their own tenant
elseif ($user_role === 'admin' && $user_tenant === $tenant_id) {
    $hasAccess = true;
}
// Parent can access their own tenant
elseif ($user_role === 'parent' && isset($_SESSION['school_reference']) && $_SESSION['school_reference'] === $tenant_id) {
    // Additional check: only profile photos and payment receipts for parents
    if (in_array($type, ['profile_photos', 'payment_receipts'])) {
        $hasAccess = true;
    }
}
// Teacher/Student can access their tenant's files (with restrictions)
elseif (in_array($user_role, ['teacher', 'student']) && $user_tenant === $tenant_id) {
    // Allow access to specific types only
    if (in_array($type, ['documents', 'profile_photos', 'reports'])) {
        $hasAccess = true;
    }
}

// Check access
if (!$hasAccess) {
    http_response_code(403);
    die('Access denied');
}

// ========================================================================
// FILE VALIDATION & SERVING
// ========================================================================

// Validate tenant access to file (prevents path traversal)
if (!$fs->validateTenantAccess($tenant_id, $type . '/' . $filename)) {
    http_response_code(403);
    die('Invalid file access');
}

// Get file path
$filePath = $fs->getTenantPath($tenant_id, $type) . '/' . basename($filename);

// Check if file exists
if (!file_exists($filePath)) {
    http_response_code(404);
    die('File not found');
}

// Get file info
$fileInfo = pathinfo($filePath);
$extension = strtolower($fileInfo['extension']);
$filesize = filesize($filePath);

// Set content type based on extension
$contentTypes = [
    // Images
    'jpg' => 'image/jpeg',
    'jpeg' => 'image/jpeg',
    'png' => 'image/png',
    'gif' => 'image/gif',
    'webp' => 'image/webp',
    'svg' => 'image/svg+xml',
    
    // Documents
    'pdf' => 'application/pdf',
    'doc' => 'application/msword',
    'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'xls' => 'application/vnd.ms-excel',
    'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'txt' => 'text/plain',
    'csv' => 'text/csv',
    
    // Archives
    'zip' => 'application/zip',
    'tar' => 'application/x-tar',
    'gz' => 'application/gzip',
    
    // Other
    'json' => 'application/json',
    'xml' => 'application/xml',
    'html' => 'text/html'
];

$contentType = $contentTypes[$extension] ?? 'application/octet-stream';

// Determine if file should be displayed inline or downloaded
$inline_types = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf', 'txt'];
$disposition = in_array($extension, $inline_types) ? 'inline' : 'attachment';

// Clear any previous output
if (ob_get_level()) {
    ob_end_clean();
}

// Set headers
header('Content-Type: ' . $contentType);
header('Content-Length: ' . $filesize);
header('Content-Disposition: ' . $disposition . '; filename="' . basename($filename) . '"');
header('Cache-Control: private, max-age=3600'); // Cache for 1 hour
header('Pragma: private');
header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 3600) . ' GMT');

// Security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: SAMEORIGIN');

// Serve file
readfile($filePath);

// Log access (optional)
try {
    $stmt = $conn->prepare("
        INSERT INTO file_access_log 
        (tenant_id, user_id, file_type, filename, accessed_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    $stmt->execute([
        $tenant_id,
        $_SESSION['user_id'],
        $type,
        $filename
    ]);
} catch (PDOException $e) {
    // Silent fail - logging is optional
}

exit;

