<?php
/**
 * School Selection Page
 * Allows admins to select which school to manage
 */

session_start();
require_once 'config.php';
require_once 'includes/school_context.php';
require_once 'includes/favicon_helper.php';

// Get database connection
$pdo = getDB();

// Check if already has a school selected via URL parameter
if (isset($_GET['school'])) {
    $reference = strtoupper(trim($_GET['school']));
    if (setCurrentSchool($reference)) {
        // Redirect to admin dashboard
        header('Location: admin/dashboard.php');
        exit;
    } else {
        $error = "Invalid or inactive school selected.";
    }
}

// Get all active schools with real student counts and revenue
try {
    // First, get basic school info
    $stmt = $pdo->query("
        SELECT * FROM academy_references 
        WHERE is_active = 1 
        ORDER BY academy_name ASC
    ");
    $schools = $stmt->fetchAll();
    
    // Then enhance each school with real data
    foreach ($schools as &$school) {
        // Get student count
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count FROM students 
            WHERE academy_reference = ?
        ");
        $stmt->execute([$school['reference_code']]);
        $school['total_students'] = $stmt->fetch()['count'];
        
        // Get revenue from completed payments
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(amount), 0) as revenue 
            FROM payment_transactions 
            WHERE academy_reference = ? AND status = 'completed'
        ");
        $stmt->execute([$school['reference_code']]);
        $school['total_revenue'] = $stmt->fetch()['revenue'];
        
        // Get overdue count
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count FROM student_payment_schedules 
            WHERE academy_reference = ? AND payment_status = 'overdue'
        ");
        $stmt->execute([$school['reference_code']]);
        $school['overdue_count'] = $stmt->fetch()['count'];
        
        // Get suspended count
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count FROM students 
            WHERE academy_reference = ? AND suspended = 1
        ");
        $stmt->execute([$school['reference_code']]);
        $school['suspended_count'] = $stmt->fetch()['count'];
    }
} catch (Exception $e) {
    // Fallback to simple query
    $stmt = $pdo->query("
        SELECT * FROM academy_references 
        WHERE is_active = 1 
        ORDER BY academy_name ASC
    ");
    $schools = $stmt->fetchAll();
}

// If only one school and not coming from auto-redirect, auto-select it
if (count($schools) === 1 && !isset($error) && !isset($_GET['manual'])) {
    setCurrentSchool($schools[0]['reference_code']);
    header('Location: admin/dashboard.php');
    exit;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Select School - Skolo-Kine</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        * {
            font-family: 'Inter', sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem 1rem;
            position: relative;
            overflow-x: hidden;
        }
        
        /* Animated background particles */
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: 
                radial-gradient(circle at 20% 50%, rgba(255, 255, 255, 0.1) 0%, transparent 50%),
                radial-gradient(circle at 80% 80%, rgba(255, 255, 255, 0.1) 0%, transparent 50%),
                radial-gradient(circle at 40% 20%, rgba(255, 255, 255, 0.05) 0%, transparent 50%);
            animation: float 20s ease-in-out infinite;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
        }
        
        .container-custom {
            max-width: 1200px;
            position: relative;
            z-index: 1;
        }
        
        .header-section {
            text-align: center;
            margin-bottom: 3rem;
            animation: fadeInDown 0.6s ease;
        }
        
        @keyframes fadeInDown {
            from {
                opacity: 0;
                transform: translateY(-30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .header-icon {
            width: 80px;
            height: 80px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1.5rem;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.3);
        }
        
        .header-icon i {
            font-size: 2.5rem;
            color: white;
        }
        
        h1 {
            color: white;
            font-weight: 800;
            font-size: 2.5rem;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 10px rgba(0, 0, 0, 0.2);
        }
        
        .subtitle {
            color: rgba(255, 255, 255, 0.9);
            font-size: 1.1rem;
            font-weight: 300;
        }
        
        .school-card {
            transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
            cursor: pointer;
            border: none;
            background: white;
            border-radius: 20px;
            overflow: hidden;
            position: relative;
            animation: fadeInUp 0.6s ease backwards;
        }
        
        .school-card:nth-child(1) { animation-delay: 0.1s; }
        .school-card:nth-child(2) { animation-delay: 0.2s; }
        .school-card:nth-child(3) { animation-delay: 0.3s; }
        .school-card:nth-child(4) { animation-delay: 0.4s; }
        
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .school-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 6px;
            background: linear-gradient(90deg, var(--school-color), var(--school-color-secondary));
            opacity: 0;
            transition: opacity 0.3s;
        }
        
        .school-card:hover::before {
            opacity: 1;
        }
        
        .school-card:hover {
            transform: translateY(-10px) scale(1.02);
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }
        
        .school-logo-container {
            width: 100px;
            height: 100px;
            margin: 0 auto 1rem;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
            background: linear-gradient(135deg, var(--school-color-light), var(--school-color-lighter));
            transition: all 0.3s;
        }
        
        .school-card:hover .school-logo-container {
            transform: rotate(5deg) scale(1.1);
        }
        
        .school-logo {
            width: 60px;
            height: 60px;
            object-fit: contain;
        }
        
        .school-name {
            font-size: 1.3rem;
            font-weight: 700;
            color: #1a1a1a;
            margin-bottom: 0.5rem;
        }
        
        .school-code {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            background: linear-gradient(135deg, var(--school-color), var(--school-color-secondary));
            color: white;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 600;
            letter-spacing: 0.5px;
            margin-bottom: 1rem;
        }
        
        .stats-row {
            display: flex;
            justify-content: space-around;
            padding: 1rem 0;
            border-top: 1px solid #f0f0f0;
            margin-top: 1rem;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--school-color);
            display: block;
        }
        
        .stat-label {
            font-size: 0.75rem;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .access-btn {
            width: 100%;
            padding: 0.875rem;
            background: linear-gradient(135deg, var(--school-color), var(--school-color-secondary));
            border: none;
            color: white;
            font-weight: 600;
            border-radius: 12px;
            transition: all 0.3s;
            margin-top: 1rem;
        }
        
        .access-btn:hover {
            transform: scale(1.02);
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.2);
            color: white;
        }
        
        .add-school-btn {
            background: rgba(255, 255, 255, 0.2);
            border: 2px solid rgba(255, 255, 255, 0.5);
            color: white;
            padding: 1rem 2rem;
            border-radius: 50px;
            font-weight: 600;
            backdrop-filter: blur(10px);
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .add-school-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            border-color: white;
            transform: scale(1.05);
            color: white;
        }
        
        .logout-link {
            color: rgba(255, 255, 255, 0.9);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            border-radius: 25px;
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            transition: all 0.3s;
        }
        
        .logout-link:hover {
            background: rgba(255, 255, 255, 0.2);
            color: white;
        }
        
        .empty-state {
            background: white;
            border-radius: 20px;
            padding: 4rem 2rem;
            text-align: center;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
        }
        
        .empty-state i {
            font-size: 5rem;
            color: #f59e0b;
            margin-bottom: 1.5rem;
        }
        
        .alert-modern {
            background: white;
            border: none;
            border-radius: 15px;
            padding: 1.5rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            animation: shake 0.5s;
        }
        
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-10px); }
            75% { transform: translateX(10px); }
        }
        
        /* Fade in animation */
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        /* Pulse animation for stats */
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }
        
        .stat-value:hover {
            animation: pulse 0.6s ease;
        }
        
        /* Shine effect on hover */
        .school-card::after {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: linear-gradient(
                45deg,
                transparent 30%,
                rgba(255, 255, 255, 0.1) 50%,
                transparent 70%
            );
            transform: rotate(45deg);
            transition: all 0.5s;
            opacity: 0;
        }
        
        .school-card:hover::after {
            opacity: 1;
            animation: shine 1.5s ease;
        }
        
        @keyframes shine {
            from { transform: translateX(-100%) rotate(45deg); }
            to { transform: translateX(100%) rotate(45deg); }
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            h1 { font-size: 2rem; }
            .school-card { margin-bottom: 1.5rem; }
            .header-icon { width: 60px; height: 60px; }
            .header-icon i { font-size: 2rem; }
            .stats-row { flex-direction: column; gap: 1rem; }
        }
        
        /* Loading animation */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container container-custom">
        <div class="header-section">
            <div class="header-icon">
                <i class="bi bi-buildings"></i>
            </div>
            <h1>Select Your School</h1>
            <p class="subtitle">Choose which school you want to manage</p>
        </div>

        <?php if (isset($error)): ?>
            <div class="alert alert-modern alert-danger mb-4" role="alert">
                <i class="bi bi-exclamation-triangle-fill me-2 fs-4"></i>
                <strong><?= htmlspecialchars($error) ?></strong>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (empty($schools)): ?>
            <div class="empty-state">
                <i class="bi bi-inbox"></i>
                <h3 class="mb-3" style="color: #1a1a1a;">No Active Schools Found</h3>
                <p class="text-muted mb-4">There are currently no active schools in the system.</p>
                <a href="/super_admin/schools/create.php" class="btn btn-primary btn-lg rounded-pill">
                    <i class="bi bi-plus-circle me-2"></i>Create Your First School
                </a>
            </div>
        <?php else: ?>
            <div class="row g-4">
                <?php foreach ($schools as $school): 
                    // Get school settings
                    $stmt = $pdo->prepare("
                        SELECT setting_key, setting_value 
                        FROM tenant_settings 
                        WHERE academy_reference = ?
                    ");
                    $stmt->execute([$school['reference_code']]);
                    $settings = [];
                    while ($row = $stmt->fetch()) {
                        $settings[$row['setting_key']] = $row['setting_value'];
                    }
                    
                    $schoolName = $settings['school_name'] ?? $school['academy_name'];
                    $logoUrl = $settings['logo_url'] ?? 'images/img-melanesys-logo.svg';
                    // Fix logo path if it starts with /
                    if (strpos($logoUrl, '/') === 0) {
                        $logoUrl = ltrim($logoUrl, '/');
                    }
                    $primaryColor = $settings['primary_color'] ?? '#4F46E5';
                    $secondaryColor = $settings['secondary_color'] ?? '#10B981';
                    $studentCount = (int)($school['total_students'] ?? 0);
                    $totalRevenue = (float)($school['total_revenue'] ?? 0);
                    $overdueCount = (int)($school['overdue_count'] ?? 0);
                    $suspendedCount = (int)($school['suspended_count'] ?? 0);
                ?>
                <div class="col-md-4 col-lg-4">
                    <a href="?school=<?= urlencode($school['reference_code']) ?>" 
                       class="text-decoration-none"
                       style="--school-color: <?= htmlspecialchars($primaryColor) ?>; 
                              --school-color-secondary: <?= htmlspecialchars($secondaryColor ?? $primaryColor) ?>;
                              --school-color-light: <?= htmlspecialchars($primaryColor) ?>33;
                              --school-color-lighter: <?= htmlspecialchars($primaryColor) ?>11;">
                        <div class="card school-card h-100">
                            <div class="card-body text-center p-4">
                                <div class="school-logo-container">
                                    <img src="<?= htmlspecialchars($logoUrl) ?>" 
                                         alt="<?= htmlspecialchars($schoolName) ?>" 
                                         class="school-logo"
                                         onerror="this.src='images/img-melanesys-logo.svg'">
                                </div>
                                
                                <h5 class="school-name">
                                    <?= htmlspecialchars($schoolName) ?>
                                </h5>
                                
                                <span class="school-code">
                                    <?= htmlspecialchars($school['reference_code']) ?>
                                </span>
                                
                                <div class="stats-row">
                                    <div class="stat-item">
                                        <span class="stat-value"><?= number_format($studentCount) ?></span>
                                        <span class="stat-label">Students</span>
                                    </div>
                                    <div class="stat-item">
                                        <span class="stat-value">E<?= number_format($totalRevenue, 0) ?></span>
                                        <span class="stat-label">Revenue</span>
                                    </div>
                                </div>
                                <?php if ($overdueCount > 0 || $suspendedCount > 0): ?>
                                <div class="mt-2">
                                    <?php if ($overdueCount > 0): ?>
                                    <small class="badge bg-warning text-dark">
                                        <i class="bi bi-exclamation-triangle"></i> <?= $overdueCount ?> Overdue
                                    </small>
                                    <?php endif; ?>
                                    <?php if ($suspendedCount > 0): ?>
                                    <small class="badge bg-danger">
                                        <i class="bi bi-ban"></i> <?= $suspendedCount ?> Suspended
                                    </small>
                                    <?php endif; ?>
                                </div>
                                <?php endif; ?>
                                
                                <button class="access-btn">
                                    <i class="bi bi-box-arrow-in-right me-2"></i>
                                    Access Dashboard
                                </button>
                            </div>
                        </div>
                    </a>
                </div>
                <?php endforeach; ?>
            </div>

            <?php if (function_exists('isSuperAdmin') && isSuperAdmin()): ?>
                <div class="text-center mt-5" style="animation: fadeIn 1s ease 0.8s backwards;">
                    <a href="super_admin/schools/create.php" class="add-school-btn">
                        <i class="bi bi-plus-circle fs-5"></i>
                        <span>Add New School</span>
                    </a>
                </div>
            <?php endif; ?>
        <?php endif; ?>

        <div class="text-center mt-5" style="animation: fadeIn 1s ease 0.9s backwards;">
            <a href="admin/logout.php" class="logout-link">
                <i class="bi bi-box-arrow-left"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Add smooth hover effects
        document.querySelectorAll('.school-card').forEach(card => {
            card.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-10px) scale(1.02)';
            });
            card.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0) scale(1)';
            });
        });
        
        // Add fade in animation
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
    </script>
</body>
</html>

