<?php
/**
 * Seed Test Students
 * Creates 5 students for KINE Academy and 5 for TEST School
 */

require_once 'config.php';
require_once 'includes/functions.php';

echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Seeding Test Students for Multi-Tenant System            ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n\n";

$pdo = getDB();

// Define test students for each school
$schools = [
    'KINE' => [
        ['first_name' => 'Alice', 'last_name' => 'Johnson', 'email' => 'alice.johnson@kine.test', 'grade_id' => 1],
        ['first_name' => 'Bob', 'last_name' => 'Smith', 'email' => 'bob.smith@kine.test', 'grade_id' => 1],
        ['first_name' => 'Charlie', 'last_name' => 'Brown', 'email' => 'charlie.brown@kine.test', 'grade_id' => 1],
        ['first_name' => 'Diana', 'last_name' => 'Prince', 'email' => 'diana.prince@kine.test', 'grade_id' => 1],
        ['first_name' => 'Edward', 'last_name' => 'Davis', 'email' => 'edward.davis@kine.test', 'grade_id' => 1]
    ],
    'TEST' => [
        ['first_name' => 'Frank', 'last_name' => 'Wilson', 'email' => 'frank.wilson@test.school', 'grade_id' => 1],
        ['first_name' => 'Grace', 'last_name' => 'Lee', 'email' => 'grace.lee@test.school', 'grade_id' => 1],
        ['first_name' => 'Henry', 'last_name' => 'Taylor', 'email' => 'henry.taylor@test.school', 'grade_id' => 1],
        ['first_name' => 'Isabel', 'last_name' => 'Martinez', 'email' => 'isabel.martinez@test.school', 'grade_id' => 1],
        ['first_name' => 'James', 'last_name' => 'Anderson', 'email' => 'james.anderson@test.school', 'grade_id' => 1]
    ]
];

// Subject IDs for enrollment (Math, English, SiSwati, Religious Ed, Social Studies)
$defaultSubjects = [1, 2, 3, 4, 5];

$totalCreated = 0;
$totalSkipped = 0;
$errors = 0;

foreach ($schools as $schoolCode => $students) {
    echo "Creating students for {$schoolCode} Academy:\n";
    echo str_repeat("-", 60) . "\n";
    
    foreach ($students as $studentData) {
        try {
            // Check if student already exists
            $stmt = $pdo->prepare("SELECT id FROM students WHERE email = ?");
            $stmt->execute([$studentData['email']]);
            
            if ($stmt->fetch()) {
                echo "  ⚠ Skipped (exists): {$studentData['first_name']} {$studentData['last_name']}\n";
                $totalSkipped++;
                continue;
            }
            
            // Create student with academy_reference
            $studentId = createStudent([
                'first_name' => $studentData['first_name'],
                'last_name' => $studentData['last_name'],
                'email' => $studentData['email'],
                'phone' => '+268 ' . rand(7000, 7999) . ' ' . rand(1000, 9999),
                'grade_id' => $studentData['grade_id'],
                'password_plain' => $studentData['first_name'] . '123!',
                'subject_ids' => $defaultSubjects,
                'academy_reference' => $schoolCode
            ]);
            
            // Mark email as verified for testing
            $stmt = $pdo->prepare("UPDATE students SET email_verified = 1 WHERE id = ?");
            $stmt->execute([$studentId]);
            
            echo "  ✓ Created: {$studentData['first_name']} {$studentData['last_name']} (ID: {$studentId})\n";
            $totalCreated++;
            
        } catch (Exception $e) {
            echo "  ✗ Error creating {$studentData['first_name']}: " . $e->getMessage() . "\n";
            $errors++;
        }
    }
    
    echo "\n";
}

// Create sample payments for some students
echo "Creating sample payments:\n";
echo str_repeat("-", 60) . "\n";

try {
    // Get 2 students from KINE
    $stmt = $pdo->prepare("SELECT id FROM students WHERE academy_reference = 'KINE' AND email LIKE '%@kine.test' LIMIT 2");
    $stmt->execute();
    $kineStudents = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    foreach ($kineStudents as $studentId) {
        $refId = 'KINE_SEED_' . $studentId . '_' . time();
        $stmt = $pdo->prepare("
            INSERT INTO payments 
            (student_id, amount, currency, reference_id, external_id, msisdn, status, payment_method, academy_reference, platform_fee, admin_earnings, created_at)
            VALUES (?, 1.50, 'SZL', ?, ?, '+26876123456', 'SUCCESSFUL', 'MOMO', 'KINE', 0.50, 1.00, NOW())
        ");
        $stmt->execute([$studentId, $refId, 'EXT_' . $refId]);
        echo "  ✓ Payment for KINE student ID: {$studentId}\n";
    }
    
    // Get 2 students from TEST
    $stmt = $pdo->prepare("SELECT id FROM students WHERE academy_reference = 'TEST' AND email LIKE '%@test.school' LIMIT 2");
    $stmt->execute();
    $testStudents = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    foreach ($testStudents as $studentId) {
        $refId = 'TEST_SEED_' . $studentId . '_' . time();
        $stmt = $pdo->prepare("
            INSERT INTO payments 
            (student_id, amount, currency, reference_id, external_id, msisdn, status, payment_method, academy_reference, platform_fee, admin_earnings, created_at)
            VALUES (?, 1.50, 'SZL', ?, ?, '+26876789012', 'SUCCESSFUL', 'MOMO', 'TEST', 0.50, 1.00, NOW())
        ");
        $stmt->execute([$studentId, $refId, 'EXT_' . $refId]);
        echo "  ✓ Payment for TEST student ID: {$studentId}\n";
    }
    
} catch (Exception $e) {
    echo "  ⚠ Error creating payments: " . $e->getMessage() . "\n";
}

echo "\n";

// Summary
echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Seeding Summary                                           ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n";
echo "  ✓ Students Created: $totalCreated\n";
echo "  ⚠ Skipped (existing): $totalSkipped\n";
echo "  ✗ Errors: $errors\n";
echo "\n";

// Show summary
echo "Student Summary:\n";
echo str_repeat("-", 60) . "\n";

$stmt = $pdo->query("
    SELECT 
        academy_reference,
        COUNT(*) as student_count,
        GROUP_CONCAT(CONCAT(first_name, ' ', last_name) SEPARATOR ', ') as students
    FROM students
    WHERE academy_reference IN ('KINE', 'TEST')
    GROUP BY academy_reference
");

while ($row = $stmt->fetch()) {
    echo "{$row['academy_reference']} Academy:\n";
    echo "  Students: {$row['student_count']}\n";
    echo "  Names: {$row['students']}\n\n";
}

// Show payment summary
echo "Payment Summary:\n";
echo str_repeat("-", 60) . "\n";

$stmt = $pdo->query("
    SELECT 
        academy_reference,
        COUNT(*) as payment_count,
        SUM(amount) as total_amount,
        SUM(admin_earnings) as school_earnings,
        SUM(platform_fee) as platform_fees
    FROM payments
    WHERE academy_reference IN ('KINE', 'TEST') AND status = 'SUCCESSFUL'
    GROUP BY academy_reference
");

while ($row = $stmt->fetch()) {
    echo "{$row['academy_reference']} Academy:\n";
    echo "  Payments: {$row['payment_count']}\n";
    echo "  Total: E{$row['total_amount']}\n";
    echo "  School Earnings: E{$row['school_earnings']}\n";
    echo "  Platform Fees: E{$row['platform_fees']}\n\n";
}

echo "✅ Seeding complete!\n\n";

echo "Login Credentials for Test Students:\n";
echo str_repeat("-", 60) . "\n";
echo "Username: Will be auto-generated (check dashboard)\n";
echo "Password: [FirstName]123! (e.g., Alice123!)\n";
echo "\nTo test:\n";
echo "1. Go to: http://localhost/Multi-Tanent/select_school.php\n";
echo "2. Select 'Kine Academy' - see 5 students\n";
echo "3. Select 'Test School' - see different 5 students\n";
echo "4. Verify complete data isolation!\n\n";

