<?php
/**
 * Student Self-Registration
 * Public registration form - no login required
 * School-specific URL: register_student.php?school=KINE
 */

session_start();
require_once 'includes/functions.php';
require_once 'includes/email_functions.php';
require_once 'includes/url_helper.php';
require_once 'includes/favicon_helper.php';

// Generate CSRF token if not exists
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$pdo = getDB();
$error = '';
$success = false;

// Get school from URL parameter or clean URL path
$school_code = trim($_GET['school'] ?? '');

// If not in query string, try to get from clean URL path
// Clean URL format: /school-code/register
if (empty($school_code)) {
    $uri = $_SERVER['REQUEST_URI'] ?? '';
    // Extract school code from path like /Multi-Tanent/kine/register
    if (preg_match('#/([a-zA-Z0-9_-]+)/register#', $uri, $matches)) {
        $school_code = strtoupper($matches[1]);
    }
}

if (empty($school_code)) {
    die("Error: School code not specified. Please use the registration link provided by your school.");
}

// Validate school exists and is active
$stmt = $pdo->prepare("
    SELECT reference_code, academy_name, contact_email, is_active 
    FROM academy_references 
    WHERE reference_code = ?
");
$stmt->execute([$school_code]);
$school = $stmt->fetch();

if (!$school) {
    die("Error: Invalid school code. Please check your registration link.");
}

if (!$school['is_active']) {
    die("Error: This school is currently not accepting registrations. Please contact the school administration.");
}

// Get school settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$school_code]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Check if self-registration is enabled
$self_registration_enabled = ($settings['student_self_registration'] ?? '1') == '1';
if (!$self_registration_enabled) {
    die("Error: Student self-registration is not enabled for this school. Please contact the school administration.");
}

// Get available grades for this school
$stmt = $pdo->prepare("
    SELECT g.* FROM grades g
    INNER JOIN school_grades sg ON g.id = sg.grade_id
    WHERE sg.academy_reference = ? AND sg.is_active = 1
    ORDER BY g.name
");
$stmt->execute([$school_code]);
$available_grades = $stmt->fetchAll();

// If no school_grades configured, show all grades (backward compatibility)
if (empty($available_grades)) {
    $available_grades = getGrades();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_application'])) {
    // Collect data
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $date_of_birth = trim($_POST['date_of_birth'] ?? '');
    $gender = $_POST['gender'] ?? '';
    $grade_id = intval($_POST['grade_id'] ?? 0);
    $subject_ids = $_POST['subjects'] ?? [];
    
    $parent_name = trim($_POST['parent_name'] ?? '');
    $parent_email = trim($_POST['parent_email'] ?? '');
    $parent_phone = trim($_POST['parent_phone'] ?? '');
    $parent_relationship = $_POST['parent_relationship'] ?? 'parent';
    
    // Validation
    if (empty($first_name) || empty($last_name) || empty($email) || empty($grade_id)) {
        $error = "Please fill in all required fields.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "Please enter a valid email address.";
    } else {
        // Check minimum subjects requirement (configurable by admin)
        $min_subjects = intval($settings['minimum_required_subjects'] ?? 0);
        if ($min_subjects > 0 && count($subject_ids) < $min_subjects) {
            $error = "Please select at least {$min_subjects} subject(s).";
        } else {
        try {
            // Check email not already registered for this school
            $stmt = $pdo->prepare("
                SELECT id FROM students WHERE email = ? AND academy_reference = ?
            ");
            $stmt->execute([$email, $school_code]);
            if ($stmt->fetch()) {
                $error = "This email is already registered for this school.";
            } else {
                // Check pending applications
                $stmt = $pdo->prepare("
                    SELECT id FROM student_applications 
                    WHERE email = ? AND academy_reference = ? 
                    AND application_status NOT IN ('rejected', 'completed')
                ");
                $stmt->execute([$email, $school_code]);
                if ($stmt->fetch()) {
                    $error = "You already have a pending application. Please check your email.";
                } else {
                    // Calculate fees
                    $subject_price = floatval($settings['subject_fee'] ?? 350);
                    $total_amount = count($subject_ids) * $subject_price;
                    
                    // Generate verification token
                    $verification_token = bin2hex(random_bytes(32));
                    
                    // Create application
                    $pdo->beginTransaction();
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO student_applications 
                        (academy_reference, first_name, last_name, full_name, email, phone, 
                         date_of_birth, gender, grade_id, parent_name, parent_email, parent_phone,
                         parent_relationship, payment_amount, email_verification_token, 
                         application_status, payment_status, submitted_at)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'submitted', 'unpaid', NOW())
                    ");
                    
                    $full_name = $first_name . ' ' . $last_name;
                    
                    $stmt->execute([
                        $school_code,
                        $first_name,
                        $last_name,
                        $full_name,
                        $email,
                        $phone,
                        $date_of_birth ?: null,
                        $gender ?: null,
                        $grade_id,
                        $parent_name,
                        $parent_email,
                        $parent_phone,
                        $parent_relationship,
                        $total_amount,
                        $verification_token
                    ]);
                    
                    $application_id = $pdo->lastInsertId();
                    
                    // Save subject selections
                    $stmt = $pdo->prepare("
                        INSERT INTO student_application_subjects (application_id, subject_id)
                        VALUES (?, ?)
                    ");
                    foreach ($subject_ids as $subject_id) {
                        $stmt->execute([$application_id, intval($subject_id)]);
                    }
                    
                    $pdo->commit();
                    
                    // Generate verification code for SMS
                    require_once 'includes/sms_functions.php';
                    $verification_code = generateVerificationCode(6);
                    
                    // Try SMS first if enabled
                    $sms_sent = false;
                    $email_sent = false;
                    
                    if (defined('SMS_ENABLED') && SMS_ENABLED) {
                        try {
                            $sms_result = sendVerificationSMS(
                                $phone, 
                                $first_name, 
                                $verification_code,
                                $school['academy_name']
                            );
                            $sms_sent = $sms_result['success'];
                            
                            if ($sms_sent) {
                                $_SESSION['verification_method'] = 'sms';
                                $_SESSION['verification_phone'] = $phone;
                                $_SESSION['verification_code'] = $verification_code;
                                error_log("Verification SMS sent successfully to: " . $phone);
                            } else {
                                error_log("Failed to send SMS: " . ($sms_result['error'] ?? 'Unknown error'));
                            }
                        } catch (Exception $sms_exception) {
                            error_log("SMS sending exception: " . $sms_exception->getMessage());
                        }
                    }
                    
                    // Fallback to email if SMS fails (or SMS is disabled)
                    if (!$sms_sent) {
                        // Send verification email
                        // Use smart URL detection (auto-detects localhost vs production)
                        $verification_link = generateURL('verify_registration.php') . "?token=" . urlencode($verification_token);
                        
                        $email_subject = "Verify Your Registration - " . $school['academy_name'];
                        
                        // Create HTML email with improved mobile compatibility
                        $email_message = '
                    <!DOCTYPE html>
                    <html>
                    <head>
                        <meta charset="UTF-8">
                        <meta name="viewport" content="width=device-width, initial-scale=1.0">
                        <meta http-equiv="X-UA-Compatible" content="IE=edge">
                        <title>Email Verification</title>
                    </head>
                    <body style="margin:0;padding:0;font-family:Arial,sans-serif;background-color:#f4f4f4;">
                        <table role="presentation" style="width:100%;border-collapse:collapse;background-color:#f4f4f4;padding:20px 0;">
                            <tr>
                                <td align="center">
                                    <table role="presentation" style="max-width:600px;width:100%;background-color:#ffffff;border-radius:8px;box-shadow:0 2px 8px rgba(0,0,0,0.1);">
                                        <tr>
                                            <td style="padding:40px 30px;text-align:center;background:linear-gradient(135deg, #667eea 0%, #764ba2 100%);border-radius:8px 8px 0 0;">
                                                <h1 style="margin:0;color:#ffffff;font-size:28px;">Welcome to ' . htmlspecialchars($school['academy_name']) . '!</h1>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td style="padding:40px 30px;">
                                                <p style="font-size:16px;color:#333333;line-height:1.6;margin:0 0 20px;">Dear ' . htmlspecialchars($full_name) . ',</p>
                                                <p style="font-size:16px;color:#333333;line-height:1.6;margin:0 0 30px;">Thank you for registering! Please verify your email address to proceed with your application.</p>
                                                
                                                <table role="presentation" style="width:100%;margin:30px 0;">
                                                    <tr>
                                                        <td align="center">
                                                            <a href="' . $verification_link . '" style="display:inline-block;background-color:#0d6efd;color:#ffffff;font-size:16px;font-weight:bold;text-decoration:none;padding:15px 40px;border-radius:5px;text-align:center;">Verify Email & Continue</a>
                                                        </td>
                                                    </tr>
                                                </table>
                                                
                                                <p style="font-size:14px;color:#666666;margin:20px 0;text-align:center;">Or copy this link: <br><span style="color:#0d6efd;word-break:break-all;">' . $verification_link . '</span></p>
                                                
                                                <div style="background-color:#f8f9fa;padding:20px;border-radius:5px;margin:30px 0;">
                                                    <h3 style="margin:0 0 15px;color:#333333;font-size:18px;">Application Summary</h3>
                                                    <table style="width:100%;font-size:14px;color:#333333;">
                                                        <tr><td style="padding:5px 0;"><strong>School:</strong></td><td style="padding:5px 0;">' . htmlspecialchars($school['academy_name']) . '</td></tr>
                                                        <tr><td style="padding:5px 0;"><strong>Subjects:</strong></td><td style="padding:5px 0;">' . count($subject_ids) . '</td></tr>
                                                        <tr><td style="padding:5px 0;"><strong>Total Fee:</strong></td><td style="padding:5px 0;">E' . number_format($total_amount, 2) . '</td></tr>
                                                    </table>
                                                </div>
                                                
                                                <p style="font-size:13px;color:#999999;margin:30px 0 0;text-align:center;">This link will expire in 48 hours. If you did not register, please ignore this email.</p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td style="padding:20px 30px;background-color:#f8f9fa;border-radius:0 0 8px 8px;text-align:center;">
                                                <p style="margin:0;font-size:12px;color:#666666;">&copy; ' . date('Y') . ' ' . htmlspecialchars($school['academy_name']) . '. All rights reserved.</p>
                                            </td>
                                        </tr>
                                    </table>
                                </td>
                            </tr>
                        </table>
                    </body>
                    </html>';
                    
                        // Try to send email using PHPMailer
                        $email_error = '';
                        
                        try {
                            $email_sent = sendEmail($email, $full_name, $email_subject, $email_message);
                            if ($email_sent) {
                                $_SESSION['verification_method'] = 'email';
                                error_log("Verification email sent successfully to: " . $email);
                            }
                        } catch (Exception $email_exception) {
                            $email_error = $email_exception->getMessage();
                            error_log("Email sending exception: " . $email_error);
                        }
                        
                        // Store verification link in session for fallback
                        $_SESSION['verification_link'] = $verification_link;
                        $_SESSION['verification_token'] = $verification_token;
                        
                        if (!$email_sent) {
                            error_log("Failed to send verification email to: " . $email . " Error: " . $email_error);
                            // Continue anyway - we'll show the verification link on success page
                        }
                    }
                    
                    // Store data in session
                    $_SESSION['application_id'] = $application_id;
                    $_SESSION['pending_application_id'] = $application_id;
                    $_SESSION['application_email'] = $email;
                    $_SESSION['sms_sent_status'] = $sms_sent;
                    $_SESSION['email_sent_status'] = $email_sent;
                    $_SESSION['student_name'] = $first_name;
                    $_SESSION['school_name'] = $school['academy_name'];
                    $_SESSION['verification_email'] = $email;
                    
                    if (!$sms_sent && !$email_sent) {
                        $_SESSION['verification_method'] = 'direct';
                        error_log("Both SMS and email failed - will show direct verification link");
                    }
                    
                    // Success
                    $success = true;
                }
            }
        } catch (Exception $e) {
            if (isset($pdo) && $pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $error = "Error submitting application: " . $e->getMessage();
            error_log("Registration error: " . $e->getMessage());
        }
        }
    }
}

// Get subjects for selected grade (AJAX loaded)
$selected_grade_id = intval($_GET['grade_id'] ?? 0);
$subjects_for_grade = [];
if ($selected_grade_id > 0) {
    $stmt = $pdo->prepare("
        SELECT s.id, s.name, s.code 
        FROM subjects s
        INNER JOIN grade_subject gs ON s.id = gs.subject_id
        WHERE gs.grade_id = ?
        ORDER BY s.name
    ");
    $stmt->execute([$selected_grade_id]);
    $subjects_for_grade = $stmt->fetchAll();
    
    // If AJAX request, return JSON
    if (isset($_GET['ajax'])) {
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'subjects' => $subjects_for_grade]);
        exit;
    }
}

// Get school logo and colors
$school_logo = $settings['school_logo'] ?? '../images/img-melanesys-logo.svg';
$primary_color = $settings['primary_color'] ?? '#667eea';
$secondary_color = $settings['secondary_color'] ?? '#764ba2';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Registration - <?= htmlspecialchars($school['academy_name']) ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="<?= htmlspecialchars($school_logo) ?>">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: <?= $primary_color ?>;
            --secondary: <?= $secondary_color ?>;
        }
        body {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            min-height: 100vh;
            padding: 20px 0;
        }
        .registration-container {
            max-width: 800px;
            margin: 0 auto;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        .card-header {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
            border-radius: 1rem 1rem 0 0 !important;
            padding: 2rem;
        }
        .school-logo {
            max-width: 100px;
            max-height: 100px;
            object-fit: contain;
        }
        .progress {
            height: 25px;
        }
        .progress-step {
            font-size: 0.85rem;
        }
        .subject-card {
            cursor: pointer;
            transition: all 0.3s;
            border: 2px solid #dee2e6;
        }
        .subject-card:hover {
            border-color: var(--primary);
            transform: scale(1.02);
        }
        .subject-card.selected {
            border-color: var(--primary);
            background: linear-gradient(135deg, rgba(102,126,234,0.1) 0%, rgba(118,75,162,0.1) 100%);
        }
        
        /* Mobile-friendly date dropdowns */
        @media (max-width: 768px) {
            .form-select {
                font-size: 16px; /* Prevents auto-zoom on iOS */
                padding: 12px 8px;
            }
        }
        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            border: none;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102,126,234,0.4);
        }
        .step-indicator {
            opacity: 0.5;
        }
        .step-indicator.active {
            opacity: 1;
            font-weight: bold;
        }
        .required-badge {
            background: #dc3545;
            color: white;
            font-size: 0.7rem;
            padding: 2px 6px;
            border-radius: 3px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="registration-container">
            
            <?php if ($success): ?>
            <!-- Success Message -->
            <div class="card">
                <div class="card-body text-center py-5">
                    <div class="mb-4">
                        <i class="bi bi-envelope-check display-1 text-success"></i>
                    </div>
                    <h2 class="text-success mb-3">Registration Submitted!</h2>
                    <p class="lead">Thank you for registering with <?= htmlspecialchars($school['academy_name']) ?>!</p>
                    
                    <?php 
                    $verification_method = $_SESSION['verification_method'] ?? 'email';
                    $sms_sent = $_SESSION['sms_sent_status'] ?? false;
                    $email_sent = $_SESSION['email_sent_status'] ?? false;
                    ?>
                    
                    <?php if ($verification_method === 'sms' && $sms_sent): ?>
                    <!-- SMS Verification -->
                    <div class="alert alert-success mt-4">
                        <i class="bi bi-phone-fill"></i>
                        <strong>SMS Sent!</strong>
                        <p class="mb-0">We've sent a 6-digit verification code to <strong><?= htmlspecialchars($_SESSION['verification_phone'] ?? '') ?></strong></p>
                    </div>
                    
                    <div class="card mt-3">
                        <div class="card-body">
                            <h5 class="card-title text-center mb-3">Enter Verification Code</h5>
                            <form action="verify_sms_code.php" method="POST">
                                <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?? '' ?>">
                                <input type="hidden" name="phone" value="<?= htmlspecialchars($_SESSION['verification_phone'] ?? '') ?>">
                                
                                <div class="mb-3">
                                    <input type="text" name="code" class="form-control form-control-lg text-center" 
                                           style="font-size: 2em; letter-spacing: 0.5em; font-weight: bold;" 
                                           maxlength="6" pattern="[0-9]{6}" 
                                           placeholder="000000" required autofocus>
                                    <small class="text-muted d-block text-center mt-2">Valid for <?= defined('SMS_VERIFICATION_EXPIRY') ? SMS_VERIFICATION_EXPIRY : 15 ?> minutes</small>
                                </div>
                                
                                <div class="d-grid gap-2">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <i class="bi bi-check-circle me-2"></i>Verify & Continue
                                    </button>
                                </div>
                            </form>
                            
                            <div class="text-center mt-3">
                                <a href="resend_verification.php?method=sms" class="btn btn-link">
                                    <i class="bi bi-arrow-clockwise me-1"></i>Didn't receive code? Resend SMS
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <?php elseif ($verification_method === 'email' && $email_sent): ?>
                    <!-- Email Verification -->
                    <div class="alert alert-success mt-4">
                        <h5><i class="bi bi-envelope-check me-2"></i>Email Sent!</h5>
                        <p class="mb-0">We've sent a verification link to <strong><?= htmlspecialchars($_SESSION['application_email']) ?></strong></p>
                    </div>
                    
                    <div class="alert alert-info mt-4">
                        <h5><i class="bi bi-info-circle me-2"></i>Next Steps:</h5>
                        <ol class="text-start mb-0">
                            <li class="mb-2"><strong>Check your email</strong> (<?= htmlspecialchars($_SESSION['application_email']) ?>)</li>
                            <li class="mb-2"><strong>Click the verification link</strong> we sent you</li>
                            <li class="mb-2"><strong>Complete payment</strong> to activate your account</li>
                            <li><strong>Receive your login credentials</strong> via email</li>
                        </ol>
                    </div>
                    
                    <p class="text-muted mt-4">
                        <small>Didn't receive the email? Check your spam folder or <a href="<?= htmlspecialchars($_SESSION['verification_link'] ?? '#') ?>" class="text-decoration-none">click here to verify directly</a></small>
                    </p>
                    
                    <?php else: ?>
                    <!-- Direct Link Fallback (SMS and Email both failed) -->
                    <div class="alert alert-warning mt-4">
                        <h5><i class="bi bi-exclamation-triangle me-2"></i>Verification Link</h5>
                        <p class="mb-3">We couldn't send the verification via SMS or email, but you can verify your account directly:</p>
                        <div class="d-grid gap-2 mb-3">
                            <a href="<?= htmlspecialchars($_SESSION['verification_link'] ?? '#') ?>" class="btn btn-primary btn-lg">
                                <i class="bi bi-check-circle me-2"></i>Verify & Continue to Payment
                            </a>
                        </div>
                        <small class="text-muted">
                            Or copy this link:<br>
                            <code style="word-break: break-all; font-size: 0.85rem;"><?= htmlspecialchars($_SESSION['verification_link'] ?? '') ?></code>
                        </small>
                    </div>
                    
                    <div class="alert alert-info mt-4">
                        <h5><i class="bi bi-info-circle me-2"></i>Next Steps:</h5>
                        <ol class="text-start mb-0">
                            <li class="mb-2"><strong>Click the verification button above</strong> to proceed</li>
                            <li class="mb-2"><strong>Complete payment</strong> to activate your account</li>
                            <li><strong>Receive your login credentials</strong> via email/SMS</li>
                        </ol>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <?php else: ?>
            <!-- Registration Form -->
            <div class="card">
                <div class="card-header text-center">
                    <img src="<?= htmlspecialchars($school_logo) ?>" alt="Logo" class="school-logo mb-3">
                    <h2 class="mb-2">Student Registration</h2>
                    <h4 class="mb-0"><?= htmlspecialchars($school['academy_name']) ?></h4>
                </div>
                <div class="card-body p-4">
                    
                    <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-exclamation-triangle-fill me-2"></i><?= htmlspecialchars($error) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Progress Steps -->
                    <div class="mb-4">
                        <div class="d-flex justify-content-between mb-2">
                            <small class="step-indicator active"><i class="bi bi-1-circle-fill me-1"></i>Personal Info</small>
                            <small class="step-indicator"><i class="bi bi-2-circle me-1"></i>Academic Info</small>
                            <small class="step-indicator"><i class="bi bi-3-circle me-1"></i>Parent Info</small>
                            <small class="step-indicator"><i class="bi bi-4-circle me-1"></i>Review</small>
                        </div>
                        <div class="progress">
                            <div class="progress-bar progress-bar-striped progress-bar-animated" 
                                 style="width: 25%">Step 1 of 4</div>
                        </div>
                    </div>
                    
                    <form method="POST" id="registrationForm" class="needs-validation" novalidate>
                        
                        <!-- Step 1: Personal Information -->
                        <div class="registration-step" id="step1">
                            <h5 class="mb-3"><i class="bi bi-person-fill me-2"></i>Personal Information</h5>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">First Name <span class="text-danger">*</span></label>
                                    <input type="text" name="first_name" class="form-control" 
                                           value="<?= htmlspecialchars($_POST['first_name'] ?? '') ?>" required>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Last Name <span class="text-danger">*</span></label>
                                    <input type="text" name="last_name" class="form-control" 
                                           value="<?= htmlspecialchars($_POST['last_name'] ?? '') ?>" required>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Email Address <span class="text-danger">*</span></label>
                                <input type="email" name="email" class="form-control" 
                                       value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required>
                                <small class="text-muted">We'll send your login credentials to this email</small>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Phone Number</label>
                                    <div class="input-group">
                                        <span class="input-group-text">+268</span>
                                        <input type="tel" name="phone" class="form-control" 
                                               placeholder="7687 3207"
                                               value="<?= htmlspecialchars($_POST['phone'] ?? '') ?>">
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Date of Birth</label>
                                    <?php
                                    // Parse existing date if available
                                    $existing_dob = $_POST['date_of_birth'] ?? '';
                                    $selected_year = '';
                                    $selected_month = '';
                                    $selected_day = '';
                                    
                                    if (!empty($existing_dob)) {
                                        $date_parts = explode('-', $existing_dob);
                                        if (count($date_parts) == 3) {
                                            $selected_year = $date_parts[0];
                                            $selected_month = $date_parts[1];
                                            $selected_day = $date_parts[2];
                                        }
                                    }
                                    
                                    $current_year = date('Y');
                                    $min_year = $current_year - 35; // Max 25 years old
                                    $max_year = $current_year - 5;  // Min 5 years old
                                    ?>
                                    
                                    <!-- Mobile-friendly dropdown selectors -->
                                    <div class="row g-2">
                                        <div class="col-4">
                                            <select class="form-select" id="dob_day" onchange="updateDOBField()">
                                                <option value="">Day</option>
                                                <?php for ($d = 1; $d <= 31; $d++): ?>
                                                    <option value="<?= sprintf('%02d', $d) ?>" <?= $selected_day == sprintf('%02d', $d) ? 'selected' : '' ?>>
                                                        <?= $d ?>
                                                    </option>
                                                <?php endfor; ?>
                                            </select>
                                        </div>
                                        <div class="col-4">
                                            <select class="form-select" id="dob_month" onchange="updateDOBField()">
                                                <option value="">Month</option>
                                                <?php 
                                                $months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
                                                for ($m = 1; $m <= 12; $m++): 
                                                ?>
                                                    <option value="<?= sprintf('%02d', $m) ?>" <?= $selected_month == sprintf('%02d', $m) ? 'selected' : '' ?>>
                                                        <?= $months[$m-1] ?>
                                                    </option>
                                                <?php endfor; ?>
                                            </select>
                                        </div>
                                        <div class="col-4">
                                            <select class="form-select" id="dob_year" onchange="updateDOBField()">
                                                <option value="">Year</option>
                                                <?php for ($y = $max_year; $y >= $min_year; $y--): ?>
                                                    <option value="<?= $y ?>" <?= $selected_year == $y ? 'selected' : '' ?>>
                                                        <?= $y ?>
                                                    </option>
                                                <?php endfor; ?>
                                            </select>
                                        </div>
                                    </div>
                                    
                                    <!-- Hidden field that stores the actual date value -->
                                    <input type="hidden" name="date_of_birth" id="date_of_birth" value="<?= htmlspecialchars($existing_dob) ?>">
                                    
                                    <small class="text-muted">Select day, month, and year from dropdowns</small>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Gender</label>
                                <div class="btn-group w-100" role="group">
                                    <input type="radio" class="btn-check" name="gender" value="male" id="male">
                                    <label class="btn btn-outline-primary" for="male">
                                        <i class="bi bi-person me-1"></i>Male
                                    </label>
                                    
                                    <input type="radio" class="btn-check" name="gender" value="female" id="female">
                                    <label class="btn btn-outline-primary" for="female">
                                        <i class="bi bi-person-dress me-1"></i>Female
                                    </label>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Step 2: Academic Information -->
                        <div class="registration-step d-none" id="step2">
                            <h5 class="mb-3"><i class="bi bi-book-fill me-2"></i>Academic Information</h5>
                            
                            <div class="mb-3">
                                <label class="form-label">Select Your Grade <span class="text-danger">*</span></label>
                                <select name="grade_id" id="grade_select" class="form-select form-select-lg" 
                                        onchange="loadSubjects(this.value)" required>
                                    <option value="">-- Choose your grade level --</option>
                                    <?php foreach ($available_grades as $grade): ?>
                                    <option value="<?= $grade['id'] ?>" 
                                            <?= (isset($_POST['grade_id']) && $_POST['grade_id'] == $grade['id']) ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($grade['name']) ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div id="subjects_container" class="d-none">
                                <label class="form-label">
                                    Select Your Subjects <?= ($settings['minimum_required_subjects'] ?? 0) > 0 ? '<span class="text-danger">*</span>' : '' ?>
                                    <?php 
                                    $min_subjects = intval($settings['minimum_required_subjects'] ?? 0);
                                    if ($min_subjects > 0):
                                    ?>
                                    <span class="badge bg-info">Minimum <?= $min_subjects ?> subject<?= $min_subjects > 1 ? 's' : '' ?></span>
                                    <?php else: ?>
                                    <span class="badge bg-secondary">Optional</span>
                                    <?php endif; ?>
                                </label>
                                <div id="subjects_loading" class="text-center py-4 d-none">
                                    <div class="spinner-border text-primary"></div>
                                    <p class="mt-2">Loading subjects...</p>
                                </div>
                                <div id="subjects_list" class="row g-2"></div>
                                <div class="mt-2">
                                    <small class="text-muted">
                                        <i class="bi bi-info-circle me-1"></i>
                                        Selected: <strong id="subject_count">0</strong> subjects | 
                                        Fee: <strong id="total_fee">E0.00</strong>
                                    </small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Step 3: Parent/Guardian Information -->
                        <div class="registration-step d-none" id="step3">
                            <h5 class="mb-3"><i class="bi bi-people-fill me-2"></i>Parent/Guardian Information</h5>
                            
                            <div class="mb-3">
                                <label class="form-label">Parent/Guardian Full Name <span class="text-danger">*</span></label>
                                <input type="text" name="parent_name" class="form-control" 
                                       value="<?= htmlspecialchars($_POST['parent_name'] ?? '') ?>" required>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Parent Email <span class="text-danger">*</span></label>
                                    <input type="email" name="parent_email" class="form-control" 
                                           value="<?= htmlspecialchars($_POST['parent_email'] ?? '') ?>" required>
                                    <small class="text-muted">For payment notifications</small>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Parent Phone (MoMo) <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <span class="input-group-text">+268</span>
                                        <input type="tel" name="parent_phone" class="form-control" 
                                               placeholder="7687 3207"
                                               value="<?= htmlspecialchars($_POST['parent_phone'] ?? '') ?>" required>
                                    </div>
                                    <small class="text-muted">For MoMo payment</small>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Relationship</label>
                                <select name="parent_relationship" class="form-select">
                                    <option value="parent">Parent</option>
                                    <option value="father">Father</option>
                                    <option value="mother">Mother</option>
                                    <option value="guardian">Guardian</option>
                                    <option value="grandparent">Grandparent</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>
                        </div>
                        
                        <!-- Step 4: Review & Submit -->
                        <div class="registration-step d-none" id="step4">
                            <h5 class="mb-3"><i class="bi bi-check2-square me-2"></i>Review Your Information</h5>
                            
                            <div class="card bg-light mb-3">
                                <div class="card-body">
                                    <h6 class="card-title">Student Information</h6>
                                    <div id="review_student"></div>
                                </div>
                            </div>
                            
                            <div class="card bg-light mb-3">
                                <div class="card-body">
                                    <h6 class="card-title">Academic Information</h6>
                                    <div id="review_academic"></div>
                                </div>
                            </div>
                            
                            <div class="card bg-light mb-3">
                                <div class="card-body">
                                    <h6 class="card-title">Parent/Guardian Information</h6>
                                    <div id="review_parent"></div>
                                </div>
                            </div>
                            
                            <div class="card bg-primary text-white">
                                <div class="card-body">
                                    <div class="row align-items-center">
                                        <div class="col-md-8">
                                            <h5 class="mb-0">Total Registration Fee</h5>
                                            <small>Payable after email verification</small>
                                        </div>
                                        <div class="col-md-4 text-end">
                                            <h2 class="mb-0" id="review_total_fee">E0.00</h2>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="form-check mt-3">
                                <input class="form-check-input" type="checkbox" id="agree_terms" required>
                                <label class="form-check-label" for="agree_terms">
                                    I confirm that the information provided is accurate and I agree to the 
                                    <a href="#" target="_blank">terms and conditions</a>.
                                </label>
                            </div>
                        </div>
                        
                        <!-- Navigation Buttons -->
                        <div class="d-flex justify-content-between mt-4">
                            <button type="button" class="btn btn-secondary" id="prevBtn" onclick="changeStep(-1)" style="display:none;">
                                <i class="bi bi-arrow-left me-2"></i>Previous
                            </button>
                            <div class="ms-auto">
                                <button type="button" class="btn btn-primary" id="nextBtn" onclick="changeStep(1)">
                                    Next <i class="bi bi-arrow-right ms-2"></i>
                                </button>
                                <button type="submit" name="submit_application" class="btn btn-success d-none" id="submitBtn">
                                    <i class="bi bi-send me-2"></i>Submit Application
                                </button>
                            </div>
                        </div>
                    </form>
                    
                    <div class="text-center mt-4">
                        <small class="text-muted">
                            Already registered? <a href="admin/login.php">Login here</a>
                        </small>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const totalSteps = 4;
        let currentStep = 1;
        const subjectFee = <?= floatval($settings['subject_fee'] ?? 350) ?>;
        const minimumRequiredSubjects = <?= intval($settings['minimum_required_subjects'] ?? 0) ?>;
        
        // Load subjects for selected grade
        async function loadSubjects(gradeId) {
            if (!gradeId) {
                document.getElementById('subjects_container').classList.add('d-none');
                return;
            }
            
            document.getElementById('subjects_container').classList.remove('d-none');
            document.getElementById('subjects_loading').classList.remove('d-none');
            document.getElementById('subjects_list').innerHTML = '';
            
            try {
                const response = await fetch(`?school=<?= urlencode($school_code) ?>&grade_id=${gradeId}&ajax=1`);
                const data = await response.json();
                
                if (data.success && data.subjects.length > 0) {
                    let html = '';
                    data.subjects.forEach(subject => {
                        // No hardcoded required subjects - admin controls this via settings
                        html += `
                            <div class="col-md-6">
                                <div class="subject-card card p-2" 
                                     onclick="toggleSubject(this, ${subject.id})">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" 
                                               name="subjects[]" value="${subject.id}"
                                               id="subject_${subject.id}"
                                               onchange="updateFee()">
                                        <label class="form-check-label w-100" for="subject_${subject.id}">
                                            <strong>${subject.name}</strong>
                                            <br><small class="text-muted">${subject.code}</small>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        `;
                    });
                    document.getElementById('subjects_list').innerHTML = html;
                    updateFee();
                } else {
                    document.getElementById('subjects_list').innerHTML = 
                        '<div class="alert alert-warning">No subjects available for this grade.</div>';
                }
            } catch (error) {
                document.getElementById('subjects_list').innerHTML = 
                    '<div class="alert alert-danger">Error loading subjects. Please try again.</div>';
            } finally {
                document.getElementById('subjects_loading').classList.add('d-none');
            }
        }
        
        // Toggle subject selection
        function toggleSubject(card, subjectId) {
            const checkbox = card.querySelector('input[type="checkbox"]');
            if (!checkbox.disabled) {
                checkbox.checked = !checkbox.checked;
                if (checkbox.checked) {
                    card.classList.add('selected');
                } else {
                    card.classList.remove('selected');
                }
                updateFee();
            }
        }
        
        // Update fee calculation
        function updateFee() {
            const checked = document.querySelectorAll('input[name="subjects[]"]:checked').length;
            const total = checked * subjectFee;
            
            document.getElementById('subject_count').textContent = checked;
            document.getElementById('total_fee').textContent = 'E' + total.toFixed(2);
            document.getElementById('review_total_fee').textContent = 'E' + total.toFixed(2);
        }
        
        // Update date of birth field from dropdowns
        function updateDOBField() {
            const day = document.getElementById('dob_day').value;
            const month = document.getElementById('dob_month').value;
            const year = document.getElementById('dob_year').value;
            
            if (day && month && year) {
                // Combine into YYYY-MM-DD format
                const dateValue = year + '-' + month + '-' + day;
                document.getElementById('date_of_birth').value = dateValue;
            } else {
                document.getElementById('date_of_birth').value = '';
            }
        }
        
        // Multi-step form navigation
        function changeStep(direction) {
            // Validate current step before proceeding
            if (direction === 1 && !validateStep(currentStep)) {
                return;
            }
            
            // Hide current step
            document.getElementById('step' + currentStep).classList.add('d-none');
            
            // Update step
            currentStep += direction;
            
            // Show new step
            if (currentStep === 4) {
                updateReview();
            }
            document.getElementById('step' + currentStep).classList.remove('d-none');
            
            // Update buttons
            document.getElementById('prevBtn').style.display = currentStep === 1 ? 'none' : 'block';
            document.getElementById('nextBtn').style.display = currentStep === 4 ? 'none' : 'block';
            document.getElementById('submitBtn').classList.toggle('d-none', currentStep !== 4);
            
            // Update progress
            updateProgress();
            
            // Scroll to top
            window.scrollTo({top: 0, behavior: 'smooth'});
        }
        
        // Validate current step
        function validateStep(step) {
            const stepDiv = document.getElementById('step' + step);
            const inputs = stepDiv.querySelectorAll('input[required], select[required]');
            
            let valid = true;
            inputs.forEach(input => {
                if (!input.value && !input.disabled) {
                    input.classList.add('is-invalid');
                    valid = false;
                } else {
                    input.classList.remove('is-invalid');
                }
            });
            
            // Special validation for step 2 (subjects)
            if (step === 2 && minimumRequiredSubjects > 0) {
                const selectedSubjects = document.querySelectorAll('input[name="subjects[]"]:checked').length;
                if (selectedSubjects < minimumRequiredSubjects) {
                    alert(`Please select at least ${minimumRequiredSubjects} subject${minimumRequiredSubjects > 1 ? 's' : ''}.`);
                    valid = false;
                }
            }
            
            return valid;
        }
        
        // Update progress bar
        function updateProgress() {
            const percent = (currentStep / totalSteps) * 100;
            const progressBar = document.querySelector('.progress-bar');
            progressBar.style.width = percent + '%';
            progressBar.textContent = `Step ${currentStep} of ${totalSteps}`;
            
            // Update step indicators
            document.querySelectorAll('.step-indicator').forEach((indicator, index) => {
                if (index + 1 <= currentStep) {
                    indicator.classList.add('active');
                } else {
                    indicator.classList.remove('active');
                }
            });
        }
        
        // Update review section
        function updateReview() {
            const form = document.getElementById('registrationForm');
            const formData = new FormData(form);
            
            // Student info
            const dobValue = formData.get('date_of_birth');
            let dobDisplay = '';
            if (dobValue) {
                // Format date nicely
                const dobDate = new Date(dobValue);
                const options = { year: 'numeric', month: 'long', day: 'numeric' };
                dobDisplay = '<p><strong>Date of Birth:</strong> ' + dobDate.toLocaleDateString('en-US', options) + '</p>';
            }
            
            document.getElementById('review_student').innerHTML = `
                <p><strong>Name:</strong> ${formData.get('first_name')} ${formData.get('last_name')}</p>
                <p><strong>Email:</strong> ${formData.get('email')}</p>
                <p><strong>Phone:</strong> +268 ${formData.get('phone')}</p>
                ${dobDisplay}
            `;
            
            // Academic info
            const gradeSelect = document.getElementById('grade_select');
            const gradeName = gradeSelect.options[gradeSelect.selectedIndex].text;
            const subjects = Array.from(document.querySelectorAll('input[name="subjects[]"]:checked'))
                .map(cb => cb.nextElementSibling.querySelector('strong').textContent);
            
            document.getElementById('review_academic').innerHTML = `
                <p><strong>Grade:</strong> ${gradeName}</p>
                <p><strong>Subjects (${subjects.length}):</strong></p>
                <ul class="mb-0">
                    ${subjects.map(s => '<li>' + s + '</li>').join('')}
                </ul>
            `;
            
            // Parent info
            document.getElementById('review_parent').innerHTML = `
                <p><strong>Name:</strong> ${formData.get('parent_name')}</p>
                <p><strong>Email:</strong> ${formData.get('parent_email')}</p>
                <p><strong>Phone:</strong> +268 ${formData.get('parent_phone')}</p>
                <p><strong>Relationship:</strong> ${formData.get('parent_relationship')}</p>
            `;
        }
    </script>
</body>
</html>

