<?php
require_once '../../config.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

// Validate CSRF token
if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
    http_response_code(403);
    echo json_encode(['error' => 'Invalid security token']);
    exit;
}

// Validate required parameters
$required_fields = ['student_id', 'amount', 'phone', 'description'];
foreach ($required_fields as $field) {
    if (empty($_POST[$field])) {
        http_response_code(400);
        echo json_encode(['error' => "Missing required field: $field"]);
        exit;
    }
}

$student_id = (int)$_POST['student_id'];
$amount = (float)$_POST['amount'];
$phone = trim($_POST['phone']);
$description = trim($_POST['description']);

// Validate phone number format (Swaziland format)
if (!preg_match('/^268[0-9]{8}$/', $phone)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid phone number format. Use format: 268XXXXXXXX']);
    exit;
}

// Validate amount
if ($amount <= 0 || $amount > 1000) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid amount. Must be between 0.01 and 1000']);
    exit;
}

try {
    $pdo = getDB();
    
    // Verify student exists
    $stmt = $pdo->prepare("SELECT id, first_name, last_name, email FROM students WHERE id = ?");
    $stmt->execute([$student_id]);
    $student = $stmt->fetch();
    
    if (!$student) {
        http_response_code(404);
        echo json_encode(['error' => 'Student not found']);
        exit;
    }
    
    // Generate unique references
    $referenceId = generateUuidV4();         // UUID for X-Reference-Id and status lookups
    $externalId = 'SK' . time() . rand(1000, 9999); // Business reference
    
    // Prepare payment request data
    // Include academy reference in payment message so admin knows payment source
    $payerMessage = ACADEMY_NAME . ' - ' . $description;
    $payeeNote = ACADEMY_NAME . ' - Payment from: ' . $student['first_name'] . ' ' . $student['last_name'];
    
    $payment_data = [
        'amount' => number_format($amount, 2, '.', ''),
        'currency' => MOMO_CURRENCY,
        'externalId' => $externalId,
        'payer' => [
            'partyIdType' => 'MSISDN',
            'partyId' => $phone
        ],
        'payerMessage' => $payerMessage,
        'payeeNote' => $payeeNote
    ];
    
    // MTN MoMo API request
    $api_url = MOMO_COLLECTION_BASE_URL . '/v1_0/requesttopay';
    $accessToken = getMomoAccessToken();
    $headers = [
        'Authorization: Bearer ' . $accessToken,
        'X-Reference-Id: ' . $referenceId,
        'X-Target-Environment: ' . MOMO_ENV,
        'Content-Type: application/json',
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payment_data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("CURL Error: $error");
    }
    
    // Log payment request (store both IDs)
    // Set academy reference - default to 'KINE'
    $academy_reference = DEFAULT_ACADEMY_REFERENCE;
    
    $stmt = $pdo->prepare("
        INSERT INTO payments (student_id, amount, currency, reference_id, external_id, msisdn, status, academy_reference, created_at) 
        VALUES (?, ?, ?, ?, ?, ?, 'PENDING', ?, NOW())
    ");
    $stmt->execute([
        $student_id,
        $amount,
        MOMO_CURRENCY,
        $referenceId,
        $externalId,
        $phone,
        $academy_reference
    ]);
    
    if ($http_code == 202) {
        // Payment request accepted
        echo json_encode([
            'success' => true,
            'reference_id' => $referenceId,
            'external_id' => $externalId,
            'message' => 'Payment request sent successfully. Please check your phone for the payment prompt.',
            'status' => 'PENDING'
        ]);
    } else {
        // Payment request failed
        $error_data = json_decode($response, true);
        $error_message = $error_data['message'] ?? 'Payment request failed';
        
        // Update payment status to failed
        $stmt = $pdo->prepare("UPDATE payments SET status = 'FAILED', error_message = ? WHERE reference_id = ?");
        $stmt->execute([$error_message, $referenceId]);
        
        http_response_code(400);
        echo json_encode([
            'error' => $error_message,
            'reference_id' => $referenceId,
            'external_id' => $externalId
        ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

// Function to get MTN MoMo access token
function getMomoAccessToken() {
    $token_url = MOMO_COLLECTION_BASE_URL . '/token/';
    $headers = [
        'Authorization: Basic ' . base64_encode(MOMO_API_USER_ID . ':' . MOMO_API_KEY),
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $token_url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200) {
        $data = json_decode($response, true);
        if (!empty($data['access_token'])) {
            return $data['access_token'];
        }
    }
    
    throw new Exception('Failed to get access token');
}
?>
