<?php
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/wallet_functions.php';

// Log callback for debugging
function logCallback($message) {
    $logFile = __DIR__ . '/../../logs/momo_callback.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Get callback data
$input = file_get_contents('php://input');
$headers = getallheaders();

logCallback("Callback received: " . $input);
logCallback("Headers: " . json_encode($headers));

// Parse callback data
$callbackData = json_decode($input, true);

if (!$callbackData) {
    logCallback("Invalid JSON in callback");
    http_response_code(400);
    echo json_encode(['error' => 'Invalid JSON']);
    exit;
}

// Extract payment information
$referenceId = $callbackData['referenceId'] ?? null;
$status = $callbackData['status'] ?? null;
$amount = $callbackData['amount'] ?? null;
$currency = $callbackData['currency'] ?? null;
$financialTransactionId = $callbackData['financialTransactionId'] ?? null;

logCallback("Parsed data - Reference: $referenceId, Status: $status, Amount: $amount");

if (!$referenceId) {
    logCallback("No reference ID in callback");
    http_response_code(400);
    echo json_encode(['error' => 'No reference ID']);
    exit;
}

// Find payment record
$pdo = getDB();
$stmt = $pdo->prepare('SELECT * FROM payments WHERE reference_id = ? LIMIT 1');
$stmt->execute([$referenceId]);
$payment = $stmt->fetch();

if (!$payment) {
    logCallback("Payment not found for reference: $referenceId");
    http_response_code(404);
    echo json_encode(['error' => 'Payment not found']);
    exit;
}

logCallback("Found payment ID: " . $payment['id'] . " with status: " . $payment['status']);

// Update payment status
$newStatus = 'FAILED';
if ($status === 'SUCCESSFUL') {
    $newStatus = 'SUCCESSFUL';
} elseif ($status === 'PENDING') {
    $newStatus = 'PENDING';
}

// Update payment record with raw response
$stmt = $pdo->prepare('UPDATE payments SET status = ?, updated_at = NOW(), raw_response = ? WHERE id = ?');
$stmt->execute([$newStatus, $input, $payment['id']]);

logCallback("Updated payment status to: $newStatus");

// If payment is successful, update student status
if ($newStatus === 'SUCCESSFUL') {
    try {
        // Update student payment status
        $stmt = $pdo->prepare("
            UPDATE students 
            SET payment_status = 'PAID', 
                payment_date = NOW(),
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$payment['student_id']]);
        
        logCallback("Student payment status updated for student: " . $payment['student_id']);
        
        // Credit admin wallet (deduct platform fee)
        $academy_reference = $payment['academy_reference'] ?? DEFAULT_ACADEMY_REFERENCE;
        $wallet_result = creditAdminWallet($payment['id'], PLATFORM_ADMIN_ID, $academy_reference);
        
        if ($wallet_result['success']) {
            logCallback("Admin wallet credited successfully. Reference: " . $academy_reference);
        } else {
            logCallback("Failed to credit admin wallet: " . $wallet_result['error']);
        }
        
        // Send success email if function exists
        if (function_exists('sendPaymentSuccessEmail')) {
            $student = getStudentById($payment['student_id']);
            if ($student) {
                sendPaymentSuccessEmail($student['email'], $student['first_name'], $payment['amount'], MOMO_CURRENCY, [], $referenceId);
            }
        }
        
        logCallback("Payment success processing completed");
        
    } catch (Exception $e) {
        logCallback("Error processing successful payment: " . $e->getMessage());
    }
} else {
    // Send failure email if function exists
    if (function_exists('sendPaymentFailureEmail')) {
        $student = getStudentById($payment['student_id']);
        if ($student) {
            sendPaymentFailureEmail($student['email'], $student['first_name'], $payment['amount'], MOMO_CURRENCY, $referenceId);
        }
    }
    
    logCallback("Payment failed, failure processing completed");
}

// Respond to MTN MoMo
http_response_code(200);
echo json_encode(['status' => 'received']);

logCallback("Callback processing completed");
?>