<?php
/**
 * Card Payment Request Handler (PayGate Integration)
 * Initiates credit/debit card payment via PayGate
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/functions.php';

startSession();

// CSRF protection
if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
    http_response_code(403);
    die(json_encode(['error' => 'Invalid security token']));
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('Invalid request method');
}

// Get and validate inputs
$student_id = intval($_POST['student_id'] ?? 0);
$amount = floatval($_POST['amount'] ?? 0);
$description = trim($_POST['description'] ?? 'Kine Academy - Student Registration');

if ($student_id <= 0 || $amount <= 0) {
    die('Invalid payment parameters');
}

try {
    $pdo = getDB();
    
    // Get student details
    $stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
    $stmt->execute([$student_id]);
    $student = $stmt->fetch();
    
    if (!$student) {
        die('Student not found');
    }
    
    // Generate unique references
    $reference = 'KINE_CARD_' . $student_id . '_' . time();
    $external_id = 'SK_CARD_' . time() . rand(1000, 9999);
    
    // Set academy reference
    $academy_reference = DEFAULT_ACADEMY_REFERENCE;
    
    // Create payment record
    $stmt = $pdo->prepare("
        INSERT INTO payments 
        (student_id, amount, currency, reference_id, external_id, msisdn, status, payment_method, academy_reference, created_at) 
        VALUES (?, ?, 'SZL', ?, ?, '', 'PENDING', 'CARD', ?, NOW())
    ");
    $stmt->execute([
        $student_id,
        $amount,
        $reference,
        $external_id,
        $academy_reference
    ]);
    $payment_id = $pdo->lastInsertId();
    
    // Prepare PayGate request (new API format)
    $return_url = PUBLIC_BASE_URL . '/payments/card/return.php';
    $notify_url = PUBLIC_BASE_URL . '/payments/card/notify.php';
    
    $paygate_data = [
        'PAYGATE_ID' => PAYGATE_ID,
        'REFERENCE' => $reference,
        'AMOUNT' => number_format($amount * 100, 0, '', ''), // Amount in cents (SZL 3.00 = 300)
        'CURRENCY' => 'SZL',
        'RETURN_URL' => $return_url,
        'TRANSACTION_DATE' => date('Y-m-d H:i:s'),
        'LOCALE' => 'en-sz',
        'COUNTRY' => 'SZL',
        'EMAIL' => $student['email'],
        'NOTIFY_URL' => $notify_url,
        'USER1' => $student_id,
        'USER3' => ACADEMY_NAME . ' - ' . $description
    ];
    
    // Generate checksum using new format
    $checksum_string = '';
    foreach ($paygate_data as $key => $value) {
        $checksum_string .= $value;
    }
    $checksum_string .= PAYGATE_SECRET;
    $paygate_data['CHECKSUM'] = md5($checksum_string);
    
    // Initialize PayGate transaction
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, PAYGATE_URL);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($paygate_data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("CURL Error: $error");
    }
    
    // Parse response
    parse_str($response, $response_data);
    
    if (isset($response_data['PAY_REQUEST_ID']) && isset($response_data['CHECKSUM'])) {
        // Verify checksum
        $checksum_verify = '';
        foreach ($response_data as $key => $value) {
            if ($key !== 'CHECKSUM') {
                $checksum_verify .= $value;
            }
        }
        $checksum_verify .= PAYGATE_SECRET;
        
        if (md5($checksum_verify) !== $response_data['CHECKSUM']) {
            throw new Exception('Invalid checksum from PayGate');
        }
        
        // Update payment with PayGate reference
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET paygate_reference = ?,
                raw_response = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $response_data['PAY_REQUEST_ID'],
            json_encode($response_data),
            $payment_id
        ]);
        
        // Redirect to PayGate payment page
        $payment_url = 'https://secure.paygate.co.za/payweb3/process.trans?PAY_REQUEST_ID=' . urlencode($response_data['PAY_REQUEST_ID']);
        
        header('Location: ' . $payment_url);
        exit;
        
    } else {
        throw new Exception('Failed to initialize PayGate payment: ' . $response);
    }
    
} catch (Exception $e) {
    error_log("Card payment error: " . $e->getMessage());
    die('Payment initialization failed: ' . htmlspecialchars($e->getMessage()));
}
?>

