<?php
/**
 * Payment Status Page
 * Shows real-time payment status for reminder-based payments
 */

session_start();
require_once 'config.php';
require_once 'includes/functions.php';
require_once 'includes/url_helper.php';
require_once 'includes/favicon_helper.php';

$pdo = getDB();
$reference = $_GET['ref'] ?? '';

if (empty($reference)) {
    header('Location: ' . generateURL('index.php'));
    exit;
}

// Get transaction
$stmt = $pdo->prepare("
    SELECT pt.*, s.full_name, s.email, ar.academy_name
    FROM payment_transactions pt
    INNER JOIN students s ON pt.student_id = s.id
    INNER JOIN academy_references ar ON pt.academy_reference = ar.reference_code
    WHERE pt.transaction_reference = ?
");
$stmt->execute([$reference]);
$transaction = $stmt->fetch();

if (!$transaction) {
    die("Transaction not found");
}

// Check current status via API if pending
if ($transaction['status'] === 'pending' || $transaction['status'] === 'processing') {
    // Check MTN MoMo status
    if ($transaction['payment_method'] === 'momo' && $transaction['external_reference']) {
        try {
            require_once 'includes/functions.php';
            $status_result = checkMTNMoMoPaymentStatus($transaction['external_reference']);
            
            if ($status_result['status'] === 'SUCCESSFUL') {
                // Update transaction
                $pdo->prepare("
                    UPDATE payment_transactions 
                    SET status = 'completed', paid_at = NOW()
                    WHERE id = ?
                ")->execute([$transaction['id']]);
                
                // Process payment success
                require_once 'includes/payment_reminder_manager.php';
                $manager = new PaymentReminderManager();
                $manager->processPaymentSuccess($transaction['id']);
                
                $transaction['status'] = 'completed';
            } elseif ($status_result['status'] === 'FAILED') {
                $pdo->prepare("
                    UPDATE payment_transactions 
                    SET status = 'failed', failed_at = NOW(), error_message = ?
                    WHERE id = ?
                ")->execute(['Payment failed', $transaction['id']]);
                
                $transaction['status'] = 'failed';
            }
        } catch (Exception $e) {
            error_log("Status check error: " . $e->getMessage());
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Status - <?= APP_NAME ?></title>
    <?php echoFavicon($transaction['academy_reference'] ?? null); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px 0;
        }
        .status-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .status-header {
            padding: 30px;
            text-align: center;
            color: white;
        }
        .status-pending { background: #ffc107; }
        .status-processing { background: #17a2b8; }
        .status-completed { background: #28a745; }
        .status-failed { background: #dc3545; }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="status-card">
                    
                    <?php if ($transaction['status'] === 'completed'): ?>
                    <!-- Success -->
                    <div class="status-header status-completed">
                        <i class="bi bi-check-circle" style="font-size: 80px;"></i>
                        <h2 class="mt-3">Payment Successful!</h2>
                    </div>
                    <div class="p-4">
                        <div class="alert alert-success">
                            <h5><i class="bi bi-check-circle-fill"></i> Payment Received</h5>
                            <p class="mb-0">Your payment of <strong>SZL <?= number_format($transaction['amount'], 2) ?></strong> has been received.</p>
                        </div>
                        
                        <div class="mb-3">
                            <strong>Transaction Details:</strong><br>
                            Reference: <?= htmlspecialchars($transaction['transaction_reference']) ?><br>
                            Date: <?= date('F j, Y g:i A', strtotime($transaction['paid_at'])) ?><br>
                            Method: <?= strtoupper($transaction['payment_method']) ?>
                        </div>
                        
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i> Your account has been reactivated and is now in good standing.
                            You will receive a confirmation email shortly.
                        </div>
                        
                        <div class="d-grid">
                            <a href="<?= generateURL('index.php') ?>" class="btn btn-primary">
                                <i class="bi bi-house"></i> Return to Homepage
                            </a>
                        </div>
                    </div>
                    
                    <?php elseif ($transaction['status'] === 'pending' || $transaction['status'] === 'processing'): ?>
                    <!-- Pending -->
                    <div class="status-header status-processing">
                        <i class="bi bi-hourglass-split" style="font-size: 80px;"></i>
                        <h2 class="mt-3">Payment Processing</h2>
                    </div>
                    <div class="p-4 text-center">
                        <div class="alert alert-info">
                            <h5><i class="bi bi-info-circle"></i> Waiting for Payment</h5>
                            <p class="mb-0">Please check your phone for the payment prompt.</p>
                        </div>
                        
                        <div class="spinner-border text-primary my-4" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        
                        <p class="text-muted">This page will automatically refresh when payment is received.</p>
                        
                        <div class="d-grid gap-2 mt-4">
                            <button onclick="location.reload()" class="btn btn-outline-primary">
                                <i class="bi bi-arrow-clockwise"></i> Check Status Now
                            </button>
                        </div>
                    </div>
                    
                    <script>
                        // Auto-refresh every 5 seconds
                        setTimeout(function() {
                            location.reload();
                        }, 5000);
                    </script>
                    
                    <?php elseif ($transaction['status'] === 'failed'): ?>
                    <!-- Failed -->
                    <div class="status-header status-failed">
                        <i class="bi bi-x-circle" style="font-size: 80px;"></i>
                        <h2 class="mt-3">Payment Failed</h2>
                    </div>
                    <div class="p-4">
                        <div class="alert alert-danger">
                            <h5><i class="bi bi-exclamation-triangle-fill"></i> Payment Not Completed</h5>
                            <p class="mb-0">Your payment could not be processed.</p>
                        </div>
                        
                        <?php if ($transaction['error_message']): ?>
                        <p><strong>Reason:</strong> <?= htmlspecialchars($transaction['error_message']) ?></p>
                        <?php endif; ?>
                        
                        <div class="d-grid gap-2 mt-4">
                            <a href="pay_reminder.php?token=<?= urlencode($_SESSION['payment_token'] ?? '') ?>" class="btn btn-primary">
                                <i class="bi bi-arrow-counterclockwise"></i> Try Again
                            </a>
                        </div>
                    </div>
                    
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</body>
</html>

