<?php
/**
 * Payment Method Selection Page
 * Allows students to choose between Mobile Money and Card payment
 */

require_once 'config.php';
require_once 'includes/functions.php';

startSession();

// Get student info from session (after registration)
$student_id = $_SESSION['pending_payment_student_id'] ?? null;
$amount = $_SESSION['pending_payment_amount'] ?? 0;

if (!$student_id || !$amount) {
    header('Location: index.php');
    exit;
}

$pdo = getDB();
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

if (!$student) {
    header('Location: index.php');
    exit;
}

// Get subjects for this student
$subjects = getSubjectsForStudent($student_id);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Choose Payment Method - <?php echo APP_NAME; ?></title>
    
    <link rel="icon" type="image/svg+xml" href="images/img-melanesys-logo.svg">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        :root {
            --brand-primary: #6f42c1;
            --brand-accent: #20c997;
            --brand-soft: #f3f0ff;
        }
        
        body {
            background: linear-gradient(135deg, var(--brand-soft), #e8fff9);
            min-height: 100vh;
            padding: 2rem 0;
        }
        
        .payment-container {
            max-width: 900px;
            margin: 0 auto;
        }
        
        .payment-header {
            text-align: center;
            margin-bottom: 3rem;
        }
        
        .payment-header h1 {
            color: var(--brand-primary);
            font-weight: 700;
            margin-bottom: 1rem;
        }
        
        .payment-summary {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            margin-bottom: 3rem;
        }
        
        .payment-options {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 2rem;
        }
        
        .payment-card {
            background: white;
            border: 3px solid #e0e0e0;
            border-radius: 16px;
            padding: 2.5rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .payment-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 5px;
            background: linear-gradient(90deg, var(--brand-primary), var(--brand-accent));
            transform: scaleX(0);
            transition: transform 0.3s ease;
        }
        
        .payment-card:hover {
            border-color: var(--brand-primary);
            box-shadow: 0 12px 30px rgba(111, 66, 193, 0.2);
            transform: translateY(-8px);
        }
        
        .payment-card:hover::before {
            transform: scaleX(1);
        }
        
        .payment-icon {
            font-size: 4rem;
            color: var(--brand-primary);
            margin-bottom: 1rem;
        }
        
        .payment-card h3 {
            color: #2c3e50;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .payment-card p {
            color: #7f8c8d;
            margin-bottom: 1.5rem;
        }
        
        .payment-features {
            list-style: none;
            padding: 0;
            margin: 1.5rem 0;
        }
        
        .payment-features li {
            padding: 0.5rem 0;
            color: #34495e;
        }
        
        .payment-features i {
            color: var(--brand-accent);
            margin-right: 0.5rem;
        }
        
        .recommended-badge {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: var(--brand-accent);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 600;
        }
    </style>
</head>
<body>
    <div class="payment-container">
        <!-- Header -->
        <div class="payment-header">
            <h1>Choose Your Payment Method</h1>
            <p class="lead text-muted">Complete your registration by making a payment</p>
        </div>
        
        <!-- Payment Summary -->
        <div class="payment-summary">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h5>Payment Details</h5>
                    <p class="mb-1"><strong>Student:</strong> <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></p>
                    <p class="mb-1"><strong>Grade:</strong> <?php echo htmlspecialchars(getGradeName($student['grade_id'])); ?></p>
                    <p class="mb-1"><strong>Subjects:</strong> <?php echo count($subjects); ?></p>
                </div>
                <div class="col-md-6 text-end">
                    <h3 class="text-primary mb-0">Total Amount</h3>
                    <h2 class="mb-0" style="color: var(--brand-primary);">SZL <?php echo number_format($amount, 2); ?></h2>
                    <small class="text-muted">Academy: Kine Academy</small>
                </div>
            </div>
        </div>
        
        <!-- Payment Options -->
        <div class="payment-options">
            <!-- Mobile Money Option (Active) -->
            <div class="payment-card" id="momo-card">
                <span class="recommended-badge">Recommended</span>
                <i class="bi bi-phone-fill payment-icon"></i>
                <h3>Mobile Money</h3>
                <p>Pay securely with MTN Mobile Money</p>
                
                <ul class="payment-features">
                    <li><i class="bi bi-check-circle-fill"></i> Instant payment</li>
                    <li><i class="bi bi-check-circle-fill"></i> No extra fees</li>
                    <li><i class="bi bi-check-circle-fill"></i> Secure & fast</li>
                    <li><i class="bi bi-check-circle-fill"></i> Automatic enrollment</li>
                </ul>
                
                <button onclick="showPhoneModal()" class="btn btn-primary btn-lg w-100">
                    <i class="bi bi-phone me-2"></i> Continue with MTN Mobile Money
                </button>
                
                <div class="alert alert-info mt-3 mb-0" style="font-size: 0.9rem;">
                    <i class="bi bi-info-circle-fill me-2"></i>
                    Enter your MTN number to complete payment
                </div>
            </div>
            
            <!-- Card Payment Option (Disabled - Coming Soon) -->
            <div class="payment-card" id="card-card" style="opacity: 0.7; position: relative; cursor: not-allowed;">
                <span class="recommended-badge" style="background: linear-gradient(135deg, #6c757d 0%, #495057 100%);">Coming Soon</span>
                <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255,255,255,0.6); border-radius: 15px; z-index: 1;"></div>
                <i class="bi bi-credit-card-fill payment-icon" style="color: #6c757d;"></i>
                <h3 style="color: #6c757d;">Credit/Debit Card</h3>
                <p style="color: #6c757d;">Pay with Visa, Mastercard, Amex</p>
                
                <ul class="payment-features">
                    <li><i class="bi bi-check-circle-fill text-muted"></i> <span style="color: #6c757d;">All cards accepted</span></li>
                    <li><i class="bi bi-check-circle-fill text-muted"></i> <span style="color: #6c757d;">Secure checkout</span></li>
                    <li><i class="bi bi-check-circle-fill text-muted"></i> <span style="color: #6c757d;">International cards</span></li>
                    <li><i class="bi bi-check-circle-fill text-muted"></i> <span style="color: #6c757d;">3D Secure protection</span></li>
                </ul>
                
                <button class="btn btn-secondary btn-lg w-100" disabled style="cursor: not-allowed;">
                    <i class="bi bi-credit-card me-2"></i> Coming Soon
                </button>
                
                <div class="alert alert-warning mt-3 mb-0" style="font-size: 0.9rem; background: #fff3cd; border: none;">
                    <i class="bi bi-clock-fill me-2"></i>
                    <strong>Available Soon!</strong> Card payments will be enabled shortly.
                </div>
            </div>
        </div>
        
        <!-- Accepted Payment Methods -->
        <div class="text-center mt-4">
            <p class="text-muted small mb-2">Payment methods:</p>
            <div class="d-flex justify-content-center gap-2 flex-wrap">
                <span class="badge bg-success p-2" style="font-size: 0.9rem;">
                    <i class="bi bi-check-circle-fill me-1"></i> MTN Mobile Money
                </span>
                <span class="badge bg-secondary p-2" style="font-size: 0.9rem;">
                    <i class="bi bi-clock-fill me-1"></i> Card Payments (Soon)
                </span>
            </div>
        </div>
    </div>
    
    <!-- Phone Number Modal -->
    <div class="modal fade" id="phoneModal" tabindex="-1" aria-labelledby="phoneModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 10px 40px rgba(0,0,0,0.2);">
                <div class="modal-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 20px 20px 0 0; padding: 25px; border: none;">
                    <h5 class="modal-title text-white fw-bold" id="phoneModalLabel">
                        <i class="bi bi-phone-fill me-2"></i> Enter MTN Mobile Money Number
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" style="padding: 30px;">
                    <div class="mb-4">
                        <label for="phoneNumber" class="form-label fw-semibold" style="color: #495057; font-size: 0.95rem;">
                            <i class="bi bi-phone text-primary me-2"></i> MTN Mobile Money Number
                        </label>
                        <div class="input-group input-group-lg">
                            <span class="input-group-text" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; font-weight: bold; border: none; border-radius: 12px 0 0 12px; padding: 15px 20px; font-size: 1.1rem;">
                                +268
                            </span>
                            <input type="tel" 
                                   class="form-control form-control-lg" 
                                   id="phoneNumber" 
                                   placeholder="XXXXXXXX"
                                   maxlength="8"
                                   style="border: 2px solid #e9ecef; border-left: none; border-radius: 0 12px 12px 0; padding: 15px; font-size: 1.1rem; letter-spacing: 1px;"
                                   autocomplete="tel">
                        </div>
                        <div id="phoneError" class="invalid-feedback d-block" style="display: none !important; margin-top: 8px;"></div>
                        <small class="text-muted d-block mt-2">
                            <i class="bi bi-info-circle me-1"></i> Enter 8 digits (e.g., 76873207). The +268 prefix is added automatically.
                        </small>
                    </div>
                    
                    <div class="alert alert-info" style="background: #e7f3ff; border: none; border-radius: 12px;">
                        <i class="bi bi-shield-check text-primary me-2"></i>
                        <small>You will receive a payment prompt on your phone to approve the transaction.</small>
                    </div>
                </div>
                <div class="modal-footer" style="border: none; padding: 20px 30px 30px;">
                    <button type="button" class="btn btn-light btn-lg" data-bs-dismiss="modal" style="border-radius: 12px; padding: 12px 30px;">
                        <i class="bi bi-x-circle me-2"></i> Cancel
                    </button>
                    <button type="button" class="btn btn-primary btn-lg" onclick="submitPhone()" style="border-radius: 12px; padding: 12px 30px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none;">
                        <i class="bi bi-check-circle me-2"></i> Continue to Payment
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- MoMo Payment Form (Hidden) -->
    <div id="momoForm" class="d-none">
        <form action="payments/mtn/request.php" method="POST" id="momoPaymentForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
            <input type="hidden" name="amount" value="<?php echo $amount; ?>">
            <input type="hidden" name="description" value="Kine Academy - Student Registration">
        </form>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let phoneModal;
        
        // Initialize modal on page load
        document.addEventListener('DOMContentLoaded', function() {
            phoneModal = new bootstrap.Modal(document.getElementById('phoneModal'));
            
            // Focus on input when modal opens
            document.getElementById('phoneModal').addEventListener('shown.bs.modal', function() {
                document.getElementById('phoneNumber').focus();
            });
            
            // Clear input and errors when modal closes
            document.getElementById('phoneModal').addEventListener('hidden.bs.modal', function() {
                document.getElementById('phoneNumber').value = '';
                document.getElementById('phoneNumber').classList.remove('is-invalid');
                document.getElementById('phoneError').style.display = 'none';
            });
            
            // Allow Enter key to submit
            document.getElementById('phoneNumber').addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    submitPhone();
                }
            });
            
            // Real-time validation
            document.getElementById('phoneNumber').addEventListener('input', function(e) {
                let value = e.target.value.replace(/\D/g, ''); // Remove non-digits
                e.target.value = value;
                
                // Remove error styling on input
                if (value.length > 0) {
                    e.target.classList.remove('is-invalid');
                    document.getElementById('phoneError').style.display = 'none';
                }
            });
        });
        
        function showPhoneModal() {
            phoneModal.show();
        }
        
        function submitPhone() {
            const phoneInput = document.getElementById('phoneNumber');
            const phoneDigits = phoneInput.value.trim();
            const errorDiv = document.getElementById('phoneError');
            
            // Validate phone number (8 digits only)
            if (!phoneDigits) {
                phoneInput.classList.add('is-invalid');
                errorDiv.textContent = 'Please enter your 8-digit MTN Mobile Money number';
                errorDiv.style.display = 'block';
                phoneInput.focus();
                return;
            }
            
            if (!/^[0-9]{8}$/.test(phoneDigits)) {
                phoneInput.classList.add('is-invalid');
                errorDiv.textContent = 'Invalid format. Please enter exactly 8 digits (e.g., 76873207)';
                errorDiv.style.display = 'block';
                phoneInput.focus();
                return;
            }
            
            // Valid phone number - prepend 268 and submit form
            phoneInput.classList.remove('is-invalid');
            errorDiv.style.display = 'none';
            
            // Prepend 268 to create full phone number
            const fullPhone = '268' + phoneDigits;
            
            // Add phone to form
            const form = document.getElementById('momoPaymentForm');
            const hiddenPhone = document.createElement('input');
            hiddenPhone.type = 'hidden';
            hiddenPhone.name = 'phone';
            hiddenPhone.value = fullPhone;
            form.appendChild(hiddenPhone);
            
            // Close modal and submit
            phoneModal.hide();
            
            // Show loading state
            setTimeout(function() {
                form.submit();
            }, 300);
        }
        
        // Add hover effect for MoMo card
        document.getElementById('momo-card').addEventListener('mouseenter', function() {
            this.style.transform = 'scale(1.02)';
            this.style.transition = 'transform 0.3s ease';
        });
        
        document.getElementById('momo-card').addEventListener('mouseleave', function() {
            this.style.transform = 'scale(1)';
        });
    </script>
</body>
</html>

