<?php
/**
 * Payment for Student Application
 * Handles payment after email verification
 */

session_start();
require_once 'includes/functions.php';
require_once 'includes/favicon_helper.php';

$pdo = getDB();

// Check if application ID in session
if (!isset($_SESSION['pending_application_id'])) {
    header('Location: index.php');
    exit;
}

$application_id = $_SESSION['pending_application_id'];

// Get application details
$stmt = $pdo->prepare("
    SELECT sa.*, g.name as grade_name, ar.academy_name as school_name
    FROM student_applications sa
    JOIN grades g ON sa.grade_id = g.id
    JOIN academy_references ar ON sa.academy_reference COLLATE utf8mb4_general_ci = ar.reference_code COLLATE utf8mb4_general_ci
    WHERE sa.id = ? AND sa.email_verified = 1
");
$stmt->execute([$application_id]);
$application = $stmt->fetch();

if (!$application) {
    die("Error: Application not found or email not verified.");
}

// Get selected subjects
$stmt = $pdo->prepare("
    SELECT s.name, s.code 
    FROM subjects s
    JOIN student_application_subjects sas ON s.id = sas.subject_id
    WHERE sas.application_id = ?
");
$stmt->execute([$application_id]);
$selected_subjects = $stmt->fetchAll();

// Get school settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$application['academy_reference']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$currency_symbol = $settings['currency_symbol'] ?? 'E';

$payment_reference = 'APP' . time() . rand(1000, 9999);

// Handle payment submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['process_payment'])) {
    $payment_method = $_POST['payment_method'] ?? 'momo';
    $momo_number = trim($_POST['momo_number'] ?? '');
    
    if ($payment_method === 'momo' && empty($momo_number)) {
        $error = "Please enter your MoMo number.";
    } else {
        try {
            $pdo->beginTransaction();
            
            // Update application
            $stmt = $pdo->prepare("
                UPDATE student_applications 
                SET payment_status = 'paid',
                    payment_method = ?,
                    payment_reference = ?,
                    payment_date = NOW(),
                    application_status = 'approved'
                WHERE id = ?
            ");
            $stmt->execute([$payment_method, $payment_reference, $application_id]);
            
            // Create student account
            require_once 'includes/student_account_creator.php';
            $accountCreator = new StudentAccountCreator($pdo);
            $result = $accountCreator->createFromApplication($application_id);
            
            if ($result['success']) {
                // Mark application as completed
                $stmt = $pdo->prepare("
                    UPDATE student_applications 
                    SET application_status = 'completed',
                        student_id = ?,
                        username = ?,
                        completed_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$result['student_id'], $result['username'], $application_id]);
                
                $pdo->commit();
                
                // Clear session
                unset($_SESSION['pending_application_id']);
                unset($_SESSION['pending_payment_amount']);
                
                // Redirect to success page
                $_SESSION['registration_success'] = [
                    'username' => $result['username'],
                    'password' => $result['password'],
                    'email' => $application['email']
                ];
                
                header('Location: registration_complete.php');
                exit;
            } else {
                throw new Exception($result['error'] ?? 'Failed to create account');
            }
            
        } catch (Exception $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $error = "Payment processing error: " . $e->getMessage();
            error_log("Payment error for application {$application_id}: " . $e->getMessage());
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment - Student Registration</title>
    <?php echoFavicon($application['academy_reference'] ?? null); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px 0;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                
                <div class="card mb-3">
                    <div class="card-header bg-success text-white text-center">
                        <h4><i class="bi bi-check-circle me-2"></i>Email Verified Successfully!</h4>
                    </div>
                </div>
                
                <!-- Application Summary -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-file-text me-2"></i>Application Summary</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <p><strong>Student:</strong> <?= htmlspecialchars($application['full_name']) ?></p>
                                <p><strong>Email:</strong> <?= htmlspecialchars($application['email']) ?></p>
                                <p><strong>School:</strong> <?= htmlspecialchars($application['school_name']) ?></p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>Grade:</strong> <?= htmlspecialchars($application['grade_name']) ?></p>
                                <p><strong>Subjects:</strong> <?= count($selected_subjects) ?></p>
                                <p><strong>Application ID:</strong> #<?= $application['id'] ?></p>
                            </div>
                        </div>
                        
                        <hr>
                        
                        <h6>Selected Subjects:</h6>
                        <div class="row">
                            <?php foreach ($selected_subjects as $subject): ?>
                            <div class="col-md-6">
                                <span class="badge bg-primary mb-1">
                                    <i class="bi bi-book me-1"></i><?= htmlspecialchars($subject['name']) ?>
                                </span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Payment Form -->
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-credit-card me-2"></i>Complete Payment</h5>
                    </div>
                    <div class="card-body">
                        
                        <?php if (isset($error)): ?>
                        <div class="alert alert-danger">
                            <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
                        </div>
                        <?php endif; ?>
                        
                        <!-- Fee Breakdown -->
                        <div class="card bg-light mb-4">
                            <div class="card-body">
                                <h6>Fee Breakdown</h6>
                                <table class="table table-sm mb-0">
                                    <tr>
                                        <td>Number of Subjects:</td>
                                        <td class="text-end"><strong><?= count($selected_subjects) ?></strong></td>
                                    </tr>
                                    <tr>
                                        <td>Fee per Subject:</td>
                                        <td class="text-end"><?= $currency_symbol ?><?= number_format($application['payment_amount'] / count($selected_subjects), 2) ?></td>
                                    </tr>
                                    <tr class="table-primary">
                                        <td><strong>Total Amount:</strong></td>
                                        <td class="text-end"><h4 class="mb-0"><?= $currency_symbol ?><?= number_format($application['payment_amount'], 2) ?></h4></td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <form method="POST">
                            <!-- Payment Method -->
                            <div class="mb-4">
                                <label class="form-label"><strong>Select Payment Method</strong></label>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-body">
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="payment_method" 
                                                           value="momo" id="momo" checked 
                                                           onchange="toggleMomoField()">
                                                    <label class="form-check-label" for="momo">
                                                        <h6 class="mb-0">
                                                            <i class="bi bi-phone me-2"></i>MTN MoMo
                                                        </h6>
                                                        <small class="text-muted">Pay with Mobile Money</small>
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-body">
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="payment_method" 
                                                           value="card" id="card"
                                                           onchange="toggleMomoField()">
                                                    <label class="form-check-label" for="card">
                                                        <h6 class="mb-0">
                                                            <i class="bi bi-credit-card me-2"></i>Credit/Debit Card
                                                        </h6>
                                                        <small class="text-muted">Pay with Card</small>
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- MoMo Number Field -->
                            <div class="mb-4" id="momo_field">
                                <label class="form-label">MTN MoMo Number <span class="text-danger">*</span></label>
                                <div class="input-group input-group-lg">
                                    <span class="input-group-text">+268</span>
                                    <input type="tel" name="momo_number" class="form-control" 
                                           placeholder="7687 3207" required>
                                </div>
                                <small class="text-muted">You will receive a payment prompt on this number</small>
                            </div>
                            
                            <!-- Submit Button -->
                            <div class="d-grid">
                                <button type="submit" name="process_payment" class="btn btn-primary btn-lg">
                                    <i class="bi bi-lock me-2"></i>
                                    Pay <?= $currency_symbol ?><?= number_format($application['payment_amount'], 2) ?> & Complete Registration
                                </button>
                            </div>
                            
                            <p class="text-center text-muted mt-3 mb-0">
                                <small>
                                    <i class="bi bi-shield-check me-1"></i>
                                    Secure payment processing
                                </small>
                            </p>
                        </form>
                        
                    </div>
                </div>
                
            </div>
        </div>
    </div>
    
    <script>
        function toggleMomoField() {
            const momoSelected = document.getElementById('momo').checked;
            const momoField = document.getElementById('momo_field');
            const momoInput = momoField.querySelector('input');
            
            if (momoSelected) {
                momoField.style.display = 'block';
                momoInput.required = true;
            } else {
                momoField.style.display = 'none';
                momoInput.required = false;
            }
        }
    </script>
</body>
</html>

