<?php
/**
 * Payment Processing Page
 * Handles MoMo payment integration and payment recording
 */

require_once '../includes/functions.php';
require_once '../includes/parent_functions.php';
requireParentLogin();

$parent_id = $_SESSION['parent_id'];

// Get pending payment from session
if (!isset($_SESSION['pending_payment'])) {
    header('Location: dashboard.php');
    exit;
}

$payment_data = $_SESSION['pending_payment'];
$student_id = $payment_data['student_id'];

// Verify parent has access
if (!parentHasAccessToStudent($parent_id, $student_id)) {
    unset($_SESSION['pending_payment']);
    header('Location: dashboard.php');
    exit;
}

$pdo = getDB();

// Get student details
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

// Get school settings for currency
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$student['academy_reference']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}
$currency_symbol = $settings['currency_symbol'] ?? 'E';
$schoolName = $settings['school_name'] ?? 'School';

// Get subject names
$subject_ids = $payment_data['subjects'];
$placeholders = str_repeat('?,', count($subject_ids) - 1) . '?';
$stmt = $pdo->prepare("SELECT id, name, code FROM subjects WHERE id IN ($placeholders)");
$stmt->execute($subject_ids);
$selected_subjects = $stmt->fetchAll();

$payment_reference = 'PAY' . time() . rand(1000, 9999);
$payment_status = 'pending';
$payment_message = 'Processing payment...';

// Handle payment confirmation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_payment'])) {
    $payment_method = $payment_data['payment_method'];
    
    // Handle Card Payment - Redirect to gateway
    if ($payment_method === 'card') {
        // Store payment info for callback
        $_SESSION['card_payment_pending'] = [
            'student_id' => $student_id,
            'subjects' => $payment_data['subjects'],
            'total_amount' => $payment_data['total_amount'],
            'payment_period' => $payment_data['payment_period'],
            'reference' => $payment_reference,
            'parent_id' => $parent_id
        ];
        
        // Redirect to payment gateway page
        header('Location: card_payment_gateway.php');
        exit;
    }
    
    // Handle MoMo Payment - Process directly
    try {
        // Only start transaction if not already in one
        $inTransaction = $pdo->inTransaction();
        if (!$inTransaction) {
            $pdo->beginTransaction();
        }
        
        // Create payment record
        $stmt = $pdo->prepare("
            INSERT INTO payments (
                student_id, 
                academy_reference, 
                amount, 
                status, 
                payment_method, 
                momo_number,
                reference,
                payment_period,
                created_at
            ) VALUES (?, ?, ?, 'completed', ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $student_id,
            $student['academy_reference'],
            $payment_data['total_amount'],
            $payment_method,
            $payment_data['momo_number'] ?? null,
            $payment_reference,
            $payment_data['payment_period']
        ]);
        
        $payment_id = $pdo->lastInsertId();
        
        // Update student-subject relationships
        $moodle_subjects = []; // Track subjects for Moodle enrollment
        foreach ($payment_data['subjects'] as $subject_id) {
            // Check if already enrolled
            $stmt = $pdo->prepare("
                SELECT id FROM student_subject 
                WHERE student_id = ? AND subject_id = ?
            ");
            $stmt->execute([$student_id, $subject_id]);
            
            if (!$stmt->fetch()) {
                // Enroll in subject
                $stmt = $pdo->prepare("
                    INSERT INTO student_subject (student_id, subject_id)
                    VALUES (?, ?)
                ");
                $stmt->execute([$student_id, $subject_id]);
                
                // Add to Moodle enrollment list
                $moodle_subjects[] = $subject_id;
            }
        }
        
        // Auto-enroll in Moodle if enabled
        if (!empty($moodle_subjects)) {
            require_once '../includes/moodle_api.php';
            $moodle = new MoodleAPI($student['academy_reference'], $pdo);
            
            if ($moodle->isEnabled()) {
                foreach ($moodle_subjects as $subject_id) {
                    // Get subject Moodle configuration
                    $stmt = $pdo->prepare("
                        SELECT moodle_course_id, auto_enroll, moodle_sync_enabled 
                        FROM subjects 
                        WHERE id = ? AND moodle_sync_enabled = 1 AND auto_enroll = 1
                    ");
                    $stmt->execute([$subject_id]);
                    $subject_config = $stmt->fetch();
                    
                    if ($subject_config && !empty($subject_config['moodle_course_id'])) {
                        // Enroll student in Moodle course (async, don't block payment)
                        try {
                            $moodle->enrollStudent($student_id, $subject_id, $subject_config['moodle_course_id']);
                        } catch (Exception $e) {
                            error_log("Moodle auto-enrollment error: " . $e->getMessage());
                            // Don't fail payment if Moodle enrollment fails
                        }
                    }
                }
            }
        }
        
        // Update student payment status
        $stmt = $pdo->prepare("
            UPDATE students 
            SET payment_status = 'active',
                last_payment_date = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$student_id]);
        
        // Send notification to parent
        sendParentNotification(
            $parent_id,
            'Payment Successful!',
            "Your payment of {$currency_symbol}" . number_format($payment_data['total_amount'], 2) . 
            " for {$student['full_name']} has been received. Receipt: {$payment_reference}",
            'payment',
            $student_id
        );
        
        // Log activity
        logParentActivity(
            $parent_id, 
            'PAYMENT_COMPLETED', 
            "Paid E{$payment_data['total_amount']} for " . count($payment_data['subjects']) . " subjects",
            $student_id
        );
        
        // Only commit if we started the transaction
        if (!$inTransaction && $pdo->inTransaction()) {
            $pdo->commit();
        }
        
        // Clear pending payment
        unset($_SESSION['pending_payment']);
        
        // Redirect to success page
        $_SESSION['payment_success'] = [
            'reference' => $payment_reference,
            'amount' => $payment_data['total_amount'],
            'student_name' => $student['full_name'],
            'subject_count' => count($payment_data['subjects']),
            'payment_period' => $payment_data['payment_period']
        ];
        
        header('Location: payment_success.php');
        exit;
        
    } catch (Exception $e) {
        // Only rollback if we started the transaction
        if (!$inTransaction && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Payment processing error: " . $e->getMessage());
        $payment_status = 'failed';
        $payment_message = 'Payment processing failed. Please try again.';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Confirm Payment - Parent Portal</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #667eea;
            --secondary: #764ba2;
        }
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 20px 0;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
        }
        .payment-header {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
            padding: 2rem;
            border-radius: 1rem 1rem 0 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card">
                    <div class="payment-header text-center">
                        <i class="bi bi-credit-card fs-1"></i>
                        <h2 class="mt-3 mb-0">Confirm Payment</h2>
                        <p class="mb-0 opacity-75">Please review your payment details</p>
                    </div>
                    
                    <div class="card-body p-4">
                        <!-- Student Info -->
                        <div class="alert alert-light border">
                            <div class="d-flex align-items-center">
                                <i class="bi bi-person-circle fs-3 text-primary me-3"></i>
                                <div>
                                    <h5 class="mb-1"><?= htmlspecialchars($student['full_name']) ?></h5>
                                    <small class="text-muted">Username: <?= htmlspecialchars($student['username']) ?></small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Selected Subjects -->
                        <h5 class="mb-3"><i class="bi bi-book me-2"></i>Selected Subjects (<?= count($selected_subjects) ?>)</h5>
                        <div class="row g-2 mb-4">
                            <?php foreach ($selected_subjects as $subject): ?>
                                <div class="col-md-6">
                                    <div class="card bg-light">
                                        <div class="card-body py-2">
                                            <small><i class="bi bi-check-circle text-success me-2"></i><?= htmlspecialchars($subject['name']) ?></small>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <!-- Payment Details -->
                        <div class="card bg-light mb-4">
                            <div class="card-body">
                                <h5 class="mb-3"><i class="bi bi-receipt me-2"></i>Payment Details</h5>
                                
                                <div class="row mb-2">
                                    <div class="col-6">Payment Period:</div>
                                    <div class="col-6 text-end">
                                        <strong><?= $payment_data['payment_period'] === 'yearly' ? '12 Months (Yearly)' : '1 Month (Monthly)' ?></strong>
                                    </div>
                                </div>
                                
                                <div class="row mb-2">
                                    <div class="col-6">Number of Subjects:</div>
                                    <div class="col-6 text-end"><strong><?= $payment_data['subject_count'] ?></strong></div>
                                </div>
                                
                                <div class="row mb-2">
                                    <div class="col-6">Price per Subject:</div>
                                    <div class="col-6 text-end"><strong><?= $currency_symbol ?><?= number_format($payment_data['price_per_subject'], 2) ?></strong></div>
                                </div>
                                
                                <?php if ($payment_data['payment_period'] === 'yearly'): ?>
                                    <div class="row mb-2 text-success">
                                        <div class="col-6">Yearly Discount:</div>
                                        <div class="col-6 text-end"><strong>Applied</strong></div>
                                    </div>
                                <?php endif; ?>
                                
                                <hr>
                                
                                <div class="row">
                                    <div class="col-6 fs-5">Total Amount:</div>
                                    <div class="col-6 text-end">
                                        <strong class="fs-4 text-primary"><?= $currency_symbol ?><?= number_format($payment_data['total_amount'], 2) ?></strong>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Payment Method -->
                        <div class="card bg-light mb-4">
                            <div class="card-body">
                                <h5 class="mb-3"><i class="bi bi-wallet2 me-2"></i>Payment Method</h5>
                                
                                <?php if ($payment_data['payment_method'] === 'card'): ?>
                                    <div class="d-flex align-items-center">
                                        <i class="bi bi-credit-card text-success fs-1 me-3"></i>
                                        <div>
                                            <strong>Credit / Debit Card</strong>
                                            <br>
                                            <span class="text-muted">Secure payment via <?= ucfirst($schoolSettings['payment_gateway'] ?? 'PayGate') ?></span>
                                        </div>
                                    </div>
                                    <div class="alert alert-info mt-3 mb-0">
                                        <i class="bi bi-info-circle me-2"></i>
                                        <small>You'll be redirected to enter your card details securely.</small>
                                    </div>
                                <?php else: ?>
                                    <div class="d-flex align-items-center">
                                        <img src="../images/mtn-logo.png" alt="MTN MoMo" style="height: 40px;" onerror="this.style.display='none'">
                                        <i class="bi bi-phone text-primary fs-1 me-3"></i>
                                        <div class="ms-3">
                                            <strong>MTN Mobile Money</strong>
                                            <br>
                                            <span class="text-muted">+268 <?= htmlspecialchars($payment_data['momo_number'] ?? 'N/A') ?></span>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Payment Reference -->
                        <div class="alert alert-info">
                            <small class="text-muted d-block">Payment Reference</small>
                            <strong><?= $payment_reference ?></strong>
                        </div>
                        
                        <!-- Important Notice -->
                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle me-2"></i>
                            <strong>Important:</strong> You will receive a MoMo prompt on your phone. 
                            Please approve the transaction to complete payment.
                        </div>
                        
                        <!-- Action Buttons -->
                        <form method="POST">
                            <div class="d-grid gap-2">
                                <button type="submit" name="confirm_payment" class="btn btn-primary btn-lg">
                                    <i class="bi bi-check-circle me-2"></i>Confirm & Pay <?= $currency_symbol ?><?= number_format($payment_data['total_amount'], 2) ?>
                                </button>
                                <a href="make_payment.php?student_id=<?= $student_id ?>" class="btn btn-outline-secondary">
                                    <i class="bi bi-arrow-left me-2"></i>Back to Edit
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="text-center mt-3">
                    <small class="text-muted">
                        <i class="bi bi-shield-check me-1"></i>
                        Secure payment processing | Your data is encrypted
                    </small>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

