<?php
/**
 * Parent Dashboard
 * Main dashboard for parents to view and manage their children's information
 */

require_once '../includes/functions.php';
require_once '../includes/parent_functions.php';
requireParentLogin();

$parent_id = $_SESSION['parent_id'];
$parent_name = $_SESSION['parent_name'];

// Get parent's children
$children = getParentChildren($parent_id);

// Get payment summary
$summary = getParentPaymentSummary($parent_id);

// Get recent notifications
$notifications = getParentNotifications($parent_id, false);
$unread_count = count(array_filter($notifications, fn($n) => !$n['is_read']));

// Get school settings for currency
$pdo = getDB();
$academy_reference = $_SESSION['academy_reference'] ?? '';
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$academy_reference]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}
$currency_symbol = $settings['currency_symbol'] ?? 'E';
$schoolName = $settings['school_name'] ?? 'School';

// Log dashboard access
logParentActivity($parent_id, 'VIEW_DASHBOARD', 'Accessed parent dashboard');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Parent Dashboard - School Management System</title>
    
    <!-- PWA Support -->
    <link rel="manifest" href="../manifest.json">
    <meta name="theme-color" content="#667eea">
    <meta name="apple-mobile-web-app-capable" content="yes">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #667eea;
            --secondary: #764ba2;
        }
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
        }
        .navbar {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%) !important;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.08);
            transition: transform 0.2s;
        }
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.12);
        }
        .child-card {
            background: white;
        }
        .stat-card {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
        }
        @media (max-width: 768px) {
            .child-card {
                margin-bottom: 1rem;
            }
            .btn {
                min-height: 44px;
                font-size: 16px;
            }
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar navbar-dark sticky-top">
        <div class="container-fluid">
            <span class="navbar-brand mb-0 h1">
                <i class="bi bi-house-heart-fill me-2"></i>Parent Portal
            </span>
            <div class="d-flex align-items-center gap-3">
                <div class="dropdown">
                    <button class="btn btn-light position-relative" type="button" data-bs-toggle="dropdown">
                        <i class="bi bi-bell-fill"></i>
                        <?php if ($unread_count > 0): ?>
                        <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                            <?= $unread_count ?>
                        </span>
                        <?php endif; ?>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-end" style="max-width: 300px;">
                        <li><h6 class="dropdown-header">Notifications</h6></li>
                        <?php if (empty($notifications)): ?>
                        <li><span class="dropdown-item-text text-muted">No notifications</span></li>
                        <?php else: ?>
                            <?php foreach (array_slice($notifications, 0, 5) as $notif): ?>
                            <li>
                                <a class="dropdown-item <?= !$notif['is_read'] ? 'bg-light' : '' ?>" href="#">
                                    <strong><?= htmlspecialchars($notif['title']) ?></strong><br>
                                    <small><?= htmlspecialchars(substr($notif['message'], 0, 50)) ?>...</small>
                                </a>
                            </li>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </ul>
                </div>
                <div class="dropdown">
                    <button class="btn btn-light dropdown-toggle" type="button" data-bs-toggle="dropdown">
                        <i class="bi bi-person-circle me-1"></i><?= htmlspecialchars($parent_name) ?>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><a class="dropdown-item" href="#"><i class="bi bi-gear me-2"></i>Settings</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item text-danger" href="../logout.php?parent=1">
                            <i class="bi bi-box-arrow-right me-2"></i>Logout
                        </a></li>
                    </ul>
                </div>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="container-fluid my-4">
        <!-- Welcome Section -->
        <div class="row mb-4">
            <div class="col-12">
                <h2 class="mb-3">
                    👋 Welcome back, <?= htmlspecialchars($parent_name) ?>!
                </h2>
                <p class="text-muted">Manage your <?= count($children) ?> child<?= count($children) != 1 ? 'ren' : '' ?>'s education</p>
            </div>
        </div>

        <!-- Summary Stats -->
        <div class="row mb-4">
            <div class="col-md-3 mb-3">
                <div class="card stat-card">
                    <div class="card-body text-center">
                        <h3 class="mb-0"><?= $summary['total_children'] ?></h3>
                        <p class="mb-0">Total Children</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card stat-card">
                    <div class="card-body text-center">
                        <h3 class="mb-0"><?= $summary['active'] ?></h3>
                        <p class="mb-0">Active</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0"><?= $currency_symbol ?><?= number_format($summary['total_paid'], 2) ?></h3>
                        <p class="mb-0">Total Paid</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card bg-warning text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0"><?= $currency_symbol ?><?= number_format($summary['total_pending'], 2) ?></h3>
                        <p class="mb-0">Pending</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Children Cards -->
        <div class="row">
            <?php if (empty($children)): ?>
            <div class="col-12">
                <div class="alert alert-info">
                    <i class="bi bi-info-circle me-2"></i>
                    No children linked to your account yet. Please contact the school.
                </div>
            </div>
            <?php else: ?>
                <?php foreach ($children as $child): ?>
                <div class="col-lg-6 mb-4">
                    <div class="card child-card">
                        <div class="card-header bg-primary text-white">
                            <div class="d-flex justify-content-between align-items-center">
                                <h5 class="mb-0">
                                    <i class="bi bi-person-circle me-2"></i>
                                    <?= htmlspecialchars($child['full_name']) ?>
                                </h5>
                                <?php if ($child['payment_status'] === 'active'): ?>
                                    <span class="badge bg-success">Active</span>
                                <?php elseif ($child['payment_status'] === 'pending'): ?>
                                    <span class="badge bg-warning">Pending</span>
                                <?php else: ?>
                                    <span class="badge bg-danger">Suspended</span>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="row g-3">
                                <div class="col-6">
                                    <small class="text-muted d-block">Username</small>
                                    <strong><?= htmlspecialchars($child['username']) ?></strong>
                                </div>
                                <div class="col-6">
                                    <small class="text-muted d-block">Grade</small>
                                    <strong><?= htmlspecialchars($child['grade_name']) ?></strong>
                                </div>
                                <div class="col-12">
                                    <small class="text-muted d-block">Email</small>
                                    <strong><?= htmlspecialchars($child['email'] ?? 'N/A') ?></strong>
                                    <?php if ($child['email_verified']): ?>
                                        <span class="badge bg-success ms-2"><i class="bi bi-check-circle"></i> Verified</span>
                                    <?php else: ?>
                                        <span class="badge bg-warning ms-2"><i class="bi bi-clock"></i> Pending</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="d-grid gap-2 mt-3">
                                <a href="make_payment.php?student_id=<?= $child['id'] ?>" class="btn btn-success">
                                    <i class="bi bi-credit-card me-2"></i>Make Payment
                                </a>
                                <a href="view_payments.php?student_id=<?= $child['id'] ?>" class="btn btn-outline-primary">
                                    <i class="bi bi-receipt me-2"></i>View Payment History
                                </a>
                            </div>
                        </div>
                        <div class="card-footer bg-light">
                            <small class="text-muted">
                                <i class="bi bi-calendar-event me-1"></i>
                                Registered: <?= date('M d, Y', strtotime($child['created_at'])) ?>
                            </small>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

