<?php
/**
 * Card Payment Return Handler
 * Processes the response from payment gateway (PayGate/PayFast/Stripe)
 */

require_once '../includes/functions.php';
require_once '../includes/parent_functions.php';
requireParentLogin();

$parent_id = $_SESSION['parent_id'];
$pdo = getDB();

// Get pending card payment
if (!isset($_SESSION['card_payment_pending'])) {
    header('Location: dashboard.php');
    exit;
}

$payment_data = $_SESSION['card_payment_pending'];
$student_id = $payment_data['student_id'];

// Get student details
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

// Get school settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$student['academy_reference']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$payment_gateway = $settings['payment_gateway'] ?? 'paygate';
$merchant_key = $settings['paygate_secret'] ?? '';

$payment_successful = false;
$error_message = '';

// Process PayGate response
if ($payment_gateway === 'paygate' && isset($_POST['PAY_REQUEST_ID'])) {
    $pay_request_id = $_POST['PAY_REQUEST_ID'];
    $transaction_status = $_POST['TRANSACTION_STATUS'] ?? '';
    $checksum = $_POST['CHECKSUM'] ?? '';
    
    // Verify checksum
    $checksum_string = '';
    foreach ($_POST as $key => $value) {
        if ($key !== 'CHECKSUM') {
            $checksum_string .= $value;
        }
    }
    $checksum_string .= $merchant_key;
    $calculated_checksum = md5($checksum_string);
    
    if ($checksum === $calculated_checksum) {
        if ($transaction_status == '1') {
            // Payment successful
            $payment_successful = true;
        } else {
            $error_message = 'Payment was declined or cancelled.';
        }
    } else {
        $error_message = 'Invalid payment response.';
    }
}

// Process PayFast response
elseif ($payment_gateway === 'payfast' && isset($_POST['payment_status'])) {
    $payment_status = $_POST['payment_status'];
    $pf_payment_id = $_POST['pf_payment_id'] ?? '';
    
    if ($payment_status === 'COMPLETE') {
        $payment_successful = true;
    } else {
        $error_message = "Payment status: $payment_status";
    }
}

// If payment successful, process enrollment
if ($payment_successful) {
    try {
        // Only start transaction if not already in one
        $inTransaction = $pdo->inTransaction();
        if (!$inTransaction) {
            $pdo->beginTransaction();
        }
        
        // Create payment record
        $stmt = $pdo->prepare("
            INSERT INTO payments (
                student_id, 
                academy_reference, 
                amount, 
                status, 
                payment_method, 
                reference,
                payment_period,
                created_at
            ) VALUES (?, ?, ?, 'completed', 'card', ?, ?, NOW())
        ");
        $stmt->execute([
            $student_id,
            $student['academy_reference'],
            $payment_data['total_amount'],
            $payment_data['reference'],
            $payment_data['payment_period']
        ]);
        
        // Enroll in subjects
        foreach ($payment_data['subjects'] as $subject_id) {
            $stmt = $pdo->prepare("
                SELECT id FROM student_subject 
                WHERE student_id = ? AND subject_id = ?
            ");
            $stmt->execute([$student_id, $subject_id]);
            
            if (!$stmt->fetch()) {
                $stmt = $pdo->prepare("
                    INSERT INTO student_subject (student_id, subject_id)
                    VALUES (?, ?)
                ");
                $stmt->execute([$student_id, $subject_id]);
            }
        }
        
        // Update student payment status
        $stmt = $pdo->prepare("
            UPDATE students 
            SET payment_status = 'active',
                last_payment_date = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$student_id]);
        
        // Send notification
        $currency_symbol = $settings['currency_symbol'] ?? 'E';
        sendParentNotification(
            $parent_id,
            'Card Payment Successful!',
            "Your card payment of {$currency_symbol}" . number_format($payment_data['total_amount'], 2) . 
            " for {$student['full_name']} has been approved. Receipt: {$payment_data['reference']}",
            'payment',
            $student_id
        );
        
        // Log activity
        logParentActivity(
            $parent_id,
            'CARD_PAYMENT_COMPLETED',
            "Card payment successful: {$currency_symbol}{$payment_data['total_amount']}",
            $student_id
        );
        
        // Only commit if we started the transaction
        if (!$inTransaction && $pdo->inTransaction()) {
            $pdo->commit();
        }
        
        // Clear pending payment
        unset($_SESSION['card_payment_pending']);
        
        // Set success data
        $_SESSION['payment_success'] = [
            'reference' => $payment_data['reference'],
            'amount' => $payment_data['total_amount'],
            'student_name' => $student['full_name'],
            'subject_count' => count($payment_data['subjects']),
            'payment_period' => $payment_data['payment_period']
        ];
        
        // Redirect to success page
        header('Location: payment_success.php');
        exit;
        
    } catch (Exception $e) {
        // Only rollback if we started the transaction
        if (!$inTransaction && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Card payment processing error: " . $e->getMessage());
        $error_message = 'Error processing payment. Please contact support.';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Result - Parent Portal</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body p-5 text-center">
                        <?php if ($error_message): ?>
                            <i class="bi bi-x-circle text-danger" style="font-size: 4rem;"></i>
                            <h2 class="text-danger mt-3">Payment Failed</h2>
                            <p class="text-muted"><?= htmlspecialchars($error_message) ?></p>
                            
                            <div class="d-grid gap-2 mt-4">
                                <a href="make_payment.php?student_id=<?= $student_id ?>" class="btn btn-primary">
                                    <i class="bi bi-arrow-left me-2"></i>Try Again
                                </a>
                                <a href="dashboard.php" class="btn btn-outline-secondary">
                                    Back to Dashboard
                                </a>
                            </div>
                        <?php else: ?>
                            <div class="spinner-border text-primary mb-3" style="width: 3rem; height: 3rem;"></div>
                            <h4>Processing payment...</h4>
                            <p class="text-muted">Please wait while we confirm your payment.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

