<?php
/**
 * Card Payment Gateway Integration
 * Redirects to PayGate/PayFast/Stripe for secure card processing
 */

require_once '../includes/functions.php';
require_once '../includes/parent_functions.php';
requireParentLogin();

$parent_id = $_SESSION['parent_id'];

// Get pending card payment
if (!isset($_SESSION['card_payment_pending'])) {
    header('Location: dashboard.php');
    exit;
}

$payment_data = $_SESSION['card_payment_pending'];
$pdo = getDB();

// Get student details
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$payment_data['student_id']]);
$student = $stmt->fetch();

// Get school settings (payment gateway credentials)
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$student['academy_reference']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$payment_gateway = $settings['payment_gateway'] ?? 'paygate';
$merchant_id = $settings['paygate_id'] ?? '';
$merchant_key = $settings['paygate_secret'] ?? '';
$currency_symbol = $settings['currency_symbol'] ?? 'E';

// Payment details
$amount = $payment_data['total_amount'];
$reference = $payment_data['reference'];
$return_url = 'https://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/card_payment_return.php';
$notify_url = $return_url;

// Initialize payment based on gateway
if ($payment_gateway === 'paygate') {
    // PayGate Integration
    $paygate_url = 'https://secure.paygate.co.za/payweb3/initiate.trans';
    
    $data = [
        'PAYGATE_ID' => $merchant_id,
        'REFERENCE' => $reference,
        'AMOUNT' => round($amount * 100), // Convert to cents
        'CURRENCY' => 'ZAR', // Or use currency code from settings
        'RETURN_URL' => $return_url,
        'TRANSACTION_DATE' => date('Y-m-d H:i:s'),
        'LOCALE' => 'en',
        'COUNTRY' => 'SWZ',
        'EMAIL' => $_SESSION['parent_email']
    ];
    
    // Generate checksum
    $checksum_string = '';
    foreach ($data as $key => $value) {
        $checksum_string .= $value;
    }
    $checksum_string .= $merchant_key;
    $data['CHECKSUM'] = md5($checksum_string);
    
} elseif ($payment_gateway === 'payfast') {
    // PayFast Integration
    $payfast_url = 'https://www.payfast.co.za/eng/process';
    
    $data = [
        'merchant_id' => $merchant_id,
        'merchant_key' => $merchant_key,
        'return_url' => $return_url,
        'cancel_url' => $return_url,
        'notify_url' => $notify_url,
        'name_first' => $_SESSION['parent_name'],
        'email_address' => $_SESSION['parent_email'],
        'amount' => number_format($amount, 2, '.', ''),
        'item_name' => "School Fees - {$student['full_name']}",
        'custom_str1' => $reference
    ];
    
} elseif ($payment_gateway === 'stripe') {
    // Stripe integration would go here
    $error = "Stripe integration coming soon!";
} else {
    $error = "Payment gateway not configured.";
}

logParentActivity($parent_id, 'CARD_PAYMENT_INITIATED', "Redirecting to payment gateway: $payment_gateway", $payment_data['student_id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Redirecting to Payment Gateway...</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body p-5 text-center">
                        <div class="spinner-border text-primary mb-4" style="width: 4rem; height: 4rem;" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        
                        <h3 class="mb-3">Redirecting to Secure Payment...</h3>
                        <p class="text-muted">Please wait while we redirect you to our secure payment gateway.</p>
                        
                        <div class="alert alert-info text-start mt-4">
                            <i class="bi bi-shield-check me-2"></i>
                            <strong>Security Notice:</strong><br>
                            You'll be redirected to <strong><?= ucfirst($payment_gateway) ?></strong> to complete your payment.
                            Your card details are encrypted and secure.
                        </div>
                        
                        <?php if (isset($error)): ?>
                            <div class="alert alert-danger">
                                <?= htmlspecialchars($error) ?>
                            </div>
                            <a href="make_payment.php?student_id=<?= $payment_data['student_id'] ?>" class="btn btn-primary">
                                Go Back
                            </a>
                        <?php elseif ($payment_gateway === 'paygate'): ?>
                            <!-- Auto-submit form to PayGate -->
                            <form method="POST" action="<?= $paygate_url ?>" id="paygate_form">
                                <?php foreach ($data as $key => $value): ?>
                                    <input type="hidden" name="<?= $key ?>" value="<?= htmlspecialchars($value) ?>">
                                <?php endforeach; ?>
                            </form>
                            <script>
                                // Auto-submit after 2 seconds
                                setTimeout(function() {
                                    document.getElementById('paygate_form').submit();
                                }, 2000);
                            </script>
                        <?php elseif ($payment_gateway === 'payfast'): ?>
                            <!-- Auto-submit form to PayFast -->
                            <form method="POST" action="<?= $payfast_url ?>" id="payfast_form">
                                <?php foreach ($data as $key => $value): ?>
                                    <input type="hidden" name="<?= $key ?>" value="<?= htmlspecialchars($value) ?>">
                                <?php endforeach; ?>
                            </form>
                            <script>
                                // Auto-submit after 2 seconds
                                setTimeout(function() {
                                    document.getElementById('payfast_form').submit();
                                }, 2000);
                            </script>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

