<?php
/**
 * Payment Reminder System Initialization Script
 * Run this ONCE after installing the database schema
 * 
 * This script:
 * 1. Creates payment schedules for all existing students
 * 2. Initializes default settings for all tenants
 * 3. Validates the setup
 * 
 * Usage: php initialize_payment_system.php
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/includes/payment_reminder_manager.php';

echo "\n";
echo "========================================\n";
echo "Payment System Initialization\n";
echo "========================================\n\n";

try {
    $pdo = getDB();
    $manager = new PaymentReminderManager();
    
    // =====================================================
    // STEP 1: Verify Database Tables
    // =====================================================
    echo "Step 1: Verifying database tables...\n";
    
    $required_tables = [
        'tenant_payment_settings',
        'student_payment_schedules',
        'payment_reminders',
        'payment_transactions',
        'suspension_logs',
        'payment_notification_templates',
        'payment_system_audit_log'
    ];
    
    $missing_tables = [];
    foreach ($required_tables as $table) {
        $stmt = $pdo->query("SHOW TABLES LIKE '{$table}'");
        if ($stmt->rowCount() === 0) {
            $missing_tables[] = $table;
        }
    }
    
    if (!empty($missing_tables)) {
        echo "✗ ERROR: Missing tables:\n";
        foreach ($missing_tables as $table) {
            echo "  - {$table}\n";
        }
        echo "\nPlease run the migration SQL first:\n";
        echo "mysql -u root -p multi_tenant_skoloi < database/migrations/create_payment_reminder_system.sql\n\n";
        exit(1);
    }
    
    echo "✓ All required tables exist\n\n";
    
    // =====================================================
    // STEP 2: Initialize Settings for All Tenants
    // =====================================================
    echo "Step 2: Initializing payment settings for tenants...\n";
    
    $stmt = $pdo->query("SELECT reference_code, academy_name FROM academy_references WHERE is_active = 1");
    $tenants = $stmt->fetchAll();
    
    $settings_created = 0;
    foreach ($tenants as $tenant) {
        // Check if settings exist
        $stmt = $pdo->prepare("SELECT id FROM tenant_payment_settings WHERE academy_reference = ?");
        $stmt->execute([$tenant['reference_code']]);
        
        if (!$stmt->fetch()) {
            // Create default settings
            $stmt = $pdo->prepare("
                INSERT INTO tenant_payment_settings (
                    academy_reference, payment_cycle_days, reminder_days_before,
                    warning_days_after_reminder, days_before_suspension,
                    monthly_fee, unsuspension_fee, auto_suspend_enabled,
                    auto_reminder_enabled, auto_warning_enabled
                ) VALUES (?, 30, 27, 3, 35, 350.00, 50.00, 1, 1, 1)
            ");
            $stmt->execute([$tenant['reference_code']]);
            $settings_created++;
            echo "  ✓ Created settings for: {$tenant['academy_name']}\n";
        }
    }
    
    echo "✓ Settings initialized for {$settings_created} tenants\n\n";
    
    // =====================================================
    // STEP 3: Initialize Payment Schedules for Students
    // =====================================================
    echo "Step 3: Initializing payment schedules for existing students...\n";
    
    $stmt = $pdo->query("
        SELECT id, academy_reference, full_name 
        FROM students 
        WHERE email_verified = 1
          AND id NOT IN (SELECT student_id FROM student_payment_schedules)
        ORDER BY academy_reference, id
    ");
    $students = $stmt->fetchAll();
    
    echo "Found " . count($students) . " students without payment schedules\n";
    
    $initialized = 0;
    $failed = 0;
    
    foreach ($students as $student) {
        // Direct INSERT instead of stored procedure to avoid collation issues
        try {
            // Get tenant settings
            $stmt = $pdo->prepare("
                SELECT monthly_fee, payment_cycle_days 
                FROM tenant_payment_settings 
                WHERE academy_reference = ?
            ");
            $stmt->execute([$student['academy_reference']]);
            $settings = $stmt->fetch();
            
            $monthly_fee = $settings['monthly_fee'] ?? 350.00;
            $payment_cycle_days = $settings['payment_cycle_days'] ?? 30;
            
            // Insert payment schedule directly
            $stmt = $pdo->prepare("
                INSERT INTO student_payment_schedules (
                    student_id, academy_reference, last_payment_date, next_due_date,
                    payment_status, total_amount_due, balance
                ) VALUES (?, ?, CURDATE(), DATE_ADD(CURDATE(), INTERVAL ? DAY), 'active', ?, ?)
                ON DUPLICATE KEY UPDATE 
                    last_payment_date = CURDATE(),
                    next_due_date = DATE_ADD(CURDATE(), INTERVAL ? DAY)
            ");
            $stmt->execute([
                $student['id'], 
                $student['academy_reference'],
                $payment_cycle_days,
                $monthly_fee,
                $monthly_fee,
                $payment_cycle_days
            ]);
            
            $initialized++;
            if ($initialized % 10 == 0) {
                echo "  ✓ Initialized {$initialized} students...\n";
            }
        } catch (Exception $e) {
            $failed++;
            echo "  ✗ Failed for student {$student['full_name']}: " . $e->getMessage() . "\n";
        }
    }
    
    echo "✓ Initialized {$initialized} payment schedules ({$failed} failed)\n\n";
    
    // =====================================================
    // STEP 4: Add suspended column to students table
    // =====================================================
    echo "Step 4: Checking students table structure...\n";
    
    try {
        $pdo->exec("
            ALTER TABLE students 
            ADD COLUMN IF NOT EXISTS suspended TINYINT(1) NOT NULL DEFAULT 0 AFTER email_verified,
            ADD COLUMN IF NOT EXISTS suspended_at TIMESTAMP NULL AFTER suspended,
            ADD COLUMN IF NOT EXISTS suspension_reason VARCHAR(255) NULL AFTER suspended_at,
            ADD INDEX IF NOT EXISTS idx_suspended (suspended)
        ");
        echo "✓ Students table structure verified\n\n";
    } catch (Exception $e) {
        echo "⚠ Warning: " . $e->getMessage() . "\n\n";
    }
    
    // =====================================================
    // STEP 5: Validation & Summary
    // =====================================================
    echo "Step 5: Validation...\n";
    
    // Count initialized schedules
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM student_payment_schedules");
    $schedule_count = $stmt->fetch()['count'];
    
    // Count tenant settings
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM tenant_payment_settings");
    $settings_count = $stmt->fetch()['count'];
    
    echo "✓ Payment schedules: {$schedule_count}\n";
    echo "✓ Tenant settings: {$settings_count}\n\n";
    
    // =====================================================
    // FINAL SUMMARY
    // =====================================================
    echo "========================================\n";
    echo "✅ INITIALIZATION COMPLETE!\n";
    echo "========================================\n\n";
    
    echo "Next Steps:\n";
    echo "1. Configure cron job: 0 8 * * * php cron/payment_reminder_cron.php\n";
    echo "2. Test reminder: php test_payment_reminder.php\n";
    echo "3. Access admin dashboard: admin/settings/payment_settings.php\n";
    echo "4. Review logs: logs/payment_reminder_cron.log\n\n";
    
    echo "System is ready! 🎉\n\n";
    
} catch (Exception $e) {
    echo "\n✗ CRITICAL ERROR: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n\n";
    exit(1);
}

