<?php
/**
 * Wallet & Withdrawal System Functions
 * Handles admin wallet, earnings tracking, and withdrawals
 * 
 * Note: Configuration constants are defined in config.php
 * Required constants:
 * - PLATFORM_FEE_FIXED
 * - PLATFORM_MOMO_NUMBER
 * - MIN_WITHDRAWAL_AMOUNT
 * - WITHDRAWAL_PROCESSING_FEE
 */

/**
 * Credit admin wallet when payment is successful
 */
function creditAdminWallet($payment_id, $admin_id = 1, $academy_reference = 'KINE') {
    try {
        $pdo = getDB();
        
        // Use stored procedure for atomic transaction
        $stmt = $pdo->prepare("CALL sp_credit_admin_wallet(?, ?, ?)");
        $stmt->execute([$payment_id, $admin_id, $academy_reference]);
        
        return [
            'success' => true,
            'message' => 'Admin wallet credited successfully'
        ];
    } catch (Exception $e) {
        error_log("Error crediting admin wallet: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Get admin wallet balance and details
 */
function getAdminWallet($admin_id) {
    try {
        $pdo = getDB();
        
        $stmt = $pdo->prepare("
            SELECT * FROM admin_wallet WHERE admin_id = ?
        ");
        $stmt->execute([$admin_id]);
        $wallet = $stmt->fetch();
        
        if (!$wallet) {
            // Create wallet if doesn't exist
            $stmt = $pdo->prepare("
                INSERT INTO admin_wallet (admin_id, balance, currency)
                VALUES (?, 0.00, 'SZL')
            ");
            $stmt->execute([$admin_id]);
            
            return [
                'admin_id' => $admin_id,
                'balance' => 0.00,
                'total_earned' => 0.00,
                'total_withdrawn' => 0.00,
                'pending_withdrawal' => 0.00,
                'currency' => 'SZL',
                'momo_number' => null,
                'account_verified' => false
            ];
        }
        
        return $wallet;
    } catch (Exception $e) {
        error_log("Error getting admin wallet: " . $e->getMessage());
        return null;
    }
}

/**
 * Get wallet transaction history
 */
function getWalletTransactions($admin_id, $limit = 50, $offset = 0) {
    try {
        $pdo = getDB();
        
        $stmt = $pdo->prepare("
            SELECT 
                wt.*,
                p.external_id as payment_reference,
                wr.momo_reference_id as withdrawal_reference
            FROM wallet_transactions wt
            LEFT JOIN payments p ON wt.payment_id = p.id
            LEFT JOIN withdrawal_requests wr ON wt.withdrawal_id = wr.id
            WHERE wt.admin_id = ?
            ORDER BY wt.created_at DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$admin_id, $limit, $offset]);
        
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting wallet transactions: " . $e->getMessage());
        return [];
    }
}

/**
 * Get platform fees summary
 */
function getPlatformFees($admin_id = null, $start_date = null, $end_date = null) {
    try {
        $pdo = getDB();
        
        $sql = "
            SELECT 
                DATE(created_at) as date,
                academy_reference,
                COUNT(*) as transaction_count,
                SUM(gross_amount) as total_gross,
                SUM(fee_amount) as total_fees,
                SUM(net_amount) as total_net
            FROM platform_fees
            WHERE 1=1
        ";
        
        $params = [];
        
        if ($admin_id) {
            $sql .= " AND admin_id = ?";
            $params[] = $admin_id;
        }
        
        if ($start_date) {
            $sql .= " AND created_at >= ?";
            $params[] = $start_date;
        }
        
        if ($end_date) {
            $sql .= " AND created_at <= ?";
            $params[] = $end_date;
        }
        
        $sql .= " GROUP BY DATE(created_at), academy_reference ORDER BY date DESC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting platform fees: " . $e->getMessage());
        return [];
    }
}

/**
 * Request withdrawal
 */
function requestWithdrawal($admin_id, $amount, $momo_number) {
    try {
        $pdo = getDB();
        
        // Validate amount
        if ($amount < MIN_WITHDRAWAL_AMOUNT) {
            return [
                'success' => false,
                'error' => 'Minimum withdrawal amount is SZL ' . MIN_WITHDRAWAL_AMOUNT
            ];
        }
        
        // Validate MoMo number (Eswatini format: 268XXXXXXXX or 76XXXXXXX or 78XXXXXXX)
        if (!preg_match('/^(268[0-9]{8}|7[68][0-9]{7})$/', $momo_number)) {
            return [
                'success' => false,
                'error' => 'Invalid MoMo number format'
            ];
        }
        
        // Start transaction
        $pdo->beginTransaction();
        
        // Lock wallet and check balance
        $stmt = $pdo->prepare("
            SELECT balance FROM admin_wallet 
            WHERE admin_id = ? FOR UPDATE
        ");
        $stmt->execute([$admin_id]);
        $wallet = $stmt->fetch();
        
        if (!$wallet || $wallet['balance'] < $amount) {
            $pdo->rollBack();
            return [
                'success' => false,
                'error' => 'Insufficient balance. Available: SZL ' . number_format($wallet['balance'] ?? 0, 2)
            ];
        }
        
        $balance_before = $wallet['balance'];
        $balance_after = $balance_before - $amount;
        
        // Deduct from wallet (reserve funds)
        $stmt = $pdo->prepare("
            UPDATE admin_wallet 
            SET balance = balance - ?,
                pending_withdrawal = pending_withdrawal + ?,
                updated_at = NOW()
            WHERE admin_id = ?
        ");
        $stmt->execute([$amount, $amount, $admin_id]);
        
        // Create withdrawal request
        $stmt = $pdo->prepare("
            INSERT INTO withdrawal_requests 
            (admin_id, amount, momo_number, status)
            VALUES (?, ?, ?, 'PENDING')
        ");
        $stmt->execute([$admin_id, $amount, $momo_number]);
        $withdrawal_id = $pdo->lastInsertId();
        
        // Record transaction
        $stmt = $pdo->prepare("
            INSERT INTO wallet_transactions 
            (admin_id, withdrawal_id, type, amount, balance_before, balance_after, description)
            VALUES (?, ?, 'DEBIT', ?, ?, ?, 'Withdrawal request initiated')
        ");
        $stmt->execute([
            $admin_id,
            $withdrawal_id,
            $amount,
            $balance_before,
            $balance_after
        ]);
        
        $pdo->commit();
        
        // Process withdrawal via MTN MoMo
        $result = processMoMoWithdrawal($withdrawal_id, $admin_id, $amount, $momo_number);
        
        return $result;
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Error requesting withdrawal: " . $e->getMessage());
        return [
            'success' => false,
            'error' => 'System error: ' . $e->getMessage()
        ];
    }
}

/**
 * Process MTN MoMo withdrawal (Disbursement API)
 */
function processMoMoWithdrawal($withdrawal_id, $admin_id, $amount, $momo_number) {
    try {
        // Get disbursement access token
        $token = getMoMoDisbursementToken();
        
        if (!$token) {
            return [
                'success' => false,
                'error' => 'Failed to get disbursement token'
            ];
        }
        
        // Generate reference ID
        $reference_id = generateUuidV4();
        
        // Prepare transfer request
        $transfer_url = MOMO_DISBURSEMENT_BASE_URL . '/v1_0/transfer';
        
        // Ensure MoMo number is in correct format (remove 268 prefix if present)
        $momo_number_clean = $momo_number;
        if (substr($momo_number, 0, 3) === '268') {
            $momo_number_clean = substr($momo_number, 3);
        }
        
        $transfer_data = [
            'amount' => (string)number_format($amount, 2, '.', ''),
            'currency' => 'EUR', // Change to 'SZL' in production if supported
            'externalId' => 'WD_' . $withdrawal_id . '_' . time(),
            'payee' => [
                'partyIdType' => 'MSISDN',
                'partyId' => $momo_number_clean
            ],
            'payerMessage' => 'Kine Academy Withdrawal',
            'payeeNote' => 'Withdrawal from Kine Academy Platform'
        ];
        
        $ch = curl_init($transfer_url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $token,
                'X-Reference-Id: ' . $reference_id,
                'X-Target-Environment: ' . MOMO_ENV,
                'Content-Type: application/json',
                'Ocp-Apim-Subscription-Key: ' . MOMO_DISBURSEMENT_SUBSCRIPTION_KEY
            ],
            CURLOPT_POSTFIELDS => json_encode($transfer_data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        $pdo = getDB();
        
        // 202 Accepted means request was successful
        if ($http_code === 202 || $http_code === 200) {
            // Update withdrawal request
            $stmt = $pdo->prepare("
                UPDATE withdrawal_requests 
                SET status = 'PROCESSING',
                    momo_reference_id = ?,
                    processed_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$reference_id, $withdrawal_id]);
            
            return [
                'success' => true,
                'message' => 'Withdrawal is being processed',
                'reference_id' => $reference_id,
                'withdrawal_id' => $withdrawal_id
            ];
        } else {
            // Withdrawal failed - refund to wallet
            $stmt = $pdo->prepare("CALL sp_refund_withdrawal(?)");
            $stmt->execute([$withdrawal_id]);
            
            $stmt = $pdo->prepare("
                UPDATE withdrawal_requests 
                SET status = 'FAILED',
                    error_message = ?,
                    momo_response = ?,
                    processed_at = NOW()
                WHERE id = ?
            ");
            $error_msg = "HTTP {$http_code}: {$curl_error} | Response: {$response}";
            $stmt->execute([$error_msg, $response, $withdrawal_id]);
            
            return [
                'success' => false,
                'error' => 'Withdrawal failed. Funds have been refunded to your wallet.',
                'details' => $error_msg
            ];
        }
        
    } catch (Exception $e) {
        error_log("Error processing MoMo withdrawal: " . $e->getMessage());
        
        // Refund on error
        try {
            $pdo = getDB();
            $stmt = $pdo->prepare("CALL sp_refund_withdrawal(?)");
            $stmt->execute([$withdrawal_id]);
        } catch (Exception $refund_error) {
            error_log("Error refunding withdrawal: " . $refund_error->getMessage());
        }
        
        return [
            'success' => false,
            'error' => 'System error during withdrawal. Please contact support.'
        ];
    }
}

/**
 * Get MTN MoMo Disbursement access token
 */
function getMoMoDisbursementToken() {
    try {
        $token_url = MOMO_DISBURSEMENT_BASE_URL . '/token/';
        $auth = base64_encode(MOMO_DISBURSEMENT_USER_ID . ':' . MOMO_DISBURSEMENT_API_KEY);
        
        $ch = curl_init($token_url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Basic ' . $auth,
                'Ocp-Apim-Subscription-Key: ' . MOMO_DISBURSEMENT_SUBSCRIPTION_KEY
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 10
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200) {
            $data = json_decode($response, true);
            return $data['access_token'] ?? null;
        }
        
        error_log("Failed to get disbursement token. HTTP: {$http_code}, Response: {$response}");
        return null;
        
    } catch (Exception $e) {
        error_log("Error getting disbursement token: " . $e->getMessage());
        return null;
    }
}

/**
 * Check withdrawal status
 */
function checkWithdrawalStatus($reference_id) {
    try {
        $token = getMoMoDisbursementToken();
        
        if (!$token) {
            return null;
        }
        
        $status_url = MOMO_DISBURSEMENT_BASE_URL . '/v1_0/transfer/' . $reference_id;
        
        $ch = curl_init($status_url);
        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $token,
                'X-Target-Environment: ' . MOMO_ENV,
                'Ocp-Apim-Subscription-Key: ' . MOMO_DISBURSEMENT_SUBSCRIPTION_KEY
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 10
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200) {
            return json_decode($response, true);
        }
        
        return null;
        
    } catch (Exception $e) {
        error_log("Error checking withdrawal status: " . $e->getMessage());
        return null;
    }
}

/**
 * Get pending withdrawals for admin
 */
function getPendingWithdrawals($admin_id) {
    try {
        $pdo = getDB();
        
        $stmt = $pdo->prepare("
            SELECT * FROM withdrawal_requests 
            WHERE admin_id = ? 
            AND status IN ('PENDING', 'PROCESSING')
            ORDER BY requested_at DESC
        ");
        $stmt->execute([$admin_id]);
        
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting pending withdrawals: " . $e->getMessage());
        return [];
    }
}

/**
 * Get withdrawal history
 */
function getWithdrawalHistory($admin_id, $limit = 50, $offset = 0) {
    try {
        $pdo = getDB();
        
        $stmt = $pdo->prepare("
            SELECT * FROM withdrawal_requests 
            WHERE admin_id = ?
            ORDER BY requested_at DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$admin_id, $limit, $offset]);
        
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting withdrawal history: " . $e->getMessage());
        return [];
    }
}

/**
 * Get academy performance stats
 */
function getAcademyPerformance($academy_reference = null) {
    try {
        $pdo = getDB();
        
        if ($academy_reference) {
            $stmt = $pdo->prepare("SELECT * FROM v_academy_performance WHERE reference_code = ?");
            $stmt->execute([$academy_reference]);
            return $stmt->fetch();
        } else {
            $stmt = $pdo->query("SELECT * FROM v_academy_performance ORDER BY total_revenue DESC");
            return $stmt->fetchAll();
        }
    } catch (Exception $e) {
        error_log("Error getting academy performance: " . $e->getMessage());
        return null;
    }
}

/**
 * Update admin MoMo number
 */
function updateAdminMoMoNumber($admin_id, $momo_number) {
    try {
        // Validate MoMo number
        if (!preg_match('/^(268[0-9]{8}|7[68][0-9]{7})$/', $momo_number)) {
            return [
                'success' => false,
                'error' => 'Invalid MoMo number format'
            ];
        }
        
        $pdo = getDB();
        
        $stmt = $pdo->prepare("
            UPDATE admin_wallet 
            SET momo_number = ?,
                account_verified = FALSE,
                updated_at = NOW()
            WHERE admin_id = ?
        ");
        $stmt->execute([$momo_number, $admin_id]);
        
        return [
            'success' => true,
            'message' => 'MoMo number updated successfully'
        ];
        
    } catch (Exception $e) {
        error_log("Error updating MoMo number: " . $e->getMessage());
        return [
            'success' => false,
            'error' => 'Failed to update MoMo number'
        ];
    }
}