<?php
/**
 * URL Helper Functions
 * Automatically detects and preserves the correct base URL for multi-tenant setup
 */

/**
 * Get the base URL for the application
 * Uses SCRIPT_NAME (not REQUEST_URI) to get the actual file location
 * This correctly handles clean URLs and rewrites
 */
function getBaseURL() {
    // Use the actual script location, not the clean URL
    $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    
    // SCRIPT_NAME gives us the actual PHP file location after rewrite
    // For /test/register -> /Multi-Tanent/register_student.php
    // dirname() will give us /Multi-Tanent (correct!)
    $scriptDir = dirname($_SERVER['SCRIPT_NAME']);
    $scriptDir = str_replace('\\', '/', $scriptDir); // Fix Windows paths
    $scriptDir = rtrim($scriptDir, '/');
    
    // Build the base URL from the actual script location
    $baseUrl = $protocol . '://' . $host . $scriptDir;
    
    // Log for debugging
    error_log("URL Helper - Protocol: $protocol, Host: $host, ScriptDir: $scriptDir, BaseURL: $baseUrl");
    
    return $baseUrl;
}

/**
 * Generate a full URL to a specific file
 */
function generateURL($file) {
    $base = getBaseURL();
    $file = ltrim($file, '/');
    return $base . '/' . $file;
}

/**
 * Check if we're running on localhost
 */
function isLocalhost() {
    return (
        $_SERVER['SERVER_NAME'] === 'localhost' ||
        $_SERVER['SERVER_NAME'] === '127.0.0.1' ||
        strpos($_SERVER['SERVER_NAME'], 'localhost') !== false ||
        strpos($_SERVER['HTTP_HOST'], 'localhost') !== false ||
        strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false
    );
}

