<?php
/**
 * Multi-Tenant File System Manager
 * 
 * Handles all file operations for tenant-isolated storage
 * Includes security checks, path traversal prevention, and logging
 * 
 * @package Multi-Tenant System
 * @version 1.0
 */

class TenantFileSystem {
    
    /**
     * Base directory for all tenant files
     * @var string
     */
    private $baseDir;
    
    /**
     * Allowed file types per category
     * @var array
     */
    private $allowedFileTypes = [
        'documents' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv'],
        'profile_photos' => ['jpg', 'jpeg', 'png', 'gif', 'webp'],
        'payment_receipts' => ['pdf', 'jpg', 'jpeg', 'png'],
        'reports' => ['pdf', 'xls', 'xlsx', 'csv', 'html'],
        'backups' => ['sql', 'zip', 'tar', 'gz'],
        'logs' => ['log', 'txt'],
        'temp' => ['*'] // Allow all for temp
    ];
    
    /**
     * Maximum file sizes per category (in bytes)
     * @var array
     */
    private $maxFileSizes = [
        'documents' => 10485760, // 10MB
        'profile_photos' => 5242880, // 5MB
        'payment_receipts' => 5242880, // 5MB
        'reports' => 20971520, // 20MB
        'backups' => 104857600, // 100MB
        'logs' => 10485760, // 10MB
        'temp' => 20971520 // 20MB
    ];
    
    /**
     * Valid directory types
     * @var array
     */
    private $validTypes = [
        'documents',
        'profile_photos',
        'payment_receipts',
        'reports',
        'backups',
        'logs',
        'temp'
    ];
    
    /**
     * Database connection for logging
     * @var PDO
     */
    private $db;
    
    /**
     * Constructor
     * 
     * @param PDO $db Database connection for logging
     * @param string $baseDir Base directory for tenant files (default: tenants/)
     */
    public function __construct($db = null, $baseDir = null) {
        $this->db = $db;
        
        // Set base directory (default to project root / tenants)
        if ($baseDir === null) {
            $this->baseDir = dirname(__DIR__) . '/tenants';
        } else {
            $this->baseDir = rtrim($baseDir, '/');
        }
        
        // Ensure base directory exists
        if (!file_exists($this->baseDir)) {
            mkdir($this->baseDir, 0755, true);
        }
    }
    
    /**
     * Create complete directory structure for a tenant
     * 
     * @param string $tenant_id Tenant identifier (academy_reference)
     * @return array Result with success status and message
     */
    public function createTenantDirectory($tenant_id) {
        try {
            // Validate tenant ID
            if (!$this->validateTenantId($tenant_id)) {
                return [
                    'success' => false,
                    'error' => 'Invalid tenant ID format'
                ];
            }
            
            $tenantPath = $this->baseDir . '/' . $tenant_id;
            
            // Check if tenant directory already exists
            if (file_exists($tenantPath)) {
                return [
                    'success' => true,
                    'message' => 'Tenant directory already exists',
                    'path' => $tenantPath
                ];
            }
            
            // Create main tenant directory
            if (!mkdir($tenantPath, 0755, true)) {
                throw new Exception("Failed to create tenant directory");
            }
            
            // Create subdirectories
            $directories = [
                'uploads/documents',
                'uploads/profile_photos',
                'uploads/payment_receipts',
                'uploads/reports',
                'backups',
                'logs',
                'temp'
            ];
            
            foreach ($directories as $dir) {
                $fullPath = $tenantPath . '/' . $dir;
                if (!mkdir($fullPath, 0755, true)) {
                    throw new Exception("Failed to create directory: {$dir}");
                }
            }
            
            // Create config.json with default settings
            $config = [
                'tenant_id' => $tenant_id,
                'created_at' => date('Y-m-d H:i:s'),
                'storage_quota' => 1073741824, // 1GB default
                'file_retention_days' => 365,
                'auto_cleanup_temp' => true,
                'temp_cleanup_days' => 7
            ];
            
            file_put_contents(
                $tenantPath . '/config.json',
                json_encode($config, JSON_PRETTY_PRINT)
            );
            
            // Create .htaccess for security
            $htaccess = "# Deny direct access to files\n";
            $htaccess .= "Order Deny,Allow\n";
            $htaccess .= "Deny from all\n";
            $htaccess .= "\n# Allow access to images for display\n";
            $htaccess .= "<FilesMatch \"\.(jpg|jpeg|png|gif|webp)$\">\n";
            $htaccess .= "    Allow from all\n";
            $htaccess .= "</FilesMatch>\n";
            
            file_put_contents($tenantPath . '/.htaccess', $htaccess);
            
            // Log the creation
            $this->logOperation($tenant_id, 'directory_created', 'Tenant directory structure created');
            
            return [
                'success' => true,
                'message' => 'Tenant directory structure created successfully',
                'path' => $tenantPath
            ];
            
        } catch (Exception $e) {
            $this->logError($tenant_id, 'create_directory_error', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get the full path for a specific tenant directory type
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $type Directory type (documents, profile_photos, etc.)
     * @return string|false Full path or false on error
     */
    public function getTenantPath($tenant_id, $type = null) {
        // Validate tenant ID
        if (!$this->validateTenantId($tenant_id)) {
            return false;
        }
        
        $basePath = $this->baseDir . '/' . $tenant_id;
        
        // Return base path if no type specified
        if ($type === null) {
            return $basePath;
        }
        
        // Validate type
        if (!in_array($type, $this->validTypes)) {
            return false;
        }
        
        // Construct path based on type
        if (in_array($type, ['documents', 'profile_photos', 'payment_receipts', 'reports'])) {
            return $basePath . '/uploads/' . $type;
        } else {
            return $basePath . '/' . $type;
        }
    }
    
    /**
     * Upload a file to tenant storage
     * 
     * @param string $tenant_id Tenant identifier
     * @param array $file File array from $_FILES
     * @param string $type File type/category
     * @param string $customFilename Optional custom filename
     * @return array Result with success status and file info
     */
    public function uploadFile($tenant_id, $file, $type, $customFilename = null) {
        try {
            // Validate inputs
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            if (!in_array($type, $this->validTypes)) {
                throw new Exception("Invalid file type category");
            }
            
            // Check if file was uploaded
            if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
                throw new Exception("No file uploaded or invalid upload");
            }
            
            // Check for upload errors
            if ($file['error'] !== UPLOAD_ERR_OK) {
                throw new Exception($this->getUploadErrorMessage($file['error']));
            }
            
            // Validate file size
            if ($file['size'] > $this->maxFileSizes[$type]) {
                $maxSize = $this->formatBytes($this->maxFileSizes[$type]);
                throw new Exception("File size exceeds maximum allowed ({$maxSize})");
            }
            
            // Get file extension
            $originalName = basename($file['name']);
            $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
            
            // Validate file extension
            if (!$this->isAllowedFileType($extension, $type)) {
                throw new Exception("File type .{$extension} not allowed for {$type}");
            }
            
            // Sanitize filename
            if ($customFilename) {
                $filename = $this->sanitizeFilename($customFilename);
            } else {
                $filename = $this->sanitizeFilename(pathinfo($originalName, PATHINFO_FILENAME));
            }
            
            // Add timestamp to prevent overwrites
            $timestamp = time();
            $finalFilename = $filename . '_' . $timestamp . '.' . $extension;
            
            // Get destination path
            $destPath = $this->getTenantPath($tenant_id, $type);
            
            // Create directory if it doesn't exist
            if (!file_exists($destPath)) {
                $createResult = $this->createTenantDirectory($tenant_id);
                if (!$createResult['success']) {
                    throw new Exception("Failed to create tenant directory");
                }
            }
            
            $destFile = $destPath . '/' . $finalFilename;
            
            // Check storage quota
            if (!$this->checkStorageQuota($tenant_id, $file['size'])) {
                throw new Exception("Storage quota exceeded");
            }
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $destFile)) {
                throw new Exception("Failed to move uploaded file");
            }
            
            // Set proper permissions
            chmod($destFile, 0644);
            
            // Generate relative path for database storage
            $relativePath = $tenant_id . '/' . 
                           (in_array($type, ['documents', 'profile_photos', 'payment_receipts', 'reports']) 
                            ? 'uploads/' : '') . 
                           $type . '/' . $finalFilename;
            
            // Log the upload
            $this->logOperation(
                $tenant_id, 
                'file_uploaded', 
                "File uploaded: {$finalFilename} ({$type})",
                ['filename' => $finalFilename, 'size' => $file['size'], 'type' => $type]
            );
            
            return [
                'success' => true,
                'message' => 'File uploaded successfully',
                'filename' => $finalFilename,
                'original_name' => $originalName,
                'path' => $destFile,
                'relative_path' => $relativePath,
                'size' => $file['size'],
                'type' => $type,
                'url' => $this->getFileUrl($tenant_id, $type, $finalFilename)
            ];
            
        } catch (Exception $e) {
            $this->logError($tenant_id, 'upload_error', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete a file from tenant storage
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $filepath Relative path to file or just filename
     * @param string $type File type (optional if filepath includes full path)
     * @return array Result with success status
     */
    public function deleteFile($tenant_id, $filepath, $type = null) {
        try {
            // Validate tenant ID
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            // Validate tenant access to this file
            if (!$this->validateTenantAccess($tenant_id, $filepath)) {
                throw new Exception("Access denied: File does not belong to this tenant");
            }
            
            // Construct full path
            if ($type && in_array($type, $this->validTypes)) {
                $fullPath = $this->getTenantPath($tenant_id, $type) . '/' . basename($filepath);
            } else {
                // Filepath might be relative from tenant root
                $fullPath = $this->baseDir . '/' . $tenant_id . '/' . ltrim($filepath, '/');
            }
            
            // Security check: Ensure path is within tenant directory
            $realPath = realpath($fullPath);
            $tenantBase = realpath($this->baseDir . '/' . $tenant_id);
            
            if ($realPath === false || strpos($realPath, $tenantBase) !== 0) {
                throw new Exception("Invalid file path or path traversal attempt detected");
            }
            
            // Check if file exists
            if (!file_exists($fullPath)) {
                throw new Exception("File not found");
            }
            
            // Delete the file
            if (!unlink($fullPath)) {
                throw new Exception("Failed to delete file");
            }
            
            // Log the deletion
            $this->logOperation(
                $tenant_id,
                'file_deleted',
                "File deleted: " . basename($filepath),
                ['filepath' => $filepath]
            );
            
            return [
                'success' => true,
                'message' => 'File deleted successfully'
            ];
            
        } catch (Exception $e) {
            $this->logError($tenant_id, 'delete_error', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get list of files for a tenant
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $type File type category
     * @param array $options Options (sort, filter, limit, etc.)
     * @return array List of files with metadata
     */
    public function getTenantFiles($tenant_id, $type = null, $options = []) {
        try {
            // Validate tenant ID
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            // Get directory path
            if ($type) {
                if (!in_array($type, $this->validTypes)) {
                    throw new Exception("Invalid file type");
                }
                $scanPath = $this->getTenantPath($tenant_id, $type);
            } else {
                $scanPath = $this->getTenantPath($tenant_id);
            }
            
            // Check if directory exists
            if (!file_exists($scanPath)) {
                return [
                    'success' => true,
                    'files' => [],
                    'count' => 0
                ];
            }
            
            $files = [];
            
            // Scan directory recursively if no type specified
            if ($type === null) {
                $files = $this->scanDirectoryRecursive($scanPath, $tenant_id);
            } else {
                // Scan specific type directory
                $items = scandir($scanPath);
                foreach ($items as $item) {
                    if ($item === '.' || $item === '..' || $item === '.htaccess') {
                        continue;
                    }
                    
                    $fullPath = $scanPath . '/' . $item;
                    if (is_file($fullPath)) {
                        $files[] = $this->getFileMetadata($fullPath, $tenant_id, $type);
                    }
                }
            }
            
            // Apply filters
            if (isset($options['extension'])) {
                $files = array_filter($files, function($file) use ($options) {
                    return $file['extension'] === strtolower($options['extension']);
                });
            }
            
            if (isset($options['min_size'])) {
                $files = array_filter($files, function($file) use ($options) {
                    return $file['size'] >= $options['min_size'];
                });
            }
            
            if (isset($options['max_size'])) {
                $files = array_filter($files, function($file) use ($options) {
                    return $file['size'] <= $options['max_size'];
                });
            }
            
            // Sort files
            $sortBy = $options['sort_by'] ?? 'modified';
            $sortOrder = $options['sort_order'] ?? 'desc';
            
            usort($files, function($a, $b) use ($sortBy, $sortOrder) {
                $result = 0;
                switch ($sortBy) {
                    case 'name':
                        $result = strcmp($a['name'], $b['name']);
                        break;
                    case 'size':
                        $result = $a['size'] - $b['size'];
                        break;
                    case 'modified':
                    default:
                        $result = $a['modified_timestamp'] - $b['modified_timestamp'];
                        break;
                }
                return $sortOrder === 'asc' ? $result : -$result;
            });
            
            // Apply limit
            if (isset($options['limit'])) {
                $files = array_slice($files, 0, $options['limit']);
            }
            
            return [
                'success' => true,
                'files' => array_values($files),
                'count' => count($files)
            ];
            
        } catch (Exception $e) {
            $this->logError($tenant_id, 'list_files_error', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Validate tenant access to a file
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $filepath File path to validate
     * @return bool True if access is valid
     */
    public function validateTenantAccess($tenant_id, $filepath) {
        try {
            // Validate tenant ID
            if (!$this->validateTenantId($tenant_id)) {
                return false;
            }
            
            // Get tenant base directory
            $tenantBase = realpath($this->baseDir . '/' . $tenant_id);
            
            if ($tenantBase === false) {
                return false;
            }
            
            // Construct full path
            if (strpos($filepath, $this->baseDir) === 0) {
                // Already full path
                $fullPath = $filepath;
            } else {
                // Relative path from tenant root
                $fullPath = $tenantBase . '/' . ltrim($filepath, '/');
            }
            
            // Get real path (resolves .. and symlinks)
            $realPath = realpath($fullPath);
            
            // If file doesn't exist, check parent directory
            if ($realPath === false) {
                $realPath = realpath(dirname($fullPath));
                if ($realPath === false) {
                    return false;
                }
            }
            
            // Ensure the real path is within tenant directory
            if (strpos($realPath, $tenantBase) !== 0) {
                $this->logError($tenant_id, 'access_violation', "Path traversal attempt: {$filepath}");
                return false;
            }
            
            return true;
            
        } catch (Exception $e) {
            $this->logError($tenant_id, 'validation_error', $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get storage usage for a tenant
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Storage statistics
     */
    public function getStorageUsage($tenant_id) {
        try {
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            $tenantPath = $this->getTenantPath($tenant_id);
            
            if (!file_exists($tenantPath)) {
                return [
                    'success' => true,
                    'used' => 0,
                    'used_formatted' => '0 B',
                    'quota' => $this->getStorageQuota($tenant_id),
                    'quota_formatted' => $this->formatBytes($this->getStorageQuota($tenant_id)),
                    'percentage' => 0
                ];
            }
            
            $totalSize = $this->getDirectorySize($tenantPath);
            $quota = $this->getStorageQuota($tenant_id);
            $percentage = $quota > 0 ? round(($totalSize / $quota) * 100, 2) : 0;
            
            return [
                'success' => true,
                'used' => $totalSize,
                'used_formatted' => $this->formatBytes($totalSize),
                'quota' => $quota,
                'quota_formatted' => $this->formatBytes($quota),
                'percentage' => $percentage,
                'available' => max(0, $quota - $totalSize),
                'available_formatted' => $this->formatBytes(max(0, $quota - $totalSize))
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Clean up temporary files older than specified days
     * 
     * @param string $tenant_id Tenant identifier
     * @param int $days Delete files older than this many days
     * @return array Result with count of deleted files
     */
    public function cleanupTempFiles($tenant_id, $days = 7) {
        try {
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            $tempPath = $this->getTenantPath($tenant_id, 'temp');
            
            if (!file_exists($tempPath)) {
                return [
                    'success' => true,
                    'deleted' => 0,
                    'message' => 'No temp directory found'
                ];
            }
            
            $cutoffTime = time() - ($days * 86400); // days to seconds
            $deleted = 0;
            $totalSize = 0;
            
            $files = scandir($tempPath);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..') {
                    continue;
                }
                
                $fullPath = $tempPath . '/' . $file;
                
                if (is_file($fullPath) && filemtime($fullPath) < $cutoffTime) {
                    $size = filesize($fullPath);
                    if (unlink($fullPath)) {
                        $deleted++;
                        $totalSize += $size;
                    }
                }
            }
            
            $this->logOperation(
                $tenant_id,
                'temp_cleanup',
                "Cleaned up {$deleted} temporary files",
                ['deleted' => $deleted, 'size' => $totalSize, 'days' => $days]
            );
            
            return [
                'success' => true,
                'deleted' => $deleted,
                'size_freed' => $totalSize,
                'size_freed_formatted' => $this->formatBytes($totalSize),
                'message' => "Deleted {$deleted} temporary files older than {$days} days"
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get file URL for serving
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $type File type
     * @param string $filename Filename
     * @return string URL to access file
     */
    public function getFileUrl($tenant_id, $type, $filename) {
        // Generate secure file serving URL
        // You can implement a file serving script (e.g., serve_file.php)
        return '/serve_file.php?tenant=' . urlencode($tenant_id) . 
               '&type=' . urlencode($type) . 
               '&file=' . urlencode($filename);
    }
    
    // ========================================================================
    // PRIVATE HELPER METHODS
    // ========================================================================
    
    /**
     * Validate tenant ID format
     */
    private function validateTenantId($tenant_id) {
        // Allow alphanumeric, underscore, hyphen
        return preg_match('/^[a-zA-Z0-9_-]+$/', $tenant_id) && strlen($tenant_id) <= 50;
    }
    
    /**
     * Sanitize filename
     */
    private function sanitizeFilename($filename) {
        // Remove any path components
        $filename = basename($filename);
        
        // Replace spaces with underscores
        $filename = str_replace(' ', '_', $filename);
        
        // Remove any character that isn't alphanumeric, underscore, or hyphen
        $filename = preg_replace('/[^a-zA-Z0-9_-]/', '', $filename);
        
        // Limit length
        $filename = substr($filename, 0, 100);
        
        return $filename;
    }
    
    /**
     * Check if file extension is allowed for type
     */
    private function isAllowedFileType($extension, $type) {
        if (!isset($this->allowedFileTypes[$type])) {
            return false;
        }
        
        $allowed = $this->allowedFileTypes[$type];
        
        // Check for wildcard (allow all)
        if (in_array('*', $allowed)) {
            return true;
        }
        
        return in_array(strtolower($extension), $allowed);
    }
    
    /**
     * Get upload error message
     */
    private function getUploadErrorMessage($error_code) {
        switch ($error_code) {
            case UPLOAD_ERR_INI_SIZE:
                return 'File exceeds upload_max_filesize directive in php.ini';
            case UPLOAD_ERR_FORM_SIZE:
                return 'File exceeds MAX_FILE_SIZE directive in HTML form';
            case UPLOAD_ERR_PARTIAL:
                return 'File was only partially uploaded';
            case UPLOAD_ERR_NO_FILE:
                return 'No file was uploaded';
            case UPLOAD_ERR_NO_TMP_DIR:
                return 'Missing temporary folder';
            case UPLOAD_ERR_CANT_WRITE:
                return 'Failed to write file to disk';
            case UPLOAD_ERR_EXTENSION:
                return 'File upload stopped by extension';
            default:
                return 'Unknown upload error';
        }
    }
    
    /**
     * Get storage quota for tenant
     */
    private function getStorageQuota($tenant_id) {
        $configPath = $this->baseDir . '/' . $tenant_id . '/config.json';
        
        if (file_exists($configPath)) {
            $config = json_decode(file_get_contents($configPath), true);
            return $config['storage_quota'] ?? 1073741824; // Default 1GB
        }
        
        return 1073741824; // Default 1GB
    }
    
    /**
     * Check if tenant has enough storage quota
     */
    private function checkStorageQuota($tenant_id, $newFileSize) {
        $usage = $this->getStorageUsage($tenant_id);
        
        if (!$usage['success']) {
            return true; // Allow if we can't check
        }
        
        return ($usage['used'] + $newFileSize) <= $usage['quota'];
    }
    
    /**
     * Get directory size recursively
     */
    private function getDirectorySize($path) {
        $size = 0;
        
        if (!is_dir($path)) {
            return filesize($path);
        }
        
        $items = scandir($path);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }
            
            $fullPath = $path . '/' . $item;
            
            if (is_file($fullPath)) {
                $size += filesize($fullPath);
            } elseif (is_dir($fullPath)) {
                $size += $this->getDirectorySize($fullPath);
            }
        }
        
        return $size;
    }
    
    /**
     * Format bytes to human readable size
     */
    private function formatBytes($bytes, $precision = 2) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
    
    /**
     * Scan directory recursively
     */
    private function scanDirectoryRecursive($path, $tenant_id) {
        $files = [];
        
        $items = scandir($path);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..' || $item === '.htaccess') {
                continue;
            }
            
            $fullPath = $path . '/' . $item;
            
            if (is_file($fullPath)) {
                $files[] = $this->getFileMetadata($fullPath, $tenant_id);
            } elseif (is_dir($fullPath)) {
                $files = array_merge($files, $this->scanDirectoryRecursive($fullPath, $tenant_id));
            }
        }
        
        return $files;
    }
    
    /**
     * Get file metadata
     */
    private function getFileMetadata($filepath, $tenant_id, $type = null) {
        $filename = basename($filepath);
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $size = filesize($filepath);
        $modified = filemtime($filepath);
        
        // Determine type from path if not provided
        if ($type === null) {
            foreach ($this->validTypes as $validType) {
                if (strpos($filepath, '/' . $validType . '/') !== false) {
                    $type = $validType;
                    break;
                }
            }
        }
        
        return [
            'name' => $filename,
            'extension' => $extension,
            'size' => $size,
            'size_formatted' => $this->formatBytes($size),
            'modified' => date('Y-m-d H:i:s', $modified),
            'modified_timestamp' => $modified,
            'type' => $type,
            'path' => $filepath,
            'relative_path' => str_replace($this->baseDir . '/', '', $filepath),
            'url' => $type ? $this->getFileUrl($tenant_id, $type, $filename) : null,
            'is_image' => in_array($extension, ['jpg', 'jpeg', 'png', 'gif', 'webp'])
        ];
    }
    
    /**
     * Log file operation
     */
    private function logOperation($tenant_id, $operation, $message, $metadata = []) {
        // Log to file
        $logPath = $this->baseDir . '/' . $tenant_id . '/logs';
        if (!file_exists($logPath)) {
            @mkdir($logPath, 0755, true);
        }
        
        $logFile = $logPath . '/filesystem_' . date('Y-m-d') . '.log';
        $logEntry = date('Y-m-d H:i:s') . " [{$operation}] {$message}";
        
        if (!empty($metadata)) {
            $logEntry .= ' | ' . json_encode($metadata);
        }
        
        $logEntry .= PHP_EOL;
        
        @file_put_contents($logFile, $logEntry, FILE_APPEND);
        
        // Log to database if available
        if ($this->db) {
            try {
                $stmt = $this->db->prepare("
                    INSERT INTO file_operations_log 
                    (tenant_id, operation, message, metadata, created_at)
                    VALUES (?, ?, ?, ?, NOW())
                ");
                $stmt->execute([
                    $tenant_id,
                    $operation,
                    $message,
                    json_encode($metadata)
                ]);
            } catch (PDOException $e) {
                // Silent fail - table might not exist
            }
        }
    }
    
    /**
     * Log error
     */
    private function logError($tenant_id, $operation, $error) {
        error_log("TenantFileSystem Error [{$tenant_id}] [{$operation}]: {$error}");
        $this->logOperation($tenant_id, $operation . '_error', $error);
    }
}

