<?php
/**
 * Tenant File Upload System
 * 
 * Secure file upload handler for multi-tenant education platform
 * Includes validation, sanitization, and database tracking
 * 
 * @package Multi-Tenant System
 * @version 1.0
 */

class TenantFileUpload {
    
    /**
     * Database connection
     * @var PDO
     */
    private $db;
    
    /**
     * Tenant Directory Manager
     * @var TenantDirectoryManager
     */
    private $dirManager;
    
    /**
     * File size limits (in bytes)
     * @var array
     */
    private $fileLimits = [
        'document' => 10485760,  // 10MB
        'photo' => 2097152,      // 2MB
        'receipt' => 5242880,    // 5MB
        'report' => 20971520     // 20MB
    ];
    
    /**
     * Allowed MIME types per category
     * @var array
     */
    private $allowedMimeTypes = [
        'document' => [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'image/jpeg',
            'image/png',
            'image/svg+xml',
            'image/webp'
        ],
        'photo' => [
            'image/jpeg',
            'image/jpg',
            'image/png',
            'image/svg+xml',
            'image/webp'
        ],
        'receipt' => [
            'application/pdf',
            'image/jpeg',
            'image/jpg',
            'image/png'
        ],
        'report' => [
            'application/pdf',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        ]
    ];
    
    /**
     * Allowed file extensions per category
     * @var array
     */
    private $allowedExtensions = [
        'document' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'jpg', 'jpeg', 'png', 'svg', 'webp'],
        'photo' => ['jpg', 'jpeg', 'png', 'svg', 'webp'],
        'receipt' => ['pdf', 'jpg', 'jpeg', 'png'],
        'report' => ['pdf', 'xlsx']
    ];
    
    /**
     * Constructor
     * 
     * @param PDO $db Database connection
     * @param TenantDirectoryManager $dirManager Directory manager
     */
    public function __construct($db, $dirManager = null) {
        $this->db = $db;
        $this->dirManager = $dirManager;
        
        // Create directory manager if not provided
        if ($this->dirManager === null) {
            require_once __DIR__ . '/tenant_directory_manager.php';
            $this->dirManager = new TenantDirectoryManager($db);
        }
    }
    
    /**
     * Upload document file
     * 
     * @param string $tenant_id Tenant identifier
     * @param array $file File from $_FILES
     * @param array $metadata Additional metadata (description, tags, related_id, related_type)
     * @return array Upload result
     */
    public function uploadDocument($tenant_id, $file, $metadata = []) {
        try {
            // Validate file
            $validation = $this->validateFile($file, 'document');
            if (!$validation['valid']) {
                throw new Exception($validation['error']);
            }
            
            // Get upload path
            $uploadPath = $this->dirManager->getTenantUploadPath($tenant_id, 'documents');
            if (!$uploadPath) {
                throw new Exception("Invalid tenant or upload path");
            }
            
            // Generate unique filename
            $uniqueFilename = $this->generateUniqueFilename($file['name']);
            $targetPath = $uploadPath . '/' . $uniqueFilename;
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
                throw new Exception("Failed to move uploaded file");
            }
            
            // Set proper permissions
            chmod($targetPath, 0644);
            
            // Calculate relative path
            $relativePath = 'school_' . $tenant_id . '/uploads/documents/' . $uniqueFilename;
            
            // Get file hash
            $fileHash = hash_file('sha256', $targetPath);
            
            // Store in database
            $fileId = $this->storeFileMetadata([
                'tenant_id' => $tenant_id,
                'file_category' => 'document',
                'original_filename' => basename($file['name']),
                'stored_filename' => $uniqueFilename,
                'file_path' => $relativePath,
                'file_size' => $file['size'],
                'mime_type' => $validation['mime_type'],
                'file_hash' => $fileHash,
                'uploaded_by' => $_SESSION['user_id'] ?? null,
                'description' => $metadata['description'] ?? null,
                'tags' => $metadata['tags'] ?? null,
                'related_id' => $metadata['related_id'] ?? null,
                'related_type' => $metadata['related_type'] ?? null
            ]);
            
            // Log upload
            $this->logUpload($tenant_id, $fileId, 'document', $uniqueFilename, 'success');
            
            return [
                'success' => true,
                'file_id' => $fileId,
                'filename' => $uniqueFilename,
                'original_name' => $file['name'],
                'path' => $targetPath,
                'relative_path' => $relativePath,
                'size' => $file['size'],
                'url' => $this->generateFileUrl($tenant_id, 'documents', $uniqueFilename)
            ];
            
        } catch (Exception $e) {
            $this->logUpload($tenant_id, null, 'document', $file['name'] ?? 'unknown', 'error', $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Upload profile photo with image processing
     * 
     * @param string $tenant_id Tenant identifier
     * @param int $student_id Student ID
     * @param array $file File from $_FILES
     * @return array Upload result
     */
    public function uploadProfilePhoto($tenant_id, $student_id, $file) {
        try {
            // Validate file
            $validation = $this->validateFile($file, 'photo');
            if (!$validation['valid']) {
                throw new Exception($validation['error']);
            }
            
            // Additional image validation
            $imageInfo = @getimagesize($file['tmp_name']);
            if ($imageInfo === false) {
                throw new Exception("Invalid image file");
            }
            
            // Check dimensions
            list($width, $height) = $imageInfo;
            if ($width < 100 || $height < 100) {
                throw new Exception("Image too small (minimum 100x100 pixels)");
            }
            
            if ($width > 4000 || $height > 4000) {
                throw new Exception("Image too large (maximum 4000x4000 pixels)");
            }
            
            // Get upload path
            $uploadPath = $this->dirManager->getTenantUploadPath($tenant_id, 'profile_photos');
            if (!$uploadPath) {
                throw new Exception("Invalid tenant or upload path");
            }
            
            // Check for existing photo and delete it
            $this->deleteExistingProfilePhoto($tenant_id, $student_id);
            
            // Generate unique filename with student ID
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $uniqueFilename = 'student_' . $student_id . '_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $extension;
            $targetPath = $uploadPath . '/' . $uniqueFilename;
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
                throw new Exception("Failed to move uploaded file");
            }
            
            // Set proper permissions
            chmod($targetPath, 0644);
            
            // Optional: Resize/optimize image
            $this->optimizeImage($targetPath, $validation['mime_type']);
            
            // Calculate relative path
            $relativePath = 'school_' . $tenant_id . '/uploads/profile_photos/' . $uniqueFilename;
            
            // Get file hash
            $fileHash = hash_file('sha256', $targetPath);
            
            // Store in database
            $fileId = $this->storeFileMetadata([
                'tenant_id' => $tenant_id,
                'file_category' => 'photo',
                'original_filename' => basename($file['name']),
                'stored_filename' => $uniqueFilename,
                'file_path' => $relativePath,
                'file_size' => filesize($targetPath),
                'mime_type' => $validation['mime_type'],
                'file_hash' => $fileHash,
                'uploaded_by' => $_SESSION['user_id'] ?? null,
                'related_id' => $student_id,
                'related_type' => 'student',
                'description' => 'Student profile photo'
            ]);
            
            // Update student record with photo path
            $this->updateStudentPhotoPath($student_id, $relativePath);
            
            // Log upload
            $this->logUpload($tenant_id, $fileId, 'photo', $uniqueFilename, 'success');
            
            return [
                'success' => true,
                'file_id' => $fileId,
                'filename' => $uniqueFilename,
                'path' => $targetPath,
                'relative_path' => $relativePath,
                'url' => $this->generateFileUrl($tenant_id, 'profile_photos', $uniqueFilename)
            ];
            
        } catch (Exception $e) {
            $this->logUpload($tenant_id, null, 'photo', $file['name'] ?? 'unknown', 'error', $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Upload payment receipt
     * 
     * @param string $tenant_id Tenant identifier
     * @param int $payment_id Payment ID
     * @param array $file File from $_FILES
     * @return array Upload result
     */
    public function uploadPaymentReceipt($tenant_id, $payment_id, $file) {
        try {
            // Validate file
            $validation = $this->validateFile($file, 'receipt');
            if (!$validation['valid']) {
                throw new Exception($validation['error']);
            }
            
            // Get upload path
            $uploadPath = $this->dirManager->getTenantUploadPath($tenant_id, 'payment_receipts');
            if (!$uploadPath) {
                throw new Exception("Invalid tenant or upload path");
            }
            
            // Generate unique filename with payment ID
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $uniqueFilename = 'payment_' . $payment_id . '_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $extension;
            $targetPath = $uploadPath . '/' . $uniqueFilename;
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
                throw new Exception("Failed to move uploaded file");
            }
            
            // Set proper permissions
            chmod($targetPath, 0644);
            
            // Calculate relative path
            $relativePath = 'school_' . $tenant_id . '/uploads/payment_receipts/' . $uniqueFilename;
            
            // Get file hash
            $fileHash = hash_file('sha256', $targetPath);
            
            // Store in database
            $fileId = $this->storeFileMetadata([
                'tenant_id' => $tenant_id,
                'file_category' => 'receipt',
                'original_filename' => basename($file['name']),
                'stored_filename' => $uniqueFilename,
                'file_path' => $relativePath,
                'file_size' => $file['size'],
                'mime_type' => $validation['mime_type'],
                'file_hash' => $fileHash,
                'uploaded_by' => $_SESSION['user_id'] ?? null,
                'related_id' => $payment_id,
                'related_type' => 'payment',
                'description' => 'Payment receipt'
            ]);
            
            // Update payment record with receipt path
            $this->updatePaymentReceiptPath($payment_id, $relativePath);
            
            // Log upload
            $this->logUpload($tenant_id, $fileId, 'receipt', $uniqueFilename, 'success');
            
            return [
                'success' => true,
                'file_id' => $fileId,
                'filename' => $uniqueFilename,
                'path' => $targetPath,
                'relative_path' => $relativePath,
                'url' => $this->generateFileUrl($tenant_id, 'payment_receipts', $uniqueFilename)
            ];
            
        } catch (Exception $e) {
            $this->logUpload($tenant_id, null, 'receipt', $file['name'] ?? 'unknown', 'error', $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Upload generated report
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $report_type Report type (attendance, grades, etc.)
     * @param array $file File from $_FILES
     * @param array $metadata Additional metadata
     * @return array Upload result
     */
    public function uploadReport($tenant_id, $report_type, $file, $metadata = []) {
        try {
            // Validate file
            $validation = $this->validateFile($file, 'report');
            if (!$validation['valid']) {
                throw new Exception($validation['error']);
            }
            
            // Get upload path
            $uploadPath = $this->dirManager->getTenantUploadPath($tenant_id, 'reports');
            if (!$uploadPath) {
                throw new Exception("Invalid tenant or upload path");
            }
            
            // Generate unique filename with report type
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $uniqueFilename = $report_type . '_' . date('Ymd') . '_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $extension;
            $targetPath = $uploadPath . '/' . $uniqueFilename;
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
                throw new Exception("Failed to move uploaded file");
            }
            
            // Set proper permissions
            chmod($targetPath, 0644);
            
            // Calculate relative path
            $relativePath = 'school_' . $tenant_id . '/uploads/reports/' . $uniqueFilename;
            
            // Get file hash
            $fileHash = hash_file('sha256', $targetPath);
            
            // Store in database
            $fileId = $this->storeFileMetadata([
                'tenant_id' => $tenant_id,
                'file_category' => 'report',
                'original_filename' => basename($file['name']),
                'stored_filename' => $uniqueFilename,
                'file_path' => $relativePath,
                'file_size' => $file['size'],
                'mime_type' => $validation['mime_type'],
                'file_hash' => $fileHash,
                'uploaded_by' => $_SESSION['user_id'] ?? null,
                'description' => $metadata['description'] ?? "Report: {$report_type}",
                'tags' => $metadata['tags'] ?? "report,{$report_type}",
                'related_type' => 'report'
            ]);
            
            // Log upload
            $this->logUpload($tenant_id, $fileId, 'report', $uniqueFilename, 'success');
            
            return [
                'success' => true,
                'file_id' => $fileId,
                'filename' => $uniqueFilename,
                'path' => $targetPath,
                'relative_path' => $relativePath,
                'url' => $this->generateFileUrl($tenant_id, 'reports', $uniqueFilename)
            ];
            
        } catch (Exception $e) {
            $this->logUpload($tenant_id, null, 'report', $file['name'] ?? 'unknown', 'error', $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Validate uploaded file
     * 
     * @param array $file File from $_FILES
     * @param string $category File category (document, photo, receipt, report)
     * @return array Validation result
     */
    public function validateFile($file, $category) {
        try {
            // Check if file was uploaded
            $isRealUpload = isset($file['tmp_name']) && is_uploaded_file($file['tmp_name']);
            $isSimulatedUpload = isset($file['tmp_name']) && file_exists($file['tmp_name']) && defined('APP_DEBUG') && APP_DEBUG;
            if (!$isRealUpload && !$isSimulatedUpload) {
                return ['valid' => false, 'error' => 'No file uploaded or invalid upload'];
            }
            
            // Check for upload errors
            if (isset($file['error']) && $file['error'] !== UPLOAD_ERR_OK) {
                return ['valid' => false, 'error' => $this->getUploadErrorMessage($file['error'])];
            }
            
            // Check file size
            if (!isset($this->fileLimits[$category])) {
                return ['valid' => false, 'error' => 'Invalid file category'];
            }
            
            if ($file['size'] > $this->fileLimits[$category]) {
                $maxSize = $this->formatBytes($this->fileLimits[$category]);
                return ['valid' => false, 'error' => "File size exceeds maximum allowed ({$maxSize})"];
            }
            
            // Check file extension
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            if (!in_array($extension, $this->allowedExtensions[$category])) {
                $allowed = implode(', ', $this->allowedExtensions[$category]);
                return ['valid' => false, 'error' => "File type .{$extension} not allowed. Allowed: {$allowed}"];
            }
            
            // Check MIME type using finfo
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);
            
            if (!in_array($mimeType, $this->allowedMimeTypes[$category])) {
                return ['valid' => false, 'error' => "Invalid file type. MIME type not allowed: {$mimeType}"];
            }
            
            // Check magic bytes for additional security
            if (!$this->validateMagicBytes($file['tmp_name'], $mimeType)) {
                return ['valid' => false, 'error' => 'File validation failed: magic bytes mismatch'];
            }
            
            return [
                'valid' => true,
                'mime_type' => $mimeType,
                'extension' => $extension
            ];
            
        } catch (Exception $e) {
            return ['valid' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Validate file magic bytes
     * 
     * @param string $filepath Path to file
     * @param string $mimeType Expected MIME type
     * @return bool Valid or not
     */
    private function validateMagicBytes($filepath, $mimeType) {
        $handle = fopen($filepath, 'rb');
        if (!$handle) {
            return false;
        }
        
        $bytes = fread($handle, 8);
        fclose($handle);
        
        // Check magic bytes for common file types
        $magicBytes = [
            'application/pdf' => ['25504446'],  // %PDF
            'image/jpeg' => ['FFD8FFE0', 'FFD8FFE1', 'FFD8FFE2', 'FFD8FFE3'],  // JPEG markers
            'image/png' => ['89504E47'],  // PNG signature
            'image/svg+xml' => ['3C3F786D', '3C737667'],  // <?xml or <svg
            'image/webp' => ['52494646'],  // RIFF (WebP)
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => ['504B0304'],  // ZIP (DOCX)
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => ['504B0304'],  // ZIP (XLSX)
            'application/msword' => ['D0CF11E0'],  // DOC
            'application/vnd.ms-excel' => ['D0CF11E0']  // XLS
        ];
        
        if (!isset($magicBytes[$mimeType])) {
            return true; // No magic bytes to check for this type
        }
        
        $fileHeader = strtoupper(bin2hex(substr($bytes, 0, 4)));
        
        foreach ($magicBytes[$mimeType] as $magic) {
            if (strpos($fileHeader, $magic) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Sanitize filename
     * 
     * @param string $filename Original filename
     * @return string Sanitized filename
     */
    public function sanitizeFilename($filename) {
        // Strip path components first (security - remove ../, /, \)
        $filename = basename($filename);
        
        // Get extension before processing
        $pathinfo = pathinfo($filename);
        $name = $pathinfo['filename'] ?? '';
        $extension = isset($pathinfo['extension']) ? $pathinfo['extension'] : '';
        
        // Remove anything not alphanumeric, underscore or hyphen
        $name = preg_replace('/[^a-zA-Z0-9_-]/', '_', $name);
        
        // Collapse multiple underscores
        $name = preg_replace('/_+/', '_', $name);
        
        // Trim underscores and hyphens from ends
        $name = trim($name, '_-');
        
        // Limit length
        $name = substr($name, 0, 100);
        
        // Fallback if name becomes empty
        if ($name === '') {
            $name = 'file';
        }
        
        // Add extension back
        return $extension ? $name . '.' . $extension : $name;
    }
    
    /**
     * Generate unique filename
     * 
     * @param string $original_filename Original filename
     * @return string Unique filename
     */
    public function generateUniqueFilename($original_filename) {
        $pathinfo = pathinfo($original_filename);
        $extension = isset($pathinfo['extension']) ? strtolower($pathinfo['extension']) : '';
        
        // Sanitize original name
        $sanitized = $this->sanitizeFilename($pathinfo['filename']);
        
        // Generate unique filename: timestamp_random_sanitized.ext
        $timestamp = time();
        $random = bin2hex(random_bytes(8));
        
        return $timestamp . '_' . $random . '_' . $sanitized . ($extension ? '.' . $extension : '');
    }
    
    /**
     * Delete file safely
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $filepath Relative file path
     * @return array Result
     */
    public function deleteFile($tenant_id, $filepath) {
        try {
            // Get file record from database
            $stmt = $this->db->prepare("
                SELECT * FROM tenant_files 
                WHERE tenant_id = ? AND (file_path = ? OR stored_filename = ?)
                AND is_deleted = 0
            ");
            $stmt->execute([$tenant_id, $filepath, basename($filepath)]);
            $fileRecord = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$fileRecord) {
                throw new Exception("File not found or already deleted");
            }
            
            // Get full path
            $tenantRoot = $this->dirManager->getTenantRootPath($tenant_id);
            $fullPath = $tenantRoot . '/' . ltrim($fileRecord['file_path'], '/');
            
            // Soft delete in database (don't actually delete file)
            $stmt = $this->db->prepare("
                UPDATE tenant_files
                SET is_deleted = 1, deleted_at = NOW(), deleted_by = ?
                WHERE id = ?
            ");
            $stmt->execute([$_SESSION['user_id'] ?? null, $fileRecord['id']]);
            
            // Optionally, actually delete the physical file
            // if (file_exists($fullPath)) {
            //     unlink($fullPath);
            // }
            
            // Log deletion
            $this->logUpload($tenant_id, $fileRecord['id'], $fileRecord['file_category'], 
                           $fileRecord['stored_filename'], 'deleted');
            
            return [
                'success' => true,
                'message' => 'File deleted successfully'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    // ========================================================================
    // PRIVATE HELPER METHODS
    // ========================================================================
    
    /**
     * Store file metadata in database
     */
    private function storeFileMetadata($data) {
        $stmt = $this->db->prepare("
            INSERT INTO tenant_files (
                tenant_id, file_category, original_filename, stored_filename,
                file_path, file_size, mime_type, file_hash, uploaded_by,
                description, tags, related_id, related_type, upload_date
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $data['tenant_id'],
            $data['file_category'],
            $data['original_filename'],
            $data['stored_filename'],
            $data['file_path'],
            $data['file_size'],
            $data['mime_type'],
            $data['file_hash'],
            $data['uploaded_by'],
            $data['description'],
            $data['tags'],
            $data['related_id'],
            $data['related_type']
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Log upload activity
     */
    private function logUpload($tenant_id, $file_id, $category, $filename, $status, $error = null) {
        try {
            $logPath = $this->dirManager->getTenantLogPath($tenant_id, 'access');
            if ($logPath && file_exists($logPath)) {
                $logFile = $logPath . '/' . date('Y-m-d') . '_uploads.log';
                $logEntry = date('Y-m-d H:i:s') . " [{$status}] {$category}: {$filename}";
                
                if ($error) {
                    $logEntry .= " | Error: {$error}";
                }
                
                $logEntry .= PHP_EOL;
                
                @file_put_contents($logFile, $logEntry, FILE_APPEND);
            }
        } catch (Exception $e) {
            // Silent fail - logging is optional
        }
    }
    
    /**
     * Delete existing profile photo
     */
    private function deleteExistingProfilePhoto($tenant_id, $student_id) {
        try {
            $stmt = $this->db->prepare("
                SELECT id, file_path FROM tenant_files
                WHERE tenant_id = ? AND related_id = ? AND related_type = 'student'
                AND file_category = 'photo' AND is_deleted = 0
            ");
            $stmt->execute([$tenant_id, $student_id]);
            $existingPhoto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingPhoto) {
                $this->deleteFile($tenant_id, $existingPhoto['file_path']);
            }
        } catch (Exception $e) {
            // Continue even if deletion fails
        }
    }
    
    /**
     * Optimize/resize image
     */
    private function optimizeImage($filepath, $mimeType) {
        // Placeholder for image optimization
        // Can be implemented with GD or ImageMagick
        // For now, just return true
        return true;
    }
    
    /**
     * Update student photo path
     */
    private function updateStudentPhotoPath($student_id, $photo_path) {
        try {
            $stmt = $this->db->prepare("
                UPDATE students SET profile_photo = ? WHERE id = ?
            ");
            $stmt->execute([$photo_path, $student_id]);
        } catch (PDOException $e) {
            // Table might not have profile_photo column
        }
    }
    
    /**
     * Update payment receipt path
     */
    private function updatePaymentReceiptPath($payment_id, $receipt_path) {
        try {
            $stmt = $this->db->prepare("
                UPDATE payments SET receipt_path = ? WHERE id = ?
            ");
            $stmt->execute([$receipt_path, $payment_id]);
        } catch (PDOException $e) {
            // Table might not have receipt_path column
        }
    }
    
    /**
     * Generate file URL
     */
    private function generateFileUrl($tenant_id, $category, $filename) {
        return '/serve_file.php?tenant=' . urlencode($tenant_id) . 
               '&type=' . urlencode($category) . 
               '&file=' . urlencode($filename);
    }
    
    /**
     * Get upload error message
     */
    private function getUploadErrorMessage($error_code) {
        $errors = [
            UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize directive',
            UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE directive',
            UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
            UPLOAD_ERR_NO_FILE => 'No file was uploaded',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
            UPLOAD_ERR_EXTENSION => 'File upload stopped by extension'
        ];
        
        return $errors[$error_code] ?? 'Unknown upload error';
    }
    
    /**
     * Format bytes to human readable
     */
    private function formatBytes($bytes, $precision = 2) {
        $units = ['B', 'KB', 'MB', 'GB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
}

