<?php
/**
 * Tenant Directory Manager
 * 
 * Handles initialization and management of tenant directory structures
 * Creates and maintains isolated storage for each tenant/school
 * 
 * @package Multi-Tenant System
 * @version 1.0
 */

class TenantDirectoryManager {
    
    /**
     * Base directory for all tenant files
     * @var string
     */
    private $baseDir;
    
    /**
     * Database connection for logging
     * @var PDO
     */
    private $db;
    
    /**
     * Directory structure template
     * @var array
     */
    private $directoryStructure = [
        'uploads/documents',
        'uploads/profile_photos',
        'uploads/payment_receipts',
        'uploads/reports',
        'backups/database',
        'backups/files',
        'logs/access',
        'logs/errors',
        'logs/payments',
        'temp'
    ];
    
    /**
     * Default permissions
     * @var array
     */
    private $permissions = [
        'directories' => 0755,
        'files' => 0644,
        'uploads' => 0777  // Protected by .htaccess
    ];
    
    /**
     * Constructor
     * 
     * @param PDO|null $db Database connection for logging (optional)
     * @param string|null $baseDir Base directory path (default: project_root/tenants)
     */
    public function __construct($db = null, $baseDir = null) {
        $this->db = $db;
        
        // Set base directory
        if ($baseDir === null) {
            $this->baseDir = dirname(__DIR__) . '/tenants';
        } else {
            $this->baseDir = rtrim($baseDir, '/\\');
        }
        
        // Ensure base directory exists
        if (!file_exists($this->baseDir)) {
            mkdir($this->baseDir, $this->permissions['directories'], true);
        }
    }
    
    /**
     * Initialize complete tenant directory structure
     * 
     * @param string $tenant_id Tenant identifier (UUID or alphanumeric)
     * @param string $tenant_name Tenant display name (optional)
     * @param array $custom_settings Custom configuration settings (optional)
     * @return array Result with success status and details
     */
    public function initializeTenant($tenant_id, $tenant_name = null, $custom_settings = []) {
        try {
            // Validate tenant ID
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID format. Use alphanumeric, underscore, or hyphen only.");
            }
            
            $tenantRoot = $this->getTenantRootPath($tenant_id);
            
            // Check if already initialized
            if (file_exists($tenantRoot) && file_exists($tenantRoot . '/config.json')) {
                return [
                    'success' => true,
                    'message' => 'Tenant already initialized',
                    'path' => $tenantRoot,
                    'already_exists' => true
                ];
            }
            
            // Create root directory
            if (!file_exists($tenantRoot)) {
                if (!mkdir($tenantRoot, $this->permissions['directories'], true)) {
                    throw new Exception("Failed to create tenant root directory");
                }
            }
            
            // Create all subdirectories
            $createdDirs = [];
            foreach ($this->directoryStructure as $dir) {
                $fullPath = $tenantRoot . '/' . $dir;
                
                if (!file_exists($fullPath)) {
                    // Use special permissions for upload directories
                    $isUpload = strpos($dir, 'uploads/') === 0;
                    $perms = $isUpload ? $this->permissions['uploads'] : $this->permissions['directories'];
                    
                    if (!mkdir($fullPath, $perms, true)) {
                        throw new Exception("Failed to create directory: {$dir}");
                    }
                    
                    $createdDirs[] = $dir;
                }
            }
            
            // Create security files
            $this->createSecurityFiles($tenantRoot);
            
            // Create tenant configuration
            $config = $this->createTenantConfig($tenant_id, array_merge([
                'tenant_id' => $tenant_id,
                'tenant_name' => $tenant_name ?: $tenant_id,
                'created_at' => date('Y-m-d H:i:s'),
                'initialized_by' => $_SESSION['user_id'] ?? 'system',
                'storage_quota' => 1073741824, // 1GB default
                'storage_used' => 0,
                'file_retention_days' => 365,
                'auto_cleanup_temp' => true,
                'temp_cleanup_days' => 7,
                'backup_retention_days' => 30,
                'log_retention_days' => 90
            ], $custom_settings));
            
            if (!$config['success']) {
                throw new Exception("Failed to create tenant configuration");
            }
            
            // Verify all directories
            $verification = $this->verifyTenantDirectories($tenant_id);
            
            if (!$verification['success']) {
                throw new Exception("Directory verification failed: " . $verification['error']);
            }
            
            // Log initialization
            $this->log($tenant_id, 'tenant_initialized', 
                "Tenant '{$tenant_name}' initialized with " . count($createdDirs) . " directories",
                ['directories' => $createdDirs]
            );
            
            return [
                'success' => true,
                'message' => 'Tenant initialized successfully',
                'tenant_id' => $tenant_id,
                'tenant_name' => $tenant_name,
                'path' => $tenantRoot,
                'directories_created' => count($createdDirs),
                'verification' => $verification
            ];
            
        } catch (Exception $e) {
            $this->log($tenant_id, 'initialization_error', $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get tenant root directory path
     * 
     * @param string $tenant_id Tenant identifier
     * @return string Full path to tenant root directory
     */
    public function getTenantRootPath($tenant_id) {
        if (!$this->validateTenantId($tenant_id)) {
            return false;
        }
        
        return $this->baseDir . '/school_' . $tenant_id;
    }

    /**
     * Remove tenant directory structure
     *
     * @param string $tenant_id Tenant identifier
     * @return array Result with success status and details
     */
    public function deleteTenant($tenant_id) {
        $tenantRoot = $this->getTenantRootPath($tenant_id);

        if (!$tenantRoot || !file_exists($tenantRoot)) {
            return [
                'success' => true,
                'message' => 'Tenant directory not found',
                'path' => $tenantRoot
            ];
        }

        try {
            if ($this->deleteDirectoryRecursive($tenantRoot)) {
                return [
                    'success' => true,
                    'message' => 'Tenant directory removed',
                    'path' => $tenantRoot
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to remove tenant directory',
                'path' => $tenantRoot
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'path' => $tenantRoot
            ];
        }
    }

    /**
     * Recursively delete directory contents
     *
     * @param string $path
     * @return bool
     */
    private function deleteDirectoryRecursive($path) {
        if (!file_exists($path)) {
            return true;
        }

        if (is_file($path) || is_link($path)) {
            return @unlink($path);
        }

        $items = scandir($path);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }

            $fullPath = $path . DIRECTORY_SEPARATOR . $item;
            if (is_dir($fullPath)) {
                if (!$this->deleteDirectoryRecursive($fullPath)) {
                    return false;
                }
            } else {
                if (!@unlink($fullPath)) {
                    return false;
                }
            }
        }

        return @rmdir($path);
    }
    
    /**
     * Get tenant upload directory path for specific category
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $category Upload category (documents, profile_photos, etc.)
     * @return string|false Full path or false on error
     */
    public function getTenantUploadPath($tenant_id, $category) {
        if (!$this->validateTenantId($tenant_id)) {
            return false;
        }
        
        $validCategories = ['documents', 'profile_photos', 'payment_receipts', 'reports'];
        
        if (!in_array($category, $validCategories)) {
            return false;
        }
        
        return $this->getTenantRootPath($tenant_id) . '/uploads/' . $category;
    }
    
    /**
     * Get tenant backup directory path
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $type Backup type (database, files) - optional
     * @return string|false Full path or false on error
     */
    public function getTenantBackupPath($tenant_id, $type = null) {
        if (!$this->validateTenantId($tenant_id)) {
            return false;
        }
        
        $basePath = $this->getTenantRootPath($tenant_id) . '/backups';
        
        if ($type === null) {
            return $basePath;
        }
        
        $validTypes = ['database', 'files'];
        if (!in_array($type, $validTypes)) {
            return false;
        }
        
        return $basePath . '/' . $type;
    }
    
    /**
     * Get tenant log directory path
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $type Log type (access, errors, payments) - optional
     * @return string|false Full path or false on error
     */
    public function getTenantLogPath($tenant_id, $type = null) {
        if (!$this->validateTenantId($tenant_id)) {
            return false;
        }
        
        $basePath = $this->getTenantRootPath($tenant_id) . '/logs';
        
        if ($type === null) {
            return $basePath;
        }
        
        $validTypes = ['access', 'errors', 'payments'];
        if (!in_array($type, $validTypes)) {
            return false;
        }
        
        return $basePath . '/' . $type;
    }
    
    /**
     * Create or update tenant configuration file
     * 
     * @param string $tenant_id Tenant identifier
     * @param array $settings Configuration settings
     * @return array Result with success status
     */
    public function createTenantConfig($tenant_id, $settings = []) {
        try {
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            $configPath = $this->getTenantRootPath($tenant_id) . '/config.json';
            
            // Merge with existing config if it exists
            $existingConfig = [];
            if (file_exists($configPath)) {
                $existingConfig = json_decode(file_get_contents($configPath), true) ?? [];
            }
            
            // Merge settings
            $config = array_merge($existingConfig, $settings);
            
            // Add last updated timestamp
            $config['last_updated'] = date('Y-m-d H:i:s');
            
            // Write config file
            $result = file_put_contents(
                $configPath,
                json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
            );
            
            if ($result === false) {
                throw new Exception("Failed to write configuration file");
            }
            
            // Set proper permissions
            chmod($configPath, $this->permissions['files']);
            
            return [
                'success' => true,
                'message' => 'Configuration created successfully',
                'path' => $configPath,
                'config' => $config
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Verify all tenant directories exist and are writable
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Verification results with details
     */
    public function verifyTenantDirectories($tenant_id) {
        try {
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            $tenantRoot = $this->getTenantRootPath($tenant_id);
            
            if (!file_exists($tenantRoot)) {
                throw new Exception("Tenant root directory does not exist");
            }
            
            $results = [
                'exists' => [],
                'missing' => [],
                'writable' => [],
                'not_writable' => []
            ];
            
            // Check root directory
            if (file_exists($tenantRoot)) {
                $results['exists'][] = '/';
                if (is_writable($tenantRoot)) {
                    $results['writable'][] = '/';
                } else {
                    $results['not_writable'][] = '/';
                }
            }
            
            // Check all subdirectories
            foreach ($this->directoryStructure as $dir) {
                $fullPath = $tenantRoot . '/' . $dir;
                
                if (file_exists($fullPath)) {
                    $results['exists'][] = $dir;
                    
                    if (is_writable($fullPath)) {
                        $results['writable'][] = $dir;
                    } else {
                        $results['not_writable'][] = $dir;
                    }
                } else {
                    $results['missing'][] = $dir;
                }
            }
            
            // Check config file
            $configPath = $tenantRoot . '/config.json';
            if (!file_exists($configPath)) {
                $results['missing'][] = 'config.json';
            }
            
            $allGood = count($results['missing']) === 0 && count($results['not_writable']) === 0;
            
            return [
                'success' => $allGood,
                'tenant_id' => $tenant_id,
                'root_path' => $tenantRoot,
                'directories_ok' => count($results['exists']),
                'directories_missing' => count($results['missing']),
                'directories_writable' => count($results['writable']),
                'directories_not_writable' => count($results['not_writable']),
                'details' => $results,
                'message' => $allGood ? 'All directories verified' : 'Some directories have issues'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Calculate total storage usage for a tenant
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Storage information in bytes
     */
    public function calculateTenantStorageUsage($tenant_id) {
        try {
            if (!$this->validateTenantId($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            $tenantRoot = $this->getTenantRootPath($tenant_id);
            
            if (!file_exists($tenantRoot)) {
                throw new Exception("Tenant directory does not exist");
            }
            
            $usage = [
                'total' => 0,
                'uploads' => 0,
                'backups' => 0,
                'logs' => 0,
                'temp' => 0,
                'other' => 0
            ];
            
            // Calculate size by category
            foreach ($this->directoryStructure as $dir) {
                $fullPath = $tenantRoot . '/' . $dir;
                
                if (file_exists($fullPath)) {
                    $size = $this->getDirectorySize($fullPath);
                    
                    // Categorize
                    if (strpos($dir, 'uploads/') === 0) {
                        $usage['uploads'] += $size;
                    } elseif (strpos($dir, 'backups/') === 0) {
                        $usage['backups'] += $size;
                    } elseif (strpos($dir, 'logs/') === 0) {
                        $usage['logs'] += $size;
                    } elseif ($dir === 'temp') {
                        $usage['temp'] += $size;
                    } else {
                        $usage['other'] += $size;
                    }
                    
                    $usage['total'] += $size;
                }
            }
            
            // Add config file size
            $configPath = $tenantRoot . '/config.json';
            if (file_exists($configPath)) {
                $configSize = filesize($configPath);
                $usage['other'] += $configSize;
                $usage['total'] += $configSize;
            }
            
            // Get quota from config
            $quota = $this->getTenantQuota($tenant_id);
            
            return [
                'success' => true,
                'tenant_id' => $tenant_id,
                'usage' => $usage,
                'total_bytes' => $usage['total'],
                'total_formatted' => $this->formatBytes($usage['total']),
                'quota_bytes' => $quota,
                'quota_formatted' => $this->formatBytes($quota),
                'percentage' => $quota > 0 ? round(($usage['total'] / $quota) * 100, 2) : 0,
                'available_bytes' => max(0, $quota - $usage['total']),
                'available_formatted' => $this->formatBytes(max(0, $quota - $usage['total']))
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get tenant configuration
     * 
     * @param string $tenant_id Tenant identifier
     * @return array|false Configuration array or false on error
     */
    public function getTenantConfig($tenant_id) {
        if (!$this->validateTenantId($tenant_id)) {
            return false;
        }
        
        $configPath = $this->getTenantRootPath($tenant_id) . '/config.json';
        
        if (!file_exists($configPath)) {
            return false;
        }
        
        return json_decode(file_get_contents($configPath), true);
    }
    
    /**
     * Get tenant storage quota
     * 
     * @param string $tenant_id Tenant identifier
     * @return int Quota in bytes (default 1GB)
     */
    public function getTenantQuota($tenant_id) {
        $config = $this->getTenantConfig($tenant_id);
        return $config['storage_quota'] ?? 1073741824; // Default 1GB
    }
    
    /**
     * Update tenant storage quota
     * 
     * @param string $tenant_id Tenant identifier
     * @param int $quota_bytes New quota in bytes
     * @return array Result with success status
     */
    public function updateTenantQuota($tenant_id, $quota_bytes) {
        $config = $this->getTenantConfig($tenant_id);
        
        if ($config === false) {
            return [
                'success' => false,
                'error' => 'Tenant configuration not found'
            ];
        }
        
        $config['storage_quota'] = (int)$quota_bytes;
        
        return $this->createTenantConfig($tenant_id, $config);
    }
    
    // ========================================================================
    // PRIVATE HELPER METHODS
    // ========================================================================
    
    /**
     * Create security files (.htaccess, index.php) in tenant directories
     * 
     * @param string $tenantRoot Tenant root directory path
     * @return bool Success status
     */
    private function createSecurityFiles($tenantRoot) {
        // Create root .htaccess
        $rootHtaccess = "# Tenant Directory Protection\n";
        $rootHtaccess .= "# Generated: " . date('Y-m-d H:i:s') . "\n\n";
        $rootHtaccess .= "# Deny direct access to all files\n";
        $rootHtaccess .= "Order Deny,Allow\n";
        $rootHtaccess .= "Deny from all\n\n";
        $rootHtaccess .= "# Allow access only through application\n";
        $rootHtaccess .= "<Files ~ \"\\.(jpg|jpeg|png|gif|webp|pdf)$\">\n";
        $rootHtaccess .= "    # Files should be served through serve_file.php\n";
        $rootHtaccess .= "    Allow from all\n";
        $rootHtaccess .= "</Files>\n";
        
        file_put_contents($tenantRoot . '/.htaccess', $rootHtaccess);
        chmod($tenantRoot . '/.htaccess', $this->permissions['files']);
        
        // Create upload directory .htaccess (more permissive for images)
        $uploadHtaccess = "# Upload Directory Protection\n";
        $uploadHtaccess .= "Options -Indexes\n";
        $uploadHtaccess .= "DirectoryIndex disabled\n\n";
        $uploadHtaccess .= "# Prevent execution of PHP files\n";
        $uploadHtaccess .= "php_flag engine off\n\n";
        $uploadHtaccess .= "# Allow images and documents\n";
        $uploadHtaccess .= "<FilesMatch \"\\.(jpg|jpeg|png|gif|webp|pdf|doc|docx|xls|xlsx)$\">\n";
        $uploadHtaccess .= "    Order Allow,Deny\n";
        $uploadHtaccess .= "    Allow from all\n";
        $uploadHtaccess .= "</FilesMatch>\n";
        
        foreach (['documents', 'profile_photos', 'payment_receipts', 'reports'] as $category) {
            $uploadDir = $tenantRoot . '/uploads/' . $category;
            if (file_exists($uploadDir)) {
                file_put_contents($uploadDir . '/.htaccess', $uploadHtaccess);
                chmod($uploadDir . '/.htaccess', $this->permissions['files']);
            }
        }
        
        // Create index.php files to prevent directory listing
        $indexContent = "<?php\n";
        $indexContent .= "// Directory access forbidden\n";
        $indexContent .= "http_response_code(403);\n";
        $indexContent .= "die('Access denied');\n";
        
        // Add to all directories
        foreach ($this->directoryStructure as $dir) {
            $fullPath = $tenantRoot . '/' . $dir;
            if (file_exists($fullPath)) {
                file_put_contents($fullPath . '/index.php', $indexContent);
                chmod($fullPath . '/index.php', $this->permissions['files']);
            }
        }
        
        return true;
    }
    
    /**
     * Validate tenant ID format
     * 
     * @param string $tenant_id Tenant identifier
     * @return bool Valid or not
     */
    private function validateTenantId($tenant_id) {
        // Allow alphanumeric, underscore, hyphen
        // Length: 3-50 characters
        return preg_match('/^[a-zA-Z0-9_-]{3,50}$/', $tenant_id);
    }
    
    /**
     * Get directory size recursively
     * 
     * @param string $path Directory path
     * @return int Size in bytes
     */
    private function getDirectorySize($path) {
        $size = 0;
        
        if (!is_dir($path)) {
            return is_file($path) ? filesize($path) : 0;
        }
        
        $items = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($items as $item) {
            if ($item->isFile()) {
                $size += $item->getSize();
            }
        }
        
        return $size;
    }
    
    /**
     * Format bytes to human readable size
     * 
     * @param int $bytes Size in bytes
     * @param int $precision Decimal precision
     * @return string Formatted size
     */
    private function formatBytes($bytes, $precision = 2) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
    
    /**
     * Log operation to file and database
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $operation Operation type
     * @param string $message Log message
     * @param array $metadata Additional metadata
     */
    private function log($tenant_id, $operation, $message, $metadata = []) {
        // Log to file
        $logDir = $this->getTenantLogPath($tenant_id, 'access');
        if ($logDir && file_exists($logDir)) {
            $logFile = $logDir . '/' . date('Y-m-d') . '.log';
            $logEntry = date('Y-m-d H:i:s') . " [{$operation}] {$message}";
            
            if (!empty($metadata)) {
                $logEntry .= ' | ' . json_encode($metadata);
            }
            
            $logEntry .= PHP_EOL;
            
            @file_put_contents($logFile, $logEntry, FILE_APPEND);
        }
        
        // Log to database if available
        if ($this->db) {
            try {
                $stmt = $this->db->prepare("
                    INSERT INTO file_operations_log 
                    (tenant_id, operation, message, metadata, created_at)
                    VALUES (?, ?, ?, ?, NOW())
                ");
                $stmt->execute([
                    $tenant_id,
                    $operation,
                    $message,
                    !empty($metadata) ? json_encode($metadata) : null
                ]);
            } catch (PDOException $e) {
                // Silent fail - logging is optional
                error_log("TenantDirectoryManager logging error: " . $e->getMessage());
            }
        }
    }
}

