<?php
/**
 * Student Account Creator
 * Creates actual student accounts from approved applications
 * Handles username generation, password creation, and Moodle enrollment
 */

class StudentAccountCreator {
    
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Create student account from application
     * 
     * @param int $application_id Application ID
     * @param bool $bypass_payment Whether to bypass payment check (for manual approval)
     * @param int $approved_by Admin ID who approved (for manual approval)
     */
    public function createFromApplication($application_id, $bypass_payment = false, $approved_by = null) {
        try {
            // Get application details
            $stmt = $this->pdo->prepare("
                SELECT * FROM student_applications WHERE id = ?
            ");
            $stmt->execute([$application_id]);
            $app = $stmt->fetch();
            
            if (!$app) {
                return ['success' => false, 'error' => 'Application not found'];
            }
            
            // Check if already created
            if ($app['student_id']) {
                return ['success' => false, 'error' => 'Student account already created'];
            }
            
            // Verify payment and email (unless bypassed by admin)
            if (!$bypass_payment && $app['payment_status'] !== 'paid') {
                return ['success' => false, 'error' => 'Payment not completed'];
            }
            
            if (!$app['email_verified']) {
                return ['success' => false, 'error' => 'Email not verified'];
            }
            
            $this->pdo->beginTransaction();
            
            // Generate username
            $username = $this->generateUsername();
            
            // Generate password based on last name (Moodle-compatible)
            $password_plain = $this->generatePassword($app['last_name']);
            $password_hash = password_hash($password_plain, PASSWORD_DEFAULT);
            
            // Combine first and last name for full_name
            $full_name = $app['first_name'] . ' ' . $app['last_name'];
            
            // Create student record
            $stmt = $this->pdo->prepare("
                INSERT INTO students 
                (username, password_hash, full_name, email, email_verified, phone, grade_id, academy_reference, created_at)
                VALUES (?, ?, ?, ?, 1, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                $username,
                $password_hash,
                $full_name,
                $app['email'],
                $app['phone'],
                $app['grade_id'],
                $app['academy_reference']
            ]);
            
            $student_id = $this->pdo->lastInsertId();
            
            // Enroll in subjects
            $stmt = $this->pdo->prepare("
                SELECT subject_id FROM student_application_subjects WHERE application_id = ?
            ");
            $stmt->execute([$application_id]);
            $subject_ids = array_column($stmt->fetchAll(), 'subject_id');
            
            $stmt = $this->pdo->prepare("
                INSERT INTO student_subject (student_id, subject_id)
                VALUES (?, ?)
            ");
            
            foreach ($subject_ids as $subject_id) {
                $stmt->execute([$student_id, $subject_id]);
            }
            
            // Create parent account if doesn't exist
            if ($app['parent_email']) {
                $parent_id = $this->createOrLinkParent($app, $student_id);
            }
            
            // Enroll in Moodle (if enabled)
            if (class_exists('MoodleEnrollmentHandler')) {
                try {
                    require_once 'moodle_enrollment_handler.php';
                    $moodleHandler = new MoodleEnrollmentHandler($app['academy_reference'], $this->pdo);
                    $moodleHandler->enrollStudentInMultipleSubjects($student_id, $subject_ids);
                } catch (Exception $e) {
                    // Log but don't fail - Moodle enrollment is optional
                    error_log("Moodle enrollment failed for student {$student_id}: " . $e->getMessage());
                }
            }
            
            // Send credentials via email
            $this->sendCredentials($app['email'], $username, $password_plain, $app['full_name'], $app['academy_reference']);
            
            // Send SMS (if phone available)
            if ($app['phone']) {
                $this->sendSMSCredentials($app['phone'], $username, $password_plain);
            }
            
            // Update application record with created student and approval info
            $stmt = $this->pdo->prepare("
                UPDATE student_applications 
                SET student_id = ?,
                    username = ?,
                    password_plain = ?,
                    application_status = 'completed',
                    approved_by = ?,
                    approved_at = NOW(),
                    completed_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$student_id, $username, $password_plain, $approved_by, $application_id]);
            
            $this->pdo->commit();
            
            return [
                'success' => true,
                'student_id' => $student_id,
                'username' => $username,
                'password' => $password_plain
            ];
            
        } catch (Exception $e) {
            if ($this->pdo->inTransaction()) {
                $this->pdo->rollBack();
            }
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Generate unique username
     */
    private function generateUsername() {
        $prefix = date('Ymd');
        
        // Get last username for today
        $stmt = $this->pdo->prepare("
            SELECT username FROM students 
            WHERE username LIKE ? 
            ORDER BY username DESC LIMIT 1
        ");
        $stmt->execute([$prefix . '%']);
        $last_username = $stmt->fetchColumn();
        
        $seq = 1;
        if ($last_username && strpos($last_username, $prefix) === 0) {
            $tail = substr($last_username, 8);
            $seq = max(1, (int)$tail + 1);
        }
        
        $username = $prefix . str_pad((string)$seq, 2, '0', STR_PAD_LEFT);
        
        // Ensure unique
        $stmt = $this->pdo->prepare("SELECT 1 FROM students WHERE username = ?");
        while (true) {
            $stmt->execute([$username]);
            if (!$stmt->fetch()) break;
            $seq++;
            $username = $prefix . str_pad((string)$seq, 2, '0', STR_PAD_LEFT);
        }
        
        return $username;
    }
    
    /**
     * Generate secure password based on last name (Moodle-compatible)
     * 
     * Format: LastName@#### (e.g., Dlamini@2024, Smith@7531)
     * - Meets Moodle requirements: 8+ chars, uppercase, lowercase, numbers, special char
     * - Easy to remember for students
     * - Unique per student (random numbers)
     * 
     * @param string $last_name Student's last name
     * @return string Generated password
     */
    private function generatePassword($last_name) {
        // Clean and capitalize last name
        $last_name = preg_replace('/[^a-zA-Z]/', '', $last_name); // Remove special chars
        $last_name = ucfirst(strtolower($last_name)); // Capitalize first letter
        
        // If last name is too short, pad it
        if (strlen($last_name) < 4) {
            $last_name = str_pad($last_name, 4, 'X');
        }
        
        // Generate 4 random digits
        $numbers = '';
        for ($i = 0; $i < 4; $i++) {
            $numbers .= random_int(0, 9);
        }
        
        // Format: LastName@####
        // @ symbol satisfies special character requirement
        // Mixed case + numbers + special char = Moodle compatible
        $password = $last_name . '@' . $numbers;
        
        return $password;
    }
    
    /**
     * Create or link parent account
     */
    private function createOrLinkParent($application, $student_id) {
        // Check if parent exists
        $stmt = $this->pdo->prepare("
            SELECT id FROM parents 
            WHERE email = ? AND academy_reference = ?
        ");
        $stmt->execute([$application['parent_email'], $application['academy_reference']]);
        $parent = $stmt->fetch();
        
        if ($parent) {
            $parent_id = $parent['id'];
        } else {
            // Create new parent account
            $parent_password = bin2hex(random_bytes(8));
            $stmt = $this->pdo->prepare("
                INSERT INTO parents 
                (full_name, email, phone, password_hash, academy_reference, created_at)
                VALUES (?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $application['parent_name'],
                $application['parent_email'],
                $application['parent_phone'],
                password_hash($parent_password, PASSWORD_DEFAULT),
                $application['academy_reference']
            ]);
            $parent_id = $this->pdo->lastInsertId();
            
            // Send parent credentials
            $this->sendParentCredentials($application['parent_email'], $parent_password, $application['parent_name']);
        }
        
        // Link parent to student
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO parent_students (parent_id, student_id, relationship, is_primary)
            VALUES (?, ?, ?, 1)
        ");
        $stmt->execute([$parent_id, $student_id, $application['parent_relationship']]);
        
        return $parent_id;
    }
    
    /**
     * Send credentials to student via email
     */
    private function sendCredentials($email, $username, $password, $full_name, $academy_reference) {
        // Get school login URL
        $login_url = "https://{$_SERVER['HTTP_HOST']}/Multi-Tanent/" . strtolower($academy_reference) . "/login";
        
        // Get Moodle URL from school settings
        $stmt = $this->pdo->prepare("
            SELECT setting_value FROM tenant_settings 
            WHERE academy_reference = ? AND setting_key = 'moodle_url'
        ");
        $stmt->execute([$academy_reference]);
        $moodle_url = $stmt->fetchColumn();
        
        $subject = "✅ Your Student Account is Ready - Login Credentials Inside";
        $message = "
            <!DOCTYPE html>
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                    .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
                    .content { background: #ffffff; padding: 30px; border: 1px solid #e0e0e0; }
                    .credentials { background: #f8f9fa; padding: 25px; border-radius: 10px; margin: 20px 0; border-left: 4px solid #0d6efd; }
                    .credential-row { margin: 15px 0; }
                    .credential-label { font-weight: 600; color: #495057; margin-bottom: 5px; }
                    .credential-value { font-size: 20px; background: #fff; padding: 10px 15px; border-radius: 5px; font-family: 'Courier New', monospace; letter-spacing: 1px; display: inline-block; border: 2px solid #0d6efd; }
                    .button { background: #0d6efd; color: white; padding: 15px 30px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 20px 0; font-weight: 600; }
                    .button:hover { background: #0b5ed7; }
                    .important { background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin: 20px 0; border-radius: 5px; }
                    .footer { background: #f8f9fa; padding: 20px; text-align: center; color: #6c757d; font-size: 14px; border-radius: 0 0 10px 10px; }
                    .moodle-section { background: #e7f3ff; border-left: 4px solid #17a2b8; padding: 15px; margin: 20px 0; border-radius: 5px; }
                    .emoji { font-size: 24px; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>
                        <h1 style='margin: 0;'><span class='emoji'>🎓</span> Welcome to Our School!</h1>
                        <p style='margin: 10px 0 0 0;'>Your account is ready to use</p>
                    </div>
                    
                    <div class='content'>
                        <p>Dear <strong>{$full_name}</strong>,</p>
                        <p>Congratulations! Your student account has been <strong>approved and activated</strong>. You can now access the student portal and all learning resources.</p>
                        
                        <div class='credentials'>
                            <h3 style='margin-top: 0; color: #0d6efd;'>🔐 Your Login Credentials</h3>
                            <div class='credential-row'>
                                <div class='credential-label'>Username:</div>
                                <div class='credential-value'>{$username}</div>
                            </div>
                            <div class='credential-row'>
                                <div class='credential-label'>Password:</div>
                                <div class='credential-value'>{$password}</div>
                            </div>
                        </div>
                        
                        <div style='text-align: center;'>
                            <a href='{$login_url}' class='button'>
                                📱 Login to Student Portal
                            </a>
                        </div>
                        
                        <div class='moodle-section'>
                            <h4 style='margin-top: 0; color: #17a2b8;'>📚 Moodle Learning Platform</h4>
                            <p><strong>Great news!</strong> You can use the <strong>same credentials</strong> to access your Moodle courses:</p>
                            <ul style='margin: 10px 0;'>
                                <li>Username: <code>{$username}</code></li>
                                <li>Password: <code>{$password}</code></li>
                            </ul>
                            <p style='margin: 5px 0;'><small>You'll be automatically enrolled in your registered subjects on Moodle.</small></p>
                            " . ($moodle_url ? "
                            <div style='text-align: center; margin-top: 20px;'>
                                <a href='{$moodle_url}' class='button' style='background: #17a2b8; color: white; padding: 12px 25px; text-decoration: none; border-radius: 5px; display: inline-block; font-weight: 600;'>
                                    🎓 Access Moodle Now
                                </a>
                            </div>
                            " : "<p style='margin-top: 10px;'><small><em>Moodle URL will be provided by your school.</em></small></p>") . "
                        </div>
                        
                        <div class='important'>
                            <strong>⚠️ Important Security Note:</strong>
                            <ul style='margin: 10px 0;'>
                                <li>Keep your credentials safe and confidential</li>
                                <li>Do not share your password with anyone</li>
                                <li>You can change your password after first login</li>
                            </ul>
                        </div>
                        
                        <h4>✅ What's Next?</h4>
                        <ol>
                            <li><strong>Login</strong> to the student portal using the credentials above</li>
                            <li><strong>Update</strong> your profile and change your password if needed</li>
                            <li><strong>Access</strong> your subjects and course materials</li>
                            <li><strong>Join</strong> your Moodle courses using the same credentials</li>
                        </ol>
                        
                        <p style='margin-top: 30px;'>If you have any questions or need assistance, please contact the school administration.</p>
                        
                        <p style='margin-top: 20px;'>Best regards,<br><strong>School Administration</strong></p>
                    </div>
                    
                    <div class='footer'>
                        <p style='margin: 0;'>This is an automated message. Please do not reply to this email.</p>
                        <p style='margin: 5px 0 0 0;'>© " . date('Y') . " Multi-Tenant School System</p>
                    </div>
                </div>
            </body>
            </html>
        ";
        
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8\r\n";
        $headers .= "From: School Administration <noreply@melanegroup.com>\r\n";
        
        return @mail($email, $subject, $message, $headers);
    }
    
    /**
     * Send credentials via SMS
     */
    private function sendSMSCredentials($phone, $username, $password) {
        // Check if SMS is enabled
        if (!defined('SMS_ENABLED') || !SMS_ENABLED) {
            error_log("SMS disabled - skipping credentials SMS to {$phone}");
            return false;
        }
        
        try {
            require_once __DIR__ . '/sms_functions.php';
            
            // Get student and school info
            $stmt = $this->pdo->prepare("
                SELECT s.full_name, s.academy_reference, ar.academy_name
                FROM students s
                INNER JOIN academy_references ar ON s.academy_reference = ar.reference_code
                WHERE s.username = ?
            ");
            $stmt->execute([$username]);
            $student = $stmt->fetch();
            
            if ($student) {
                $login_url = "https://" . $_SERVER['HTTP_HOST'] . "/Multi-Tanent/" . strtolower($student['academy_reference']) . "/login";
                
                $result = sendCredentialsSMS(
                    $phone, 
                    $student['full_name'], 
                    $username, 
                    $password, 
                    $login_url,
                    $student['academy_name']
                );
                
                if ($result['success']) {
                    error_log("Credentials SMS sent successfully to {$phone}");
                    return true;
                } else {
                    error_log("Failed to send credentials SMS to {$phone}: " . ($result['error'] ?? 'Unknown error'));
                    return false;
                }
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("SMS credentials exception: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send parent credentials
     */
    private function sendParentCredentials($email, $password, $name) {
        $subject = "Parent Portal Account Created";
        $message = "
            <h2>Parent Portal Access</h2>
            <p>Dear {$name},</p>
            <p>A parent account has been created for you.</p>
            <div style='background:#f8f9fa;padding:20px;border-radius:10px;margin:20px 0;'>
                <h4>Your Login Credentials:</h4>
                <p><strong>Email:</strong> {$email}</p>
                <p><strong>Password:</strong> <code style='font-size:18px;background:#fff;padding:5px 10px;'>{$password}</code></p>
            </div>
            <p><a href='https://{$_SERVER['HTTP_HOST']}/Multi-Tanent/parent/login.php' 
                  style='background:#0d6efd;color:white;padding:10px 20px;text-decoration:none;border-radius:5px;display:inline-block;'>
                Login to Parent Portal
            </a></p>
            <p>From the parent portal, you can view your child's progress and make payments.</p>
        ";
        
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8\r\n";
        $headers .= "From: noreply@melanegroup.com\r\n";
        
        return @mail($email, $subject, $message, $headers);
    }
}

