<?php
/**
 * Security Headers Initialization
 * Apply security headers to all pages
 * 
 * Usage: Include this file at the top of every page (after includes/functions.php)
 * require_once 'includes/security_init.php';
 */

// Prevent clickjacking attacks
header("X-Frame-Options: SAMEORIGIN");

// Prevent MIME type sniffing
header("X-Content-Type-Options: nosniff");

// Enable XSS Protection (for older browsers)
header("X-XSS-Protection: 1; mode=block");

// Referrer Policy - don't send full URL to external sites
header("Referrer-Policy: strict-origin-when-cross-origin");

// Permissions Policy - restrict access to device features
header("Permissions-Policy: geolocation=(), microphone=(), camera=()");

// Content Security Policy (CSP)
$csp = "default-src 'self' https://cdn.jsdelivr.net https://cdn.jsdelivr.com; ";
$csp .= "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.jsdelivr.net https://cdn.jsdelivr.com; ";
$csp .= "style-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://cdn.jsdelivr.com; ";
$csp .= "img-src 'self' data: https: blob:; ";
$csp .= "font-src 'self' https://cdn.jsdelivr.net data:; ";
$csp .= "connect-src 'self'; ";
$csp .= "frame-ancestors 'self'; ";
$csp .= "base-uri 'self'; ";
$csp .= "form-action 'self';";

header("Content-Security-Policy: $csp");

// Strict Transport Security (HSTS) - ONLY enable when you have HTTPS!
// Uncomment the line below when SSL certificate is installed
// header("Strict-Transport-Security: max-age=31536000; includeSubDomains; preload");

// Additional security measures

// Ensure session is using secure settings
if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_only_cookies', 1);
    ini_set('session.cookie_secure', 0); // Set to 1 when HTTPS is enabled
    ini_set('session.cookie_samesite', 'Lax');
}

// Disable display of PHP errors in production (they'll still be logged)
if (!defined('DEVELOPMENT_MODE') || DEVELOPMENT_MODE === false) {
    ini_set('display_errors', 0);
    ini_set('display_startup_errors', 0);
    error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
}

// Set secure cookies flag globally
ini_set('session.cookie_httponly', true);

// Regenerate session ID periodically to prevent session fixation
if (isset($_SESSION) && !isset($_SESSION['created_at'])) {
    $_SESSION['created_at'] = time();
} elseif (isset($_SESSION['created_at']) && time() - $_SESSION['created_at'] > 1800) {
    // Regenerate session ID every 30 minutes
    session_regenerate_id(true);
    $_SESSION['created_at'] = time();
}

/**
 * Generate CSRF token if not exists
 */
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

/**
 * Helper function to get CSRF token
 */
if (!function_exists('getCsrfToken')) {
    function getCsrfToken() {
        return $_SESSION['csrf_token'] ?? '';
    }
}

/**
 * Helper function to validate CSRF token
 */
if (!function_exists('validateCsrfToken')) {
    function validateCsrfToken($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
}

/**
 * Generate HTML for CSRF token input field
 */
if (!function_exists('csrfField')) {
    function csrfField() {
        $token = getCsrfToken();
        return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars($token) . '">';
    }
}

?>

