<?php
/**
 * Helper utilities for creating school (tenant) accounts.
 *
 * This file centralizes the logic that provisions a new school:
 *  - Creates the admin account
 *  - Registers the academy reference
 *  - Seeds default tenant settings
 *  - Initializes wallet + tenant storage directories
 *  - Returns useful metadata (login credentials, URLs)
 *
 * It is used by both the internal super-admin panel and the public
 * self-service school registration form.
 */

require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/tenant_directory_manager.php';
require_once __DIR__ . '/school_url_generator.php';
require_once __DIR__ . '/email_functions.php';

/**
 * Fetch the column definitions for a table.
 *
 * @param PDO    $pdo
 * @param string $table
 * @return array<string,array<string,mixed>>
 */
function getTableColumnMap(PDO $pdo, string $table): array
{
    $stmt = $pdo->prepare("SHOW COLUMNS FROM `{$table}`");
    $stmt->execute();

    $columns = [];
    foreach ($stmt as $row) {
        $columns[$row['Field']] = $row;
    }

    return $columns;
}

/**
 * Create a new school tenant with its admin account.
 *
 * Expected $input keys:
 *  - school_name (string, required)
 *  - reference_code (string, required, uppercase A-Z0-9)
 *  - admin_email (string, required)
 *  - admin_password (string, required, >= 8 chars)
 *  - description (string, optional)
 *  - contact_phone (string, optional)
 *  - contact_name (string, optional)
 *  - address (string, optional)
 *
 * Supported $options:
 *  - created_by (int|null)   : admin ID that initiated creation
 *  - send_welcome_email (bool): default true
 *
 * @param array<string,mixed> $input
 * @param array<string,mixed> $options
 * @return array<string,mixed>
 * @throws Exception on validation or persistence errors
 */
function createSchoolAccount(array $input, array $options = []): array
{
    $schoolName    = trim((string)($input['school_name'] ?? ''));
    $referenceCode = strtoupper(trim((string)($input['reference_code'] ?? '')));
    $adminEmail    = trim((string)($input['admin_email'] ?? ''));
    $adminPassword = (string)($input['admin_password'] ?? '');
    $description   = trim((string)($input['description'] ?? ''));
    $contactPhone  = trim((string)($input['contact_phone'] ?? ''));
    $contactName   = trim((string)($input['contact_name'] ?? ''));
    $address       = trim((string)($input['address'] ?? ''));

    if ($schoolName === '') {
        throw new InvalidArgumentException('School name is required.');
    }

    if ($referenceCode === '' || !preg_match('/^[A-Z0-9]{2,10}$/', $referenceCode)) {
        throw new InvalidArgumentException('Reference code must be 2-10 uppercase letters and numbers.');
    }

    if ($adminEmail === '' || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
        throw new InvalidArgumentException('A valid admin email address is required.');
    }

    if (strlen($adminPassword) < 8) {
        throw new InvalidArgumentException('Admin password must be at least 8 characters long.');
    }

    $pdo = getDB();
    $pdo->beginTransaction();

    try {
        // Ensure reference code is unique
        $stmt = $pdo->prepare('SELECT id FROM academy_references WHERE reference_code = ? LIMIT 1');
        $stmt->execute([$referenceCode]);
        if ($stmt->fetchColumn()) {
            throw new Exception('Reference code already exists. Please choose a different code.');
        }

        // Inspect current schema for dynamic inserts
        $adminColumns   = getTableColumnMap($pdo, 'admins');
        $academyColumns = getTableColumnMap($pdo, 'academy_references');

        // Ensure admin email is unique when the column exists
        if (isset($adminColumns['email'])) {
            $stmt = $pdo->prepare('SELECT id FROM admins WHERE email = ? LIMIT 1');
            $stmt->execute([$adminEmail]);
            if ($stmt->fetchColumn()) {
                throw new Exception('An admin account with this email already exists.');
            }
        }

        // Build a unique admin username (e.g., kine_admin, kine_admin2)
        $baseUsername = strtolower($referenceCode) . '_admin';
        $adminUsername = $baseUsername;
        $usernameCheck = $pdo->prepare('SELECT 1 FROM admins WHERE username = ? LIMIT 1');
        $suffix = 1;
        while (true) {
            $usernameCheck->execute([$adminUsername]);
            if (!$usernameCheck->fetchColumn()) {
                break;
            }
            $suffix++;
            $adminUsername = $baseUsername . $suffix;
        }

        $passwordHash = password_hash($adminPassword, PASSWORD_DEFAULT);

        // Prepare dynamic insert for admins table
        $adminData = [
            'username'       => $adminUsername,
            'password_hash'  => $passwordHash,
        ];

        if (isset($adminColumns['email'])) {
            $adminData['email'] = $adminEmail;
        }

        if (isset($adminColumns['academy_reference'])) {
            $adminData['academy_reference'] = $referenceCode;
        }

        if (isset($adminColumns['is_active'])) {
            $adminData['is_active'] = 1;
        }

        if (isset($adminColumns['account_verified'])) {
            $adminData['account_verified'] = 0;
        }

        if (isset($adminColumns['full_name']) && $contactName !== '') {
            $adminData['full_name'] = $contactName;
        }

        $adminColumnsSql = implode(', ', array_map(fn($col) => "`{$col}`", array_keys($adminData)));
        $adminPlaceholders = implode(', ', array_fill(0, count($adminData), '?'));
        $adminStmt = $pdo->prepare("INSERT INTO admins ({$adminColumnsSql}) VALUES ({$adminPlaceholders})");
        $adminStmt->execute(array_values($adminData));
        $adminId = (int)$pdo->lastInsertId();

        // Prepare academy_references record
        $academyData = [
            'reference_code' => $referenceCode,
            'academy_name'   => $schoolName,
        ];

        if (isset($academyColumns['admin_id'])) {
            $academyData['admin_id'] = $adminId;
        }

        if (isset($academyColumns['contact_email'])) {
            $academyData['contact_email'] = $adminEmail;
        }

        if (isset($academyColumns['contact_phone']) && $contactPhone !== '') {
            $academyData['contact_phone'] = $contactPhone;
        }

        if (isset($academyColumns['description']) && $description !== '') {
            $academyData['description'] = $description;
        }

        if (isset($academyColumns['address']) && $address !== '') {
            $academyData['address'] = $address;
        }

        if (isset($academyColumns['is_active'])) {
            $academyData['is_active'] = 1;
        }

        if (isset($academyColumns['subscription_status'])) {
            $academyData['subscription_status'] = 'trial';
        }

        if (isset($academyColumns['onboarding_date'])) {
            $academyData['onboarding_date'] = date('Y-m-d');
        }

        $academyColumnsSql = implode(', ', array_map(fn($col) => "`{$col}`", array_keys($academyData)));
        $academyPlaceholders = implode(', ', array_fill(0, count($academyData), '?'));
        $academyStmt = $pdo->prepare("INSERT INTO academy_references ({$academyColumnsSql}) VALUES ({$academyPlaceholders})");
        $academyStmt->execute(array_values($academyData));
        $academyId = (int)$pdo->lastInsertId();

        // Seed default tenant settings
        $defaultSettings = [
            'school_name'        => $schoolName,
            'school_email'       => $adminEmail,
            'contact_name'       => $contactName,
            'contact_phone'      => $contactPhone,
            'primary_color'      => '#4F46E5',
            'secondary_color'    => '#10B981',
            'academic_year'      => date('Y') . '/' . (date('Y') + 1),
            'email_from_name'    => $schoolName,
            'max_students'       => 500,
            'filtering_mode'     => 'auto',
            'subject_fee'        => 350,
            'yearly_discount'    => 10,
            'currency_symbol'    => 'E',
            'self_registration'  => 1,
        ];

        foreach ($defaultSettings as $key => $value) {
            if ($value === '' || $value === null) {
                continue;
            }

            $stmt = $pdo->prepare("
                INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
                VALUES (?, ?, ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute([$referenceCode, $key, (string)$value]);
        }

        // Initialize admin wallet if table exists
        try {
            $pdo->prepare("
                INSERT INTO admin_wallet (admin_id) VALUES (?)
                ON DUPLICATE KEY UPDATE admin_id = VALUES(admin_id)
            ")->execute([$adminId]);
        } catch (Throwable $walletEx) {
            // Wallet table might not exist in some deployments
            error_log('Admin wallet initialization skipped: ' . $walletEx->getMessage());
        }

        $pdo->commit();

        // Initialize tenant directories (outside of transaction)
        $tenantManager = new TenantDirectoryManager($pdo);
        $folderResult = $tenantManager->initializeTenant($referenceCode, $schoolName);
        if (!$folderResult['success']) {
            error_log("Warning: Could not initialize tenant directories for {$referenceCode}: " . ($folderResult['error'] ?? 'Unknown error'));
        }

        // Generate URLs
        $urlGenerator = new SchoolURLGenerator($referenceCode);
        $urls = $urlGenerator->getAllURLs();
        $detectedBaseURL = method_exists($urlGenerator, 'getBaseURL')
            ? rtrim($urlGenerator->getBaseURL(), '/')
            : (defined('PUBLIC_BASE_URL') ? rtrim(PUBLIC_BASE_URL, '/') : '');

        if (method_exists($urlGenerator, 'getAdminLoginURL')) {
            $adminLoginURL = $urlGenerator->getAdminLoginURL();
        } else {
            $adminLoginURL = $detectedBaseURL . '/admin/login.php?school=' . strtolower($referenceCode);
        }

        // Optionally email credentials to the admin
        $emailSent = false;
        $shouldSendEmail = $options['send_welcome_email'] ?? true;
        if ($shouldSendEmail) {
            $recipientName = $contactName !== '' ? $contactName : $schoolName;

            $message = '
                <p>Hello ' . htmlspecialchars($recipientName) . ',</p>
                <p>Your school has been registered successfully on <strong>' . htmlspecialchars(APP_NAME) . '</strong>.</p>
                <p><strong>School:</strong> ' . htmlspecialchars($schoolName) . '<br>
                   <strong>Reference Code:</strong> ' . htmlspecialchars($referenceCode) . '<br>
                   <strong>Admin Username:</strong> ' . htmlspecialchars($adminUsername) . '<br>
                   <strong>Temporary Password:</strong> ' . htmlspecialchars($adminPassword) . '</p>
                <p>You can log in to the admin dashboard here:<br>
                   <a href="' . htmlspecialchars($adminLoginURL) . '">' . htmlspecialchars($adminLoginURL) . '</a></p>
                <p>After logging in, please update your password and complete your school profile.</p>
                <p>Welcome aboard!<br>' . htmlspecialchars(APP_NAME) . ' Team</p>
            ';

            $emailSent = sendEmail(
                $adminEmail,
                $recipientName,
                'Your ' . APP_NAME . ' School Admin Account',
                $message
            );
        }

        return [
            'school_id'        => $academyId,
            'admin_id'         => $adminId,
            'reference_code'   => $referenceCode,
            'school_name'      => $schoolName,
            'admin_username'   => $adminUsername,
            'admin_email'      => $adminEmail,
            'admin_password'   => $adminPassword,
            'urls'             => $urls,
            'login_url'        => $urls['student_login'] ?? ($detectedBaseURL . '/admin/login.php'),
            'admin_login_url'  => $adminLoginURL,
            'email_sent'       => $emailSent,
            'tenant_directory' => $folderResult['path'] ?? null,
        ];
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

