<?php
/**
 * School Context Manager
 * Handles multi-tenant school selection and context
 */

if (!function_exists('getDB')) {
    require_once __DIR__ . '/../config.php';
}

/**
 * Get the current school's reference code from session
 * @return string|null Academy reference code (e.g., 'KINE')
 */
function getCurrentSchool() {
    return $_SESSION['academy_reference'] ?? null;
}

/**
 * Get the current school's full details
 * @return array|null School details array
 */
function getCurrentSchoolData() {
    $ref = getCurrentSchool();
    if (!$ref) {
        return null;
    }
    
    // Check cache first
    if (isset($_SESSION['school_data_cache']) && 
        $_SESSION['school_data_cache']['reference_code'] === $ref) {
        return $_SESSION['school_data_cache'];
    }
    
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT * FROM academy_references 
        WHERE reference_code = ? AND is_active = 1
    ");
    $stmt->execute([$ref]);
    $school = $stmt->fetch();
    
    if ($school) {
        $_SESSION['school_data_cache'] = $school;
    }
    
    return $school;
}

/**
 * Set the current school context
 * @param string $reference_code Academy reference code
 * @return bool Success status
 */
function setCurrentSchool($reference_code) {
    $pdo = getDB();
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM academy_references 
            WHERE reference_code = ? AND is_active = 1
        ");
        $stmt->execute([$reference_code]);
        $school = $stmt->fetch();
        
        if ($school) {
            $_SESSION['academy_reference'] = $reference_code;
            $_SESSION['academy_name'] = $school['academy_name'];
            $_SESSION['academy_id'] = $school['id'];
            
            // Handle super admin access to school dashboard
            if (isSuperAdmin()) {
                // Super admin accessing school - set admin session for that school
                $_SESSION['admin_logged_in'] = true;
                $_SESSION['admin_id'] = $school['admin_id'] ?? null;
                $_SESSION['is_super_admin'] = true; // Keep super admin flag
                $_SESSION['admin_username'] = $_SESSION['admin_username'] ?? 'super_admin';
            } elseif (isset($_SESSION['admin_id'])) {
                // Keep existing admin_id from login
            } elseif (isset($school['admin_id'])) {
                $_SESSION['admin_id'] = $school['admin_id'];
            }
            
            $_SESSION['school_data_cache'] = $school;
            
            // Log school selection
            logSchoolActivity('SCHOOL_SELECTED', "Switched to {$school['academy_name']}");
            
            return true;
        }
        
        return false;
        
    } catch (PDOException $e) {
        error_log("Error setting school: " . $e->getMessage());
        return false;
    }
}

/**
 * Require a school to be selected - middleware function
 * Redirects to school selection page if no school selected
 */
function requireSchool() {
    // Skip if already on select_school.php to prevent loop
    if (isset($_SERVER['PHP_SELF']) && strpos($_SERVER['PHP_SELF'], 'select_school.php') !== false) {
        return;
    }
    
    if (!getCurrentSchool()) {
        // Check if user is logged in as admin
        if (isset($_SESSION['admin_id'])) {
            // Try to set school from admin's academy_reference
            $pdo = getDB();
            $stmt = $pdo->prepare("SELECT academy_reference FROM admins WHERE id = ?");
            $stmt->execute([$_SESSION['admin_id']]);
            $admin_ref = $stmt->fetchColumn();
            
            if ($admin_ref && setCurrentSchool($admin_ref)) {
                return; // School set successfully
            }
        }
        
        // Determine correct path to select_school.php
        $currentPath = $_SERVER['PHP_SELF'] ?? '';
        if (strpos($currentPath, '/admin/applications/') !== false ||
            strpos($currentPath, '/admin/settings/') !== false ||
            strpos($currentPath, '/admin/storage/') !== false ||
            strpos($currentPath, '/admin/moodle/') !== false) {
            // In admin subdirectory
            $redirectPath = '../../select_school.php';
        } elseif (strpos($currentPath, '/admin/') !== false) {
            // In admin root directory
            $redirectPath = '../select_school.php';
        } elseif (strpos($currentPath, '/super_admin/') !== false) {
            $redirectPath = '../select_school.php';
        } else {
            $redirectPath = 'select_school.php';
        }
        
        // Redirect to school selection
        header('Location: ' . $redirectPath);
        exit;
    }
}

/**
 * Check if user is super admin (can access all schools)
 * @return bool
 */
function isSuperAdmin() {
    return isset($_SESSION['is_super_admin']) && $_SESSION['is_super_admin'] === true;
}

/**
 * Validate that admin has access to specified school
 * @param string $academy_ref Academy reference to check
 * @return bool
 */
function validateSchoolAccess($academy_ref) {
    // Super admin can access everything
    if (isSuperAdmin()) {
        return true;
    }
    
    // Regular admin can only access their own school
    $current = getCurrentSchool();
    return $current === $academy_ref;
}

/**
 * Get a school-specific setting value
 * @param string $key Setting key
 * @param mixed $default Default value if not found
 * @return string|null Setting value
 */
function getSchoolSetting($key, $default = null) {
    $academy_ref = getCurrentSchool();
    if (!$academy_ref) {
        return $default;
    }
    
    // Check cache first
    $cacheKey = "school_setting_{$academy_ref}_{$key}";
    if (isset($_SESSION[$cacheKey])) {
        return $_SESSION[$cacheKey];
    }
    
    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            SELECT setting_value FROM tenant_settings 
            WHERE academy_reference = ? AND setting_key = ?
        ");
        $stmt->execute([$academy_ref, $key]);
        $value = $stmt->fetchColumn();
        
        $result = $value ?: $default;
        $_SESSION[$cacheKey] = $result; // Cache it
        
        return $result;
    } catch (PDOException $e) {
        error_log("Error getting school setting: " . $e->getMessage());
        return $default;
    }
}

/**
 * Set a school-specific setting value
 * @param string $key Setting key
 * @param string $value Setting value
 * @return bool Success status
 */
function setSchoolSetting($key, $value) {
    $academy_ref = getCurrentSchool();
    if (!$academy_ref) {
        return false;
    }
    
    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = CURRENT_TIMESTAMP
        ");
        $stmt->execute([$academy_ref, $key, $value, $value]);
        
        // Clear cache
        $cacheKey = "school_setting_{$academy_ref}_{$key}";
        unset($_SESSION[$cacheKey]);
        
        return true;
    } catch (PDOException $e) {
        error_log("Error setting school setting: " . $e->getMessage());
        return false;
    }
}

/**
 * Remove a tenant setting for a specific school
 *
 * @param string $academy_ref
 * @param string $key
 * @return bool
 */
function deleteTenantSettingValue($academy_ref, $key) {
    if (!$academy_ref || !$key) {
        return false;
    }

    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            DELETE FROM tenant_settings 
            WHERE academy_reference = ? AND setting_key = ?
        ");
        $stmt->execute([$academy_ref, $key]);
        return true;
    } catch (PDOException $e) {
        error_log("Error deleting tenant setting value: " . $e->getMessage());
        return false;
    }
}

/**
 * Get a tenant setting for a specific school
 *
 * @param string $academy_ref
 * @param string $key
 * @param mixed $default
 * @return mixed
 */
function getTenantSettingValue($academy_ref, $key, $default = null) {
    if (!$academy_ref || !$key) {
        return $default;
    }

    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            SELECT setting_value 
            FROM tenant_settings 
            WHERE academy_reference = ? AND setting_key = ?
            LIMIT 1
        ");
        $stmt->execute([$academy_ref, $key]);
        $value = $stmt->fetchColumn();
        return $value !== false ? $value : $default;
    } catch (PDOException $e) {
        error_log("Error getting tenant setting value: " . $e->getMessage());
        return $default;
    }
}

/**
 * Set a tenant setting for a specific school
 *
 * @param string $academy_ref
 * @param string $key
 * @param string $value
 * @return bool
 */
function setTenantSettingValue($academy_ref, $key, $value) {
    if (!$academy_ref || !$key) {
        return false;
    }

    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), updated_at = CURRENT_TIMESTAMP
        ");
        $stmt->execute([$academy_ref, $key, $value]);
        return true;
    } catch (PDOException $e) {
        error_log("Error setting tenant setting value: " . $e->getMessage());
        return false;
    }
}

/**
 * Retrieve commission settings for a school
 *
 * @param string $academy_ref
 * @return array
 */
function getSchoolCommissionSettings($academy_ref) {
    $type = getTenantSettingValue($academy_ref, 'commission_type', 'percentage');
    $valueRaw = getTenantSettingValue($academy_ref, 'commission_value', '10');
    $value = is_numeric($valueRaw) ? (float)$valueRaw : 10.0;

    if (!in_array($type, ['percentage', 'per_student'], true)) {
        $type = 'percentage';
    }

    return [
        'type' => $type,
        'value' => $value,
        'updated_at' => getTenantSettingValue($academy_ref, 'commission_updated_at')
    ];
}

/**
 * Persist commission settings for a school
 *
 * @param string $academy_ref
 * @param string $type
 * @param float $value
 * @return bool
 */
function setSchoolCommissionSettings($academy_ref, $type, $value) {
    if (!in_array($type, ['percentage', 'per_student'], true)) {
        $type = 'percentage';
    }

    if (!is_numeric($value) || $value < 0) {
        return false;
    }

    $value = (string)$value;

    $typeSaved = setTenantSettingValue($academy_ref, 'commission_type', $type);
    $valueSaved = setTenantSettingValue($academy_ref, 'commission_value', $value);
    $timestampSaved = setTenantSettingValue($academy_ref, 'commission_updated_at', date('Y-m-d H:i:s'));

    return $typeSaved && $valueSaved && $timestampSaved;
}

/**
 * Get all active schools
 * @return array Array of school data
 */
function getAllSchools() {
    $pdo = getDB();
    
    try {
        $stmt = $pdo->query("
            SELECT * FROM academy_references 
            WHERE is_active = 1 
            ORDER BY academy_name ASC
        ");
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Error getting schools: " . $e->getMessage());
        return [];
    }
}

/**
 * Fetch a single school record by ID (including admin details)
 *
 * @param int $schoolId
 * @return array|null
 */
function getSchoolById($schoolId) {
    if (!$schoolId) {
        return null;
    }

    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            SELECT ar.*, a.is_active AS admin_is_active, a.username AS admin_username
            FROM academy_references ar
            LEFT JOIN admins a ON ar.admin_id = a.id
            WHERE ar.id = ?
            LIMIT 1
        ");
        $stmt->execute([$schoolId]);
        return $stmt->fetch() ?: null;
    } catch (PDOException $e) {
        error_log("Error fetching school by ID {$schoolId}: " . $e->getMessage());
        return null;
    }
}

/**
 * Soft-delete a school (disable without data loss)
 *
 * @param array $school
 * @return bool
 */
function softDeleteSchool(array $school) {
    $pdo = getDB();
    $academy_ref = $school['reference_code'] ?? null;

    if (!$academy_ref) {
        return false;
    }

    try {
        $pdo->beginTransaction();

        try {
            $stmt = $pdo->prepare("UPDATE academy_references SET is_active = 0, subscription_status = 'cancelled' WHERE id = ?");
            $stmt->execute([$school['id']]);
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "Unknown column 'subscription_status'") !== false) {
                $fallback = $pdo->prepare("UPDATE academy_references SET is_active = 0 WHERE id = ?");
                $fallback->execute([$school['id']]);
            } else {
                throw $e;
            }
        }

        if (!empty($school['admin_id'])) {
            try {
                $stmt = $pdo->prepare("UPDATE admins SET is_active = 0 WHERE id = ?");
                $stmt->execute([$school['admin_id']]);
            } catch (PDOException $e) {
                if (strpos($e->getMessage(), "Unknown column 'is_active'") === false) {
                    throw $e;
                }
            }
        }

        setTenantSettingValue($academy_ref, 'soft_deleted', '1');
        setTenantSettingValue($academy_ref, 'soft_deleted_at', date('Y-m-d H:i:s'));

        $pdo->commit();
        logSchoolActivity('school_soft_delete', "School {$academy_ref} soft-deleted by super admin");
        return true;
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Soft delete school failed for {$academy_ref}: " . $e->getMessage());
        return false;
    }
}

/**
 * Restore a soft-deleted school
 *
 * @param array $school
 * @return bool
 */
function restoreSchool(array $school) {
    $pdo = getDB();
    $academy_ref = $school['reference_code'] ?? null;

    if (!$academy_ref) {
        return false;
    }

    try {
        $pdo->beginTransaction();

        try {
            $stmt = $pdo->prepare("UPDATE academy_references SET is_active = 1, subscription_status = 'active' WHERE id = ?");
            $stmt->execute([$school['id']]);
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "Unknown column 'subscription_status'") !== false) {
                $fallback = $pdo->prepare("UPDATE academy_references SET is_active = 1 WHERE id = ?");
                $fallback->execute([$school['id']]);
            } else {
                throw $e;
            }
        }

        if (!empty($school['admin_id'])) {
            try {
                $stmt = $pdo->prepare("UPDATE admins SET is_active = 1 WHERE id = ?");
                $stmt->execute([$school['admin_id']]);
            } catch (PDOException $e) {
                if (strpos($e->getMessage(), "Unknown column 'is_active'") === false) {
                    throw $e;
                }
            }
        }

        deleteTenantSettingValue($academy_ref, 'soft_deleted');
        deleteTenantSettingValue($academy_ref, 'soft_deleted_at');

        $pdo->commit();
        logSchoolActivity('school_restore', "School {$academy_ref} restored by super admin");
        return true;
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Restore school failed for {$academy_ref}: " . $e->getMessage());
        return false;
    }
}

/**
 * Permanently delete a school and all related data
 *
 * @param array $school
 * @return bool
 */
function permanentlyDeleteSchool(array $school) {
    $pdo = getDB();
    $academy_ref = $school['reference_code'] ?? null;
    $admin_id = $school['admin_id'] ?? null;

    if (!$academy_ref) {
        return false;
    }

    try {
        $pdo->beginTransaction();

        // Permanently delete all students first (handles Moodle cleanup)
        $studentStmt = $pdo->prepare("SELECT id FROM students WHERE academy_reference = ?");
        $studentStmt->execute([$academy_ref]);
        $studentIds = $studentStmt->fetchAll(PDO::FETCH_COLUMN);
        foreach ($studentIds as $studentId) {
            permanentlyDeleteStudent((int)$studentId);
        }

        // Delete all rows from tables containing academy_reference column (except academy_references)
        $tablesStmt = $pdo->prepare("
            SELECT c.TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS c
            JOIN INFORMATION_SCHEMA.TABLES t 
              ON t.TABLE_SCHEMA = c.TABLE_SCHEMA 
             AND t.TABLE_NAME = c.TABLE_NAME
            WHERE c.TABLE_SCHEMA = DATABASE()
              AND c.COLUMN_NAME = 'academy_reference'
              AND t.TABLE_TYPE = 'BASE TABLE'
              AND c.TABLE_NAME <> 'academy_references'
            GROUP BY c.TABLE_NAME
        ");
        $tablesStmt->execute();
        $tables = $tablesStmt->fetchAll(PDO::FETCH_COLUMN);

        foreach ($tables as $table) {
            try {
                $stmt = $pdo->prepare("DELETE FROM `{$table}` WHERE academy_reference = ?");
                $stmt->execute([$academy_ref]);
            } catch (PDOException $e) {
                error_log("Failed to delete from {$table} for {$academy_ref}: " . $e->getMessage());
            }
        }

        // Delete academy record
        $stmt = $pdo->prepare("DELETE FROM academy_references WHERE id = ?");
        $stmt->execute([$school['id']]);

        // Delete admin account (cascades to wallet tables)
        if ($admin_id) {
            try {
                $stmt = $pdo->prepare("DELETE FROM admins WHERE id = ?");
                $stmt->execute([$admin_id]);
            } catch (PDOException $e) {
                error_log("Failed to delete admin {$admin_id} for school {$academy_ref}: " . $e->getMessage());
            }
        }

        $pdo->commit();
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Permanent delete failed for {$academy_ref}: " . $e->getMessage());
        return false;
    }

    // Remove tenant directory (non-blocking)
    try {
        require_once __DIR__ . '/tenant_directory_manager.php';
        $manager = new TenantDirectoryManager($pdo);
        $directoryResult = $manager->deleteTenant($academy_ref);
        if (!$directoryResult['success']) {
            error_log("Failed to remove tenant directory for {$academy_ref}: " . ($directoryResult['error'] ?? 'Unknown error'));
        }
    } catch (Exception $e) {
        error_log("Tenant directory deletion error for {$academy_ref}: " . $e->getMessage());
    }

    logSchoolActivity('school_permanent_delete', "School {$academy_ref} permanently deleted by super admin");
    return true;
}

/**
 * Get school statistics
 * @param string $academy_ref Academy reference code
 * @return array Statistics array
 */
function getSchoolStats($academy_ref = null) {
    $academy_ref = $academy_ref ?? getCurrentSchool();
    if (!$academy_ref) {
        return [];
    }
    
    $pdo = getDB();
    
    try {
        // Get student count
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM students 
            WHERE academy_reference = ?
        ");
        $stmt->execute([$academy_ref]);
        $studentCount = $stmt->fetchColumn();
        
        // Get payment stats
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total_payments,
                SUM(amount) as total_revenue,
                SUM(admin_earnings) as total_earnings
            FROM payments 
            WHERE academy_reference = ? AND status = 'completed'
        ");
        $stmt->execute([$academy_ref]);
        $paymentStats = $stmt->fetch();
        
        // Get active student count
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM students 
            WHERE academy_reference = ? 
            AND payment_status = 'active'
        ");
        $stmt->execute([$academy_ref]);
        $activeStudents = $stmt->fetchColumn();
        
        return [
            'total_students' => $studentCount,
            'active_students' => $activeStudents,
            'total_payments' => $paymentStats['total_payments'] ?? 0,
            'total_revenue' => $paymentStats['total_revenue'] ?? 0.00,
            'total_earnings' => $paymentStats['total_earnings'] ?? 0.00,
            'academy_reference' => $academy_ref
        ];
        
    } catch (PDOException $e) {
        error_log("Error getting school stats: " . $e->getMessage());
        return [];
    }
}

/**
 * Log school-related activity
 * @param string $action Action type
 * @param string $details Action details
 */
function logSchoolActivity($action, $details) {
    // If monitoring system exists, use it
    if (function_exists('logUserActivity')) {
        call_user_func('logUserActivity', $action, $details);
    }
}

/**
 * Get school display name
 * @return string School name or default
 */
function getSchoolName() {
    $school = getCurrentSchoolData();
    return $school['academy_name'] ?? getSchoolSetting('school_name', 'Skolo-Kine');
}

/**
 * Get school color scheme
 * @return array Colors array
 */
function getSchoolColors() {
    return [
        'primary' => getSchoolSetting('primary_color', '#4F46E5'),
        'secondary' => getSchoolSetting('secondary_color', '#10B981')
    ];
}

/**
 * Clear school context (logout)
 */
function clearSchoolContext() {
    // Ensure session is started
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    if (!isset($_SESSION)) {
        return;
    }
    
    unset($_SESSION['academy_reference']);
    unset($_SESSION['academy_name']);
    unset($_SESSION['academy_id']);
    unset($_SESSION['school_data_cache']);
    
    // Clear all cached settings
    if (is_array($_SESSION)) {
        foreach ($_SESSION as $key => $value) {
            if (strpos($key, 'school_setting_') === 0) {
                unset($_SESSION[$key]);
            }
        }
    }
}