<?php

function generateQRCodeDataUri($url, $size = 300) {
    $vendorAutoload = __DIR__ . '/../vendor/autoload.php';
    $builderClass = '\\Endroid\\QrCode\\Builder\\Builder';
    $writerClass = '\\Endroid\\QrCode\\Writer\\PngWriter';
    $encodingClass = '\\Endroid\\QrCode\\Encoding\\Encoding';
    $errorLevelClass = '\\Endroid\\QrCode\\ErrorCorrectionLevel\\ErrorCorrectionLevelHigh';

    if (file_exists($vendorAutoload)) {
        require_once $vendorAutoload;

        if (
            class_exists($builderClass) &&
            class_exists($writerClass) &&
            class_exists($encodingClass) &&
            class_exists($errorLevelClass)
        ) {
            try {
                $writer = new $writerClass();
                $encoding = new $encodingClass('UTF-8');
                $errorLevel = new $errorLevelClass();

                $builder = $builderClass::create()
                    ->writer($writer)
                    ->data($url)
                    ->encoding($encoding)
                    ->errorCorrectionLevel($errorLevel)
                    ->size($size)
                    ->margin(10)
                    ->build();

                $data = $builder->getString();
                $base64 = base64_encode($data);

                return 'data:image/png;base64,' . $base64;
            } catch (\Throwable $e) {
                error_log('QR code generation failure: ' . $e->getMessage());
            }
        } else {
            error_log('Endroid QR Code library not available. Falling back to Google Charts.');
        }
    } else {
        error_log('Composer autoload not found. Falling back to Google Charts.');
    }

    $encodedUrl = urlencode($url);
    $dimension = max(100, (int)$size);
    $fallbackUrl = "https://chart.googleapis.com/chart?chs={$dimension}x{$dimension}&cht=qr&chl={$encodedUrl}";

    $qrBinary = null;

    if (function_exists('curl_init')) {
        $ch = curl_init($fallbackUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_FOLLOWLOCATION => true,
        ]);
        $qrBinary = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($status !== 200) {
            $qrBinary = null;
        }
    }

    if ($qrBinary === null) {
        $context = stream_context_create([
            'http' => ['timeout' => 5],
            'https' => ['timeout' => 5]
        ]);

        $qrBinary = @file_get_contents($fallbackUrl, false, $context);
    }

    if ($qrBinary !== false && $qrBinary !== null && !empty($qrBinary)) {
        return 'data:image/png;base64,' . base64_encode($qrBinary);
    }

    // Final fallback: Use qrserver.com API if all else fails
    if (!empty($url)) {
        $encodedUrl = urlencode($url);
        return 'https://api.qrserver.com/v1/create-qr-code/?size=' . $dimension . 'x' . $dimension . '&data=' . $encodedUrl;
    }

    return null;
}

